/* ===========================================================================

Crystalfontz CFA039A0-N-V Arduino library example.

https://www.crystalfontz.com/product/cfa039a0nvdct-480x128-graphic-usb-tft-display-module

Mark Williams (2025)
Crystalfontz America Inc.

Distributed under the "The Unlicense".
http://unlicense.org
This is free and unencumbered software released into the public domain.
For more details, see the website above.

=========================================================================== */

#include <Arduino.h>
#include "CFA039A0-N-V.h"

//////////////////////////////////////////////////////////////////////////////////////////////////////
//PROJECT SETTINGS

//interface selection (select one)
#define IFACE_I2C
//#define IFACE_SPI

//enabled debug output on the arduino virtual usb serial port
//NOTE!!! enabling this slows down the interface to the point some
//!!!     control report packets may not always respond
//#define ENABLE_DEBUG_OUTPUT

//////////////////////////////////////////////////////////////////////////////////////////////////////
// ARDUINO UNO CONNECTION DETAILS

//I2C Interface connections
// SEEEDUNIO <-> CFA039A0-N-V
// SCL <-> H1-P4
// SDA <-> H1-P3
// PIN9 <-> H1-P13 (data ready)
// GND <-> H1-P15 (ground)

//SPI Interface connections
// SEEEDUNIO <-> CFA039A0-N-V
// MOSI/PIN11 <-> H1-P8
// MISO/PIN12 <-> H1-P7
// SCK/PIN13 <-> H1-P10
// PIN10 <-> H1-P9 (slave select)
// PIN9 <-> H1-P12 (data ready)
// GND <-> H1-P15 (ground)

//////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef IFACE_I2C
# ifndef IFACE_SPI
#  ifndef IFACE_SERIAL
#    error YOU MUST SELECT A MODULE INTERFACE ABOVE
#  endif
# endif
#endif

//////////////////////////////////////////////////////////////////////////////////////////////////////

//interface settings
#ifdef IFACE_I2C
#include <Wire.h>
# define IFACE_I2C_ADDR		  0x44
# define IFACE_I2C_SPEED	  100000
# define IFACE_I2C_READY	  9
# define IFACE_I2C_REQ_LEN	4
#endif
#ifdef IFACE_SPI
# include <SPI.h>
# define IFACE_SPI_SPEED	  1000000UL
# define IFACE_SPI_BITFIRST	MSBFIRST
# define IFACE_SPI_MODE		  SPI_MODE0
# define IFACE_SPI_SS		    10
# define IFACE_SPI_READY  	9
# define IFACE_SPI_REQ_LEN	16
#endif

#define IFACE_READY_WAIT	50 /*mS*/

//////////////////////////////////////////////////////////////////////////////////////////////////////


#ifdef IFACE_SPI
SPISettings spiSettings = SPISettings(IFACE_SPI_SPEED, IFACE_SPI_BITFIRST, IFACE_SPI_MODE);
#endif

CFA039A0NV cfaModule;

//////////////////////////////////////////////////////////////////////////////////////////////////////

//rainbow RGB values + white and black
#define RAINBOW_COLORS  9
const uint8_t rainbow[RAINBOW_COLORS][3] =
{
  {0, 0, 0},      // Black
  {255, 0, 0},    // Red
  {255, 127, 0},  // Orange
  {255, 255, 0},  // Yellow
  {0, 255, 0},    // Green
  {0, 0, 255},    // Blue
  {75, 0, 130},   // Indigo
  {148, 0, 211},  // Violet
  {255, 255, 255} // White
};

//////////////////////////////////////////////////////////////////////////////////////////////////////

void setup()
{
	//run at power-on/reset of the Seeeduino/Arduino
	//setup interfaces / variables
	
	//debug out
	Serial.begin(115200);
	Serial.write("\n\nsetup()\n");

	//module interface init
#ifdef IFACE_I2C
	Wire.begin();
	pinMode(IFACE_I2C_READY, INPUT_PULLUP); //I2C-Ready
#endif

#ifdef IFACE_SPI
	pinMode(IFACE_SPI_SS, OUTPUT); //SPI-SS
	digitalWrite(IFACE_SPI_SS, HIGH); //SS deselect
	pinMode(IFACE_SPI_READY, INPUT_PULLUP); //SPI-Ready
	SPI.begin();
#endif

  //module init
#ifdef IFACE_I2C  
  cfaModule.begin(Wire, IFACE_I2C_ADDR, IFACE_I2C_READY);
#endif
#ifdef IFACE_SPI
  cfaModule.begin(SPI, spiSettings, IFACE_SPI_SS, IFACE_SPI_READY);
#endif
#ifdef ENABLE_DEBUG_OUTPUT
  cfaModule.enableDebugOutput();
#endif

  //ping the module until we get a reply
  Serial.println(F("Attemping to PING the module..."));
  while(1)
  {
    if (cfaModule.cmdPing((uint8_t*)"PING", 4) == CFA039A0NV::CFA_OK)
    {
      //we got a response
      break;
    }
    Serial.println(F("no reply"));
    delay(500);
  }
  cfaModule.clearBuffers();
  Serial.println(F("Ping reply recieved, continuing."));
}

void loop(void)
{
  bool keypadCycleEnabled = false;
  uint16_t fillColor;

  //main arduino api/framework loop
  CFA039A0NV::packet_t inPacket;

  //setup the initial display
  Serial.println(F("Inital display setup"));

  //clear the screen
  cfaModule.cmdClearDisplay();

  //set the display backlight brightness to 100%
  cfaModule.cmdSetDisplayBrightness(100);
 
  //set the display background to bright white
  cfaModule.cmdSetGraphicsBackgroundColor(0xFFFF);

  //display the crystalfontz startup image
  //this is just so we have something to display as a background, since we may not have
  //any other images available on on-board flash / micros
  cfaModule.cmdImageFileLoad(1, "CFBOOTLOGO", 0, 0, false, false, false, false, false, 1, CFA039A0NV::TOUCHREPORT_OFF);

  //place a filled semi-opaque sketch layer over the top to dim the background & image
  //first set style fill-color a to dark yellow
  cfaModule.cmdSetGraphicsFillAColor(0, 0x42C0);

  //next set the style new object opacity to 70% (value of 180)
  cfaModule.cmdSetGraphicsObjectOpacity(0, 180);

  //now create the sketch gfx object, make it cover the full screen, filled with
  //black (from style fill-a color), at a higher z-index than the CFBOOTLOGO image to cover
  //it, using the defult opacity set in the last command
  cfaModule.cmdSketchNewSurface(2, 0,0, 480,128, true, 2, CFA039A0NV::TOUCHREPORT_OFF);

  //set the fill-a default color and opacity back to defauls for the following
  //object creation
  //fill-a color (RGB565 0x3812)
  cfaModule.cmdSetGraphicsFillAColor(0, 0x3812);
  //set default opacity (value of 255)
  cfaModule.cmdSetGraphicsObjectOpacity(0, 255);

  //we can fit in three rows of 32 pixel high controls with 8 pixel spaces
  //between each

  //create a number-edit control to change backlight brightness
  cfaModule.cmdNumberEditNew(10, "LCD Bright", 10,8, 220,32, 1, 100, 100, false, CFA039A0NV::JUSTIFY_LEFT, false,
  true, true, 10, CFA039A0NV::TOUCHREPORT_OFF);

  //create a slide bar to change indicator leds
  cfaModule.cmdSliderNew(11, "Ind LED", 240,8, 230,32, 0, RAINBOW_COLORS*8, 0, false, true, CFA039A0NV::JUSTIFY_RIGHT, false, true, 10, CFA039A0NV::TOUCHREPORT_OFF);

  //create a toggle push button which will cycle through
  //keypad button colors when enabled
  cfaModule.cmdCheckboxNew(12, "KP LED Cycle", 10,48, 180,32, CFA039A0NV::CHECKBOX_NONE, true, false, false, CFA039A0NV::JUSTIFY_LEFT, false, true, 10, CFA039A0NV::TOUCHREPORT_OFF); 

  //create a progress bar to show keypad led cycle
  cfaModule.cmdProgressBarNew(13, "", 200,48, 270,32, 0, RAINBOW_COLORS*8, 0, false, CFA039A0NV::JUSTIFY_CENTER, false, false, 10, CFA039A0NV::TOUCHREPORT_OFF);

  //create toggle button to switch to touch position test
  cfaModule.cmdButtonNew(14, "Color Toggle", 160,88, 160,32, false, true, false, true, 10, CFA039A0NV::TOUCHREPORT_OFF);

  //do first LED updates
  KeypadLEDCycle();
  IndLEDUpdate(0);

  //finished setting up the display, sit in a main loop and wait for report packets
  Serial.println(F("Entring main program loop"));
  while(1)
  {
    //wait until we get a report packet from the module
    //get all report packet types
    while (cfaModule.getReportPacket((PacketReports_t)0xFF, &inPacket, 100) == CFA039A0NV::CFA_OK)
    {
      //we have a packet
    	cfaModule.debugPrintPacket("IN", &inPacket);

      //check the incoming packet for a gfx object control value update
      //data[0] is the gfx object ID of the control
      if ((inPacket.command == PRPT_NUMEDIT) && (inPacket.data[0] == 10))
      {
        //this a lcd brightness numedit control report packet
        //update the display backlight brightness to the new level
        cfaModule.cmdSetDisplayBrightness(inPacket.data[1]);
      }
      else if ((inPacket.command == PRPT_SLIDER) && (inPacket.data[0] == 11))
      {
        //this is a indication led slider control value update
        IndLEDUpdate(inPacket.data[1]); //pass low byte of value to function
      }
      else if ((inPacket.command == PRPT_CHECKBOX) && (inPacket.data[0] == 12))
      {
        //this is a keypad LED cycle checkbox control value update
        keypadCycleEnabled = (inPacket.data[1] & 1) ? true : false;
      }      
      else if ((inPacket.command == PRPT_BUTTON) && (inPacket.data[0] == 14))
      {
        //toggle background overlay color
        if (inPacket.data[1] == 0)
          //button status is up, use original yellow color
          fillColor = 0x42C0;
        else
          //button state is depressed, use a purple color
          fillColor = 0x8010;
        
        //set the color by filling the sketch object we created earlier with 
        //a filled rectangle of the above color
        //the original opacity of the sketch object is maintained, so the
        //background image is still visible through it
        cfaModule.cmdSketchDrawRectangle(2, 0,0, 479,127, 0, true, fillColor, false, fillColor);
      }
    }

    //do keypad led cycle if enabled
    if (keypadCycleEnabled == true)
    {
      KeypadLEDCycle();
    }
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////

void KeypadSetKey(uint8_t key, uint8_t R, uint8_t G, uint8_t B)
{
  cfaModule.cmdSetKeypadAndIndicatorLEDColor((CFA039A0NV::keypad_e)(1<<key), CFA039A0NV::INDICATORLED_NONE, R, G, B);
}

void RainbowGetRGB(uint8_t lastColor, uint8_t step, uint8_t *R, uint8_t *G, uint8_t *B)
{
  //step is 0 to 7
  //lastColor is 0 to RAINBOW_COLORS
  //calc last and next colors

  //limit/wrap lastColor
  lastColor = lastColor % RAINBOW_COLORS;
  
  //get next color & limit/wrap
  uint8_t nextColor = (lastColor + 1) % RAINBOW_COLORS;

  //calc RGB fade values
  *R = rainbow[lastColor][0] + ( (rainbow[nextColor][0] - rainbow[lastColor][0]) * step / 8 );
  *G = rainbow[lastColor][1] + ( (rainbow[nextColor][1] - rainbow[lastColor][1]) * step / 8 );
  *B = rainbow[lastColor][2] + ( (rainbow[nextColor][2] - rainbow[lastColor][2]) * step / 8 );
}

void KeypadLEDCycle(void)
{
  //this function fades the keypad LEDs through the colors of the rainbow
  uint8_t R,G,B;
  uint8_t color;
  static uint8_t keypadLEDStepCount = 0;

  //we'll do 8 steps between each color in the table to fade
  keypadLEDStepCount++;
  if (keypadLEDStepCount >= RAINBOW_COLORS * 8)
    keypadLEDStepCount = 0;
  
  //update progress bar with step count
  cfaModule.cmdProgressBarSetValue(13, keypadLEDStepCount);
  
  //set up key color
  color = (keypadLEDStepCount >> 3) + 0;
  RainbowGetRGB(color, keypadLEDStepCount % 8, &R,&G,&B);
  KeypadSetKey(0, R, G, B);

  //set enter key color
  color = (keypadLEDStepCount >> 3) + 2;
  if (color > RAINBOW_COLORS-1) color = color - RAINBOW_COLORS;
  RainbowGetRGB(color, keypadLEDStepCount % 8, &R,&G,&B);
  KeypadSetKey(1, R, G, B);

  //also set progress bar color here to the same
  //color as the enter key
  cfaModule.cmdSetGraphicsControlAColor(13, ((R & 0b11111000) << 8) | ((G & 0b11111100) << 3) | (B >> 3));

  //set cancel key color  
  color = (keypadLEDStepCount >> 3) + 4;
  if (color > RAINBOW_COLORS-1) color = color - RAINBOW_COLORS;
  RainbowGetRGB(color, keypadLEDStepCount % 8, &R,&G,&B);
  KeypadSetKey(2, R, G, B);
  
  //set left key color
  color = (keypadLEDStepCount >> 3) + 6;
  if (color > RAINBOW_COLORS-1) color = color - RAINBOW_COLORS;
  RainbowGetRGB(color, keypadLEDStepCount % 8, &R,&G,&B);
  KeypadSetKey(3, R, G, B);
  
  //set right key color
  color = (keypadLEDStepCount >> 3) + 8;
  if (color > RAINBOW_COLORS-1) color = color - RAINBOW_COLORS;
  RainbowGetRGB(color, keypadLEDStepCount % 8, &R,&G,&B);
  KeypadSetKey(4, R, G, B);
  
  //set down key color
  color = (keypadLEDStepCount >> 3) + 5;
  if (color > RAINBOW_COLORS-1) color = color - RAINBOW_COLORS;
  RainbowGetRGB(color, keypadLEDStepCount % 8, &R,&G,&B);
  KeypadSetKey(5, R, G, B);
}

void IndSetLED(uint8_t ledIndex, uint8_t R, uint8_t G, uint8_t B)
{
  cfaModule.cmdSetKeypadAndIndicatorLEDColor(CFA039A0NV::KEY_NONE, (CFA039A0NV::indicator_e)(1<<ledIndex), R, G, B);
}

void IndLEDUpdate(uint8_t value)
{
  //this function updates the indicator LEDs colors depending on the passed value
  //we use the rainbow colors, with 8 steps per color
  uint8_t R,G,B;
  uint8_t color;
  //led 0 (top)
  color = (value >> 3) + 0;
  RainbowGetRGB(color, value % 8, &R,&G,&B);
  IndSetLED(0, R, G, B);
  //led 1
  color = (value >> 3) + 2;
  RainbowGetRGB(color, value % 8, &R,&G,&B);
  IndSetLED(1, R, G, B);
  //led 2
  color = (value >> 3) + 5;
  RainbowGetRGB(color, value % 8, &R,&G,&B);
  IndSetLED(2, R, G, B);
  //led 3 (bottom)
  color = (value >> 3) + 8;
  RainbowGetRGB(color, value % 8, &R,&G,&B);
  IndSetLED(3, R, G, B);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////

