/* ===========================================================================

Crystalfontz CFA039A0-N-V module I2C/SPI Arduino communications library.

This library provides an easy to use interface for the CFA039A0-N-V modules
SPI/I2C packet based communications interface, and command system.

This library has been tested with the Seeeduino v4.2 development board and
the Arduino Uno. It may also be used with other development boards with
minor modifications.

See "CFA039A0-N-V.h" for library configuration.
See the library examples for usage information.

https://www.crystalfontz.com/product/cfa039a0nvdct-480x128-graphic-usb-tft-display-module

Mark Williams (2025)
Crystalfontz America Inc.

Distributed under the "The Unlicense".
http://unlicense.org
This is free and unencumbered software released into the public domain.
For more details, see the website above.

=========================================================================== */

#pragma once

#include <Arduino.h>
#include <Wire.h>
#include <SPI.h>
#include <float.h>

#include "cfa_commands.h"

//////////////////////////////////////////////////////////////////////
// LIBRARY CONFIGURATION

//packet structures support data lengths up to this length
//manually set or decrease the size if the full size is not
//needed and less memory use is required
#define PACKET_DATA_SIZE                (PACKET_CFA039_DATA_SIZE)
//incoming fifo buffer can hold up to X number of full-sized packets
#define IN_FIFO_LENGTH                  (PACKET_DATA_SIZE*2)
//default command reply timeout in mS
#define CFA_COMMAND_TIMEOUT             (500)
//default i2c address
#define CFA039A0NV_DEFAULT_I2C_ADDRESS  (0x44)
//use a lib-global incoming packet struct
#define CFA039A_USE_STATIC_PACKET

//////////////////////////////////////////////////////////////////////

#define SETPKTDATA_UINT16(i, v) packet.data[i] = ((uint16_t)v) & 0xFF; packet.data[i+1] = (((uint16_t)v) >> 8) & 0xFF;

class CFA039A0NV
{
public:
  //types
  typedef union __attribute__((__aligned__(8)))
  {
    uint8_t		b[2];
    uint16_t	w;
  } word_u;

  typedef struct __attribute__((__aligned__(8)))
  {
    uint8_t		command;
    uint8_t	  length;
    uint8_t		data[PACKET_DATA_SIZE];
    word_u	  crc;
  } packet_t;

  typedef enum 
  {
      CFA_OK = 0,
      CFA_ERROR,
      CFA_SENDERROR,
      CFA_CMDERROR,
      CFA_TIMEOUT,
      CFA_BADDATA,
  } state_e;

  typedef enum
  {
    KEY_NONE = 0,
    KEY_UP = (1<<0),
    KEY_ENTER = (1<<1),
    KEY_CANCEL = (1<<2),
    KEY_LEFT = (1<<3),
    KEY_RIGHT = (1<<4),
    KEY_DOWN = (1<<5), 
    KEY_ALL = 0b111111
  } keypad_e;

  typedef enum
  {
    INDICATORLED_NONE = 0,
    INDICATORLED_TOP = (1<<0),
    INDICATORLED_1 = (1<<0),
    INDICATORLED_2 = (1<<1),
    INDICATORLED_3 = (1<<2),
    INDICATORLED_4 = (1<<3),
    INDICATORLED_BOTTOM = (1<<3),
    INDICATORLED_ALL = 0b1111
  } indicator_e;

  typedef enum
  {
    DIM_NONE = 0,
    DIM_LCD = (1<<0),
    DIM_KEYPAD = (1<<1),
    DIM_INDICATOR = (1<<2)
  } dimmingeffects_e;

  typedef enum 
  {
    ACTMONITOR_NONE = 0,
    ACTMONITOR_KEYPAD = (1<<0),
    ACTMONITOR_TOUCH = (1<<1),
    ACTMONITOR_PACKET = (1<<2)
  } actmonitor_e;

  typedef enum
  {
    TOUCHREPORT_OFF = 0,
    TOUCHREPORT_SINGLE = 1,
    TOUCHREPORT_SINGLETRACK = 2,
    TOUCHREPORT_ALL = 3
  } touchreport_e;

  typedef enum
  {
    ATX_NONE = 0,
    ATX_AUTOPOLARITY = (1<<0),
    ATX_RESTARTINVERT = (1<<1),
    ATX_POWERINVERT = (1<<2),
    ATX_LIGHTINGFOLLOWSPOWER = (1<<3),
    ATX_MODULEFOLLOWSPOWER = (1<<4),
    ATX_KEYPADRESET = (1<<5),
    ATX_KEYPADPOWERON = (1<<6),
    ATX_KEYPADPOWEROFF = (1<<7)
  } atx_e;

  typedef enum
  {
    IFACEOPTION_NONE = 0,
    IFACEOPTION_ENABLEINTERFACE = (1<<0),
    IFACEOPTION_ENABLECMD = (1<<1),
    IFACEOPTION_ENABLEREPORTS = (1<<2),
    IFACEOPTION_ENABLEERRORS = (1<<3),
    IFACEOPTION_ENABLEALLERRORS = (1<<4),
    IFACEOPTION_DEFAULTS = IFACEOPTION_ENABLEINTERFACE |
      IFACEOPTION_ENABLECMD | IFACEOPTION_ENABLEREPORTS |
      IFACEOPTION_ENABLEERRORS,
  } interfaceoption_e;

  typedef enum
  {
    INTERFACE_SERIAL = 0,
    INTERFACE_USB = 1,
    INTERFACE_SPI = 2,
    INTERFACE_I2C = 3
  } interface_e;

  typedef enum
  {
    SERIALBAUD_19200 = 0,
    SERIALBAUD_115200 = 1,
    SERIALBAUD_9600 = 2,
  } serialbaud_e;

  typedef enum
  {
    SPI_CPOL = (1<<0),
    SPI_CPHA = (1<<1),
    SPI_LSBFIRST = (1<<2),
    SPI_MODE_0 = 0,
    SPI_MODE_1 = SPI_CPHA,
    SPI_MODE_2 = SPI_CPOL,
    SPI_MODE_3 = SPI_CPHA | SPI_CPOL
  } spimode_e;

  typedef enum
  {
    I2CSPEED_100KHZ = 0,
    I2CSPEED_400KHZ = 1
  } i2cspeed_e;

  typedef enum
  {
    GPIO_0   = 0,
    GPIO_1   = 1,
    GPIO_2   = 2,
    GPIO_3   = 3,
    GPIO_4   = 4,
    GPIO_5   = 13,
    GPIO_6   = 14,
    GPIO_7   = 15,
    GPIO_8   = 16,
    GPIO_9   = 17,
    GPIO_10  = 18,
    GPIO_11  = 19,
    GPIO_12  = 20,
    H1PIN_11 = 0,
    H1PIN_12 = 1,
    H1PIN_9  = 2,
    H1PIN_10 = 3,
    H1PIN_13 = 4,
    H1PIN_5  = 13,
    H1PIN_6  = 14,
    H1PIN_1  = 15,
    H1PIN_2  = 16,
    H1PIN_3  = 17,
    H1PIN_4  = 18,
    H1PIN_7  = 19,
    H1PIN_8  = 20,
  } gpiopins_e;

  typedef enum
  {
    DRIVEMODE_STRONGUP_RESDOWN    = 0b000,
    DRIVEMODE_STRONGUP_STRONGDOWN = 0b001,
    DRIVEMODE_HIZ_INPUT           = 0b010,
    DRIVEMODE_RESUP_STRONGDOWN    = 0b011,
    DRIVEMODE_STRONGUP_HIZDOWN    = 0b100,
    DRIVEMODE_HIZUP_STRONGDOWN    = 0b111,
  } drivemodes_e;

  typedef enum
  {
    FAN_1 = (1<<0),
    FAN_2 = (1<<1),
    FAN_3 = (1<<2),
    FAN_4 = (1<<3)
  } fanbitmask_e;

  typedef enum
  {
    FILEOPEN_CLOSE = 0,
    FILEOPEN_READONLY = 1,
    FILEOPEN_READWRITETRUNCATE = 2,
    FILEOPEN_READWRITEAPPEND = 3,
  } fileopenmode_e;

  typedef enum
  {
    FILESYSTEM_ONBOARDFLASH = 0,
    FILESYSTEM_MICROSD = 1,
    FILESYSTEM_GFXMEM = 2,
    FILESYSTEM_OSMEM = 3
  } filesystem_e;

  typedef enum
  {
    JUSTIFY_LEFT = 0,
    JUSTIFY_CENTER = 1,
    JUSTIFY_RIGHT = 2
  } justify_e;

  typedef enum
  {
    CHECKBOX_NONE = 0,
    CHECKBOX_TICKED = 1,
    CHECKBOX_CROSSED = 2
  } checkbox_e;

  //inits / settings
  CFA039A0NV();
  ~CFA039A0NV();
  void begin(TwoWire &wire, uint8_t i2c_address, uint8_t i2c_ready_pin);
  void begin(SPIClass &spi, SPISettings &spi_settings, uint8_t spi_cs_pin, uint8_t spi_ready_pin);
  void end();
  void enableDebugOutput();
  void setCmdReplyTimeout(uint16_t wait_ms);
  //coms
  state_e sendPacket(packet_t *packet);
  uint16_t recieveData(uint16_t wait_ms);
  bool getPacket(packet_t *packet);
  state_e cmdGetReply(uint8_t command, uint8_t subCommand, packet_t *packet, uint16_t wait_ms);
  state_e sendPacketAndGetReply(packet_t *packet, uint16_t wait_ms, bool matchSubCommand=false);
  state_e getReportPacket(PacketReports_t reportType, packet_t *reportPacket, uint16_t waitMS);
  void clearBuffers(void);

  //misc
  void debugPrintPacket(const char *prepend, packet_t *packet);
  
  //commands
  state_e cmdPing(uint8_t *data, uint8_t length);
  state_e cmdGetModuleVersion(char verString[32]);
  state_e cmdGetModuleSerialNumber(char serialString[32]);
  state_e cmdWriteUserFlash(char *data, uint8_t length);
  state_e cmdReadUserFlash(char *data, uint8_t *length);
  state_e cmdStoreBootState(void);
  state_e cmdLoadSavedSettings(void);
  state_e cmdRestartHost(void);
  state_e cmdPowerOffHost(void);
  state_e cmdModuleRestart(void);
  state_e cmdRestoreDefaultsReset(void);
  state_e cmdRestartIntoBootloader(void);
  state_e cmdClearDisplay(void);
  state_e cmdSetDisplayBrightness(uint8_t backlightPercent);
  state_e cmdSetDisplayAndKeypadBrightness(uint8_t backlightPercent, uint8_t keypadPercent);
  state_e cmdSetKeypadAndIndicatorLEDColor(keypad_e keys, indicator_e indicators, uint8_t R, uint8_t G, uint8_t B);
  state_e cmdSetKeypadAndIndicatorLEDColor(keypad_e keys, indicator_e indicators, uint32_t RGB888);
  state_e cmdSetLightingIdleDimming(dimmingeffects_e dimmingEffects, actmonitor_e activityMonitor,
    uint8_t level1TimoutSeconds, uint8_t level1DimPercent,
  uint8_t level2TimoutSeconds, uint8_t level2DimPercent);
  state_e cmdSetKeypadReporting(keypad_e pressReport, keypad_e releaseReport);
  state_e cmdPollKeypadState(keypad_e *currentState, keypad_e *pressedSinceLastPoll, keypad_e *releasedSinceLastPoll);
  state_e cmdSetTouchscreenReporting(touchreport_e reportingOption);
  state_e cmdATXOptions(atx_e atxOptions, uint8_t pulseLength);
  state_e cmdWatchdogReset(uint8_t watchdogTimeoutSeconds);
  state_e cmdWatchdogDisable(void);
  state_e cmdSetInterfaceSerialOptions(bool enableInterface, interfaceoption_e options, serialbaud_e baudRate = SERIALBAUD_115200);
  state_e cmdSetInterfaceUSBOptions(bool enableInterface, interfaceoption_e options);
  state_e cmdSetInterfaceSPIOptions(bool enableInterface, interfaceoption_e options, spimode_e spiMode);
  state_e cmdSetInterfaceI2COptions(bool enableInterface, interfaceoption_e options, uint8_t i2cAddress, i2cspeed_e i2cSpeed);
  state_e cmdSetGPIOPin(gpiopins_e gpioIndex, uint8_t outPWMPercent = 0, drivemodes_e driveMode = DRIVEMODE_STRONGUP_STRONGDOWN);
  state_e cmdSetGPIOPinToDefault(gpiopins_e gpioIndex);
  state_e cmdReadGPIOPin(gpiopins_e gpioIndex, bool *currentPinState, bool *fallSinceLastPoll, bool *riseSinceLastPoll, uint8_t *outPWMPercent, drivemodes_e *driveMode, bool *userMode);
  state_e cmdReadGPIOPinADC(gpiopins_e gpioIndex, uint16_t *avgSinceLastPollx16, uint16_t *minSinceLastPoll, uint16_t *maxSinceLastPoll);
  state_e cmdReadGPIOPinADC(gpiopins_e gpioIndex, float *avgSinceLastPollVolts, float *minSinceLastPollVolts, float *maxSinceLastPollVolts);
  state_e cmdReadFBSCABCount(uint8_t *fbscabCount);
  state_e cmdReadFBSCABSerialNumber(uint8_t fbscabIndex, char fbscabSerialNumber[16]);
  state_e cmdSetFBSCABFanPower(uint8_t fbscabIndex, uint8_t fan1Power, uint8_t fan2Power, uint8_t fan3Power, uint8_t fan4Power);
  state_e cmdSetFBSCABFanPowerAndSettings(uint8_t fbscabIndex, uint8_t fan1Power, uint8_t fan2Power, uint8_t fan3Power, uint8_t fan4Power,
  fanbitmask_e enableFailSafe, uint8_t failSafeTimeout,
  uint8_t fan1GlitchDelay, uint8_t fan2GlitchDelay, uint8_t fan3GlitchDelay, uint8_t fan4GlitchDelay);
  state_e cmdReadFanTach(uint8_t fbscabIndex, uint16_t *fan1RPM, uint16_t *fan2RPM, uint16_t *fan3RPM, uint16_t *fan4RPM);
  state_e cmdReadDOWTemperature(uint8_t fbscabIndex, uint8_t dowSensorIndex, uint16_t *tempX16);
  state_e cmdReadDOWTemperature(uint8_t fbscabIndex, uint8_t dowSensorIndex, float *temperature);
  state_e cmdFBSCABResetAndSearch(void);
  state_e cmdFBSCABSetAutoFanControl(uint8_t fbscabIndex, uint8_t fanIndex, bool afcEnabled, bool fanOffUnderMinPower,
    uint8_t afcResponsiveness, uint8_t dowTempIndex, int8_t targetTemperature, uint8_t minFanPower, uint8_t maxFanPower);
  state_e cmdFileOpen(fileopenmode_e openMode, char *fileName);
  state_e cmdFileClose(void);
  state_e cmdFileSeek(uint32_t seekLocation);
  state_e cmdFileRead(uint8_t *data, uint8_t *length);
  state_e cmdFileWrite(uint8_t *data, uint8_t length);
  state_e cmdFileDelete(char *filename);
  state_e cmdFileCopy(char *sourceFilename, char *destFilename);
  state_e cmdFilesystemFormat(filesystem_e filesystem);
  state_e cmdReadStorageSize(filesystem_e storage, uint32_t *totalCapacity, uint32_t *availableSpace);
  state_e cmdSetGraphicsOptions(bool manualFrameUpdate);
  state_e cmdGraphicsManualFrameUpdate(void);
  state_e cmdSetGraphicsBackgroundColor(uint16_t colorRGB565);
  state_e cmdRemoveGraphicsObject(uint8_t objID);
  state_e cmdMoveGraphicsObject(uint8_t objID, int16_t X, int16_t Y);
  state_e cmdSetGraphicsObjectTouchReporting(uint8_t objID, touchreport_e touchOption);
  state_e cmdSetGraphicsObjectZIndex(uint8_t objID, uint8_t zIndex);
  state_e cmdSetGraphicsObjectFontSlot(uint8_t objID, uint8_t fontSlot);
  state_e cmdSetGraphicsObjectFontSize(uint8_t objID, uint8_t fontSize);
  state_e cmdSetGraphicsObjectCornerRadius(uint8_t objID, uint8_t cornerRadius);
  state_e cmdSetGraphicsObjectOpacity(uint8_t objID, uint8_t opacityValue);
  state_e cmdSetGraphicsFillAColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsBorderAColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsControlAColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsTextAColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsFillBColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsBorderBColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsControlBColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsTextBColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsFillCColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsBorderCColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsControlCColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdSetGraphicsTextCColor(uint8_t objID, uint16_t colorRGB565);
  state_e cmdVideoFileLoad(uint8_t objID, char *filename, int16_t X, int16_t Y,
    bool startPlaying,
  uint8_t zIndex, touchreport_e touchReportingMode, uint16_t videoStartFrame, uint16_t videoEndFrame,
  uint8_t playLoops);
  state_e cmdVideoPlay(uint8_t objID);
  state_e cmdVideoPauseToggle(uint8_t objID);
  state_e cmdVideoControl(uint8_t objID, uint16_t videoStartFrame, uint16_t videoEndFrame, uint8_t playLoops);
  state_e cmdImageFileLoad(uint8_t objID, const char *filename,
    int16_t X, int16_t Y,
  bool disabledState, bool isButton, bool toggleButton, bool toggleStartDown, bool enableStateChangeReportPackets, 
  uint8_t zIndex, touchreport_e touchReportingMode);
  state_e cmdImageChangeOptions(uint8_t objID,
    bool disabledState, bool isButton, bool toggleButton, bool toggleStartDown, bool enableStateChangeReportPackets);
  state_e cmdImageFileLoadDownState(uint8_t objID, const char *filename);
  state_e cmdImageFileLoadDisabledState(uint8_t objID, const char *filename);
  state_e cmdImageLoadFromHost(void);
  state_e cmdImageSetButtonState(uint8_t objID, bool setDownState);
  state_e cmdImageReadButtonState(uint8_t objID, bool *currentState, bool *pressSinceLastPoll, bool *releasedSinceLastPoll);
  state_e cmdTTFLoadFont(uint8_t fontSlot, const char *filename, bool deferredLoading);
  state_e cmdTTFUnloadFont(uint8_t fontSlot);
  state_e cmdTTFGetTextDimensions(uint8_t objID, const char *text,
    uint16_t *renderedTextWidth, uint16_t *renderedTextHeight);
  state_e cmdTTFNewText(uint8_t objID, const char *text,
    int16_t X, int16_t Y,
  justify_e justifyText, uint8_t zIndex, touchreport_e touchReporting,
  uint16_t *renderedTextWidth, uint16_t *renderedTextHeight);
  state_e cmdTTFTextChangeOptions(uint8_t objID, const char *text,
    justify_e justifyText,
  uint16_t *renderedTextWidth, uint16_t *renderedTextHeight);
  state_e cmdSketchNewSurface(uint8_t objID,
    int16_t X, int16_t Y,
  uint16_t width, uint16_t height,
  bool backgroundFill, uint8_t zIndex, touchreport_e touchReporting);
  state_e cmdSketchDrawLine(uint8_t objID,
    uint16_t xStart, uint16_t yStart, uint16_t xEnd, uint16_t yEnd,
  uint16_t colorRGB565);
  state_e cmdSketchDrawRectangle(uint8_t objID,
    uint16_t xStart, uint16_t yStart, uint16_t xEnd, uint16_t yEnd,
  uint8_t cornerRadius,
  bool fillWithColor, uint16_t fillRGB565, bool drawBorder, uint16_t borderRGB565);
  state_e cmdSketchDrawCircle(uint8_t objID,
    uint16_t centerX, uint16_t centerY, uint16_t radius,
  bool fillWithColor, uint16_t fillRGB565, bool drawBorder, uint16_t borderRGB565);
  state_e cmdSketchDrawPixel(uint8_t objID, uint16_t X, uint16_t Y, uint16_t colorRGB565);
  state_e cmdButtonNew(uint8_t objID, const char *buttonLabel,
    int16_t X, int16_t Y, uint16_t width, uint16_t height,
  bool disabledState, bool isToggleButton, bool toggleStartDown, bool enableStateChangeReportPackets, 
  uint8_t zIndex, touchreport_e touchReporting);
  state_e cmdButtonSetUpStateLabel(uint8_t objID, const char *buttonLabel);
  state_e cmdButtonSetDownStateLabel(uint8_t objID, const char *buttonLabel);
  state_e cmdButtonSetDisabledStateLabel(uint8_t objID, const char *buttonLabel);
  state_e cmdButtonSetState(uint8_t objID, bool downState);
  state_e cmdImageReadState(uint8_t objID, bool *currentState, bool *pressSinceLastPoll, bool *releasedSinceLastPoll);
  state_e cmdSliderNew(uint8_t objID, const char *textLabel,
      int16_t X, int16_t Y, uint16_t width, uint16_t height,
    int16_t minValue, int16_t maxValue, int16_t startValue,
    bool disabledState, bool showValue, justify_e valueLocation, bool noBackgroundFill, bool enableStateChangeReportPackets, 
    uint8_t zIndex, touchreport_e touchReporting);
  state_e cmdSliderSetValue(uint8_t objID, int16_t newValue);
  state_e cmdSliderReadValue(uint8_t objID, int16_t *currentValue);
  state_e cmdNumberEditNew(uint8_t objID, const char *textLabel,
      int16_t X, int16_t Y, uint16_t width, uint16_t height,
    int16_t minValue, int16_t maxValue, int16_t startValue,
    bool disabledState, justify_e labelPosition, bool noBackgroundFill, bool isPercentage, bool enableStateChangeReportPackets, 
    uint8_t zIndex, touchreport_e touchReporting);
  state_e cmdNumberEditSetValue(uint8_t objID, int16_t newValue);
  state_e cmdNumberEditReadValue(uint8_t objID, int16_t *currentValue);
  state_e cmdCheckboxNew(uint8_t objID, const char *textLabel,
      int16_t X, int16_t Y, uint16_t width, uint16_t height,
    checkbox_e startState, 
    bool tickStateEnabled, bool crossStateEnabled, bool disabledState, justify_e labelPosition,
    bool noBackgroundFill, bool enableStateChangeReportPackets, 
    uint8_t zIndex, touchreport_e touchReporting);
  state_e cmdCheckboxSetValue(uint8_t objID, checkbox_e newState);
  state_e cmdCheckboxReadValue(uint8_t objID, checkbox_e *currentState);
  state_e cmdProgressBarNew(uint8_t objID, const char *textLabel,
      int16_t X, int16_t Y, uint16_t width, uint16_t height,
    int16_t minValue, int16_t maxValue, int16_t startValue,
    bool disabledState, justify_e labelPosition, bool noBackgroundFill, bool isPercentage,
    uint8_t zIndex, touchreport_e touchReporting);
  state_e cmdProgressBarSetValue(uint8_t objID, int16_t newValue);
  state_e cmdProgressBarReadValue(uint8_t objID, int16_t *currentValue);

private:
  //pvt functions
  uint16_t getCRC(uint8_t *data, uint8_t length);
  state_e sendPacketI2C(packet_t *packet);
  state_e sendPacketSPI(packet_t *packet);
  uint16_t recieveDataI2C(uint16_t wait_ms);
  uint16_t recieveDataSPI(uint16_t wait_ms);

  //fifo handling
  typedef struct
  {
      uint16_t		Head;		// first byte of data
      uint16_t		Tail;		// last byte of data
      uint8_t     *Buffer;	// block of memory or array of data
      uint16_t    Length;     // length of the data
  } FIFO_t;

  void FIFO_Init(FIFO_t *FIFO, uint8_t *Buffer, uint16_t Size);
  uint16_t FIFO_Count(FIFO_t const *FIFO);
  uint16_t FIFO_Remaining(FIFO_t const *FIFO);
  bool FIFO_Full(FIFO_t const *FIFO);
  bool FIFO_Empty(FIFO_t const *FIFO);
  void FIFO_PushCircular(FIFO_t *FIFO, uint8_t Data);
  bool FIFO_Push(FIFO_t *FIFO, uint8_t Data);
  bool FIFO_Pop(FIFO_t *FIFO, uint8_t *Data);
  bool FIFO_Peek(FIFO_t *FIFO, uint16_t Position, uint8_t *Data);
  void FIFO_Flush(FIFO_t *FIFO);

  //misc
  bool _debug_enabled;
  uint16_t _cmd_reply_timeout;
  //i2c
  TwoWire *_wire;
  uint8_t _i2c_address;
  //spi
  SPIClass *_spi;
  SPISettings *_spi_settings;
  uint8_t _spi_cs_pin;
  //other pins
  uint8_t _ready_pin;
  //fifo
  FIFO_t _in_fifo;
  uint8_t _in_fifo_buf[IN_FIFO_LENGTH];
  //packet
#ifdef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  
};
