/* ===========================================================================

Crystalfontz CFA039A0-N-V module I2C/SPI Arduino communications library.

This library provides an easy to use interface for the CFA039A0-N-V modules
SPI/I2C packet based communications interface, and command system.

This library has been tested with the Seeeduino v4.2 development board and
the Arduino Uno. It may also be used with other development boards with
minor modifications.

See "CFA039A0-N-V.h" for library configuration.
See the library examples for usage information.

https://www.crystalfontz.com/product/cfa039a0nvdct-480x128-graphic-usb-tft-display-module

Mark Williams (2025)
Crystalfontz America Inc.

Distributed under the "The Unlicense".
http://unlicense.org
This is free and unencumbered software released into the public domain.
For more details, see the website above.

=========================================================================== */

#include "CFA039A0-N-V.h"

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

CFA039A0NV::CFA039A0NV(void)
{
  //var inits / defaults
  _spi = NULL;
  _wire = NULL;
  _debug_enabled = false;
  _cmd_reply_timeout = CFA_COMMAND_TIMEOUT;
};

CFA039A0NV::~CFA039A0NV(void)
{
  //nothing here
};

void CFA039A0NV::begin(TwoWire &wire, uint8_t i2c_address, uint8_t ready_pin)
{
  //I2C coms init
  _wire = &wire;
  _i2c_address = i2c_address;
  _ready_pin = ready_pin;
  //incoming fifo init
  FIFO_Init(&_in_fifo, _in_fifo_buf, IN_FIFO_LENGTH);
  //pin init
  pinMode(_ready_pin, INPUT_PULLUP); //data ready pin mode
}

void CFA039A0NV::begin(SPIClass &spi, SPISettings &spi_settings, uint8_t spi_cs_pin, uint8_t spi_ready_pin)
{
  //SPI coms init
  _spi = &spi;
  _spi_settings = &spi_settings;
  _spi_cs_pin = spi_cs_pin;
  _ready_pin = spi_ready_pin;
  //incoming fifo init
  FIFO_Init(&_in_fifo, _in_fifo_buf, IN_FIFO_LENGTH);
  //pin init
	pinMode(_spi_cs_pin, OUTPUT); //CS pin mode
	digitalWrite(_spi_cs_pin, HIGH); //CS pin deselect
  pinMode(_ready_pin, INPUT_PULLUP); //data ready pin mode
}

void CFA039A0NV::end(void)
{
  //nothing to do
}

void CFA039A0NV::setCmdReplyTimeout(uint16_t wait_ms)
{
  //set new command reply timeout
  _cmd_reply_timeout = wait_ms;
}

void CFA039A0NV::enableDebugOutput()
{
  //enable debugging output
  _debug_enabled = true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

CFA039A0NV::state_e CFA039A0NV::sendPacketI2C(packet_t *packet)
{
  //send on i2c
  uint8_t	remaining, sent;
  uint8_t *pdata = (uint8_t*)packet;
  
  if (_wire != NULL)
  {
    //arduino wire lib cant send more than 32 bytes at a time
    //so send data in 30 byte chunks (30 so CRC can be sent without an extra start/stop)
    remaining = packet->length+2; //not +4 as no crc sent yet
    sent = 0;
    while (remaining > 30)
    {
      _wire->beginTransmission(_i2c_address);
      _wire->write(&pdata[sent], 30);
      _wire->endTransmission();
      sent += 30;
      remaining -= 30;
    }
    //now send what's left
    if (remaining > 0)
    {
      _wire->beginTransmission(_i2c_address);
      _wire->write(&pdata[sent], remaining);
    }
    //now send the crc
    _wire->write((uint8_t*)&packet->crc, 2);
    //done
    _wire->endTransmission();

  }
  else
    return CFA_SENDERROR;
  //done
  return CFA_OK;
}

CFA039A0NV::state_e CFA039A0NV::sendPacketSPI(packet_t *packet)
{
	//any time we write SPI data, we also buffer up incoming data
	//getPacketFromFIFO() takes care of removing any trash bytes
	//between packets
  uint8_t i;
  if (_spi != NULL)
  {
    digitalWrite(_spi_cs_pin, LOW); //slave-select
    _spi->beginTransaction(*_spi_settings);
    FIFO_Push(&_in_fifo, _spi->transfer(packet->command));
    FIFO_Push(&_in_fifo, _spi->transfer(packet->length));
    for (i = 0; i < packet->length; i++)
      FIFO_Push(&_in_fifo, _spi->transfer(packet->data[i]));
    FIFO_Push(&_in_fifo, _spi->transfer(packet->crc.b[0]));
    FIFO_Push(&_in_fifo, _spi->transfer(packet->crc.b[1]));
    digitalWrite(_spi_cs_pin, HIGH); //slave-deselect
    _spi->endTransaction();
  }
  else
    return CFA_SENDERROR;
  //done
	return CFA_OK;
}

CFA039A0NV::state_e CFA039A0NV::sendPacket(packet_t *packet)
{
	//send a packet
  //calculate CRC
	packet->crc.w = getCRC((byte*)packet, 2+packet->length);
	//debug output
  debugPrintPacket("OUT", packet);
  //send the packet
  if (_wire != NULL)
  {
    //send on I2C
    return sendPacketI2C(packet);
  }
  if (_spi != NULL)
  {
    //send on SPI
    return sendPacketSPI(packet);
  }
  return CFA_ERROR;
}

uint16_t CFA039A0NV::recieveDataI2C(uint16_t wait_ms)
{
	//checks interface for incoming data
	//if any present, puts on moduleInFIFO
	uint16_t count = 0;
	//request data from the I2C slave (the CFA module)
	//since we dont know how much data is in the CFA modules outgoing buffer,
	//read 4 bytes at a time until the data ready line goes high
	unsigned long timeout;
  bool cont;

  if (_wire != NULL)
  {
    //wait for ready pin to go low indicating data is ready to read
    //to exit immediatley if ready is not low, use recieveData(0)
    timeout = millis() + wait_ms;
    while (digitalRead(_ready_pin) == HIGH)
    {
      if (millis() > timeout)
        //wait timeout
        return 0;
    }

    //while the ready pin is low, there is data to read, get it
    _wire->setWireTimeout(1000000, true); //1000mS timeout
    while(digitalRead(_ready_pin) == LOW)
    {
      //req I2C data
      _wire->requestFrom(_i2c_address, (uint8_t)4); //get 4 bytes at a time
      while(_wire->available())
      {
        //put it in the incoming fifo
        if (!FIFO_Push(&_in_fifo, _wire->read()))
        {
          //returns false if FIFO is full, stop reading
          break;
        }
        //count it for ret
        count++;
      }
    }

    //sometimes the last byte is missed by the Arduino, get data once more
    _wire->requestFrom(_i2c_address, (uint8_t)4); //get 4 bytes at a time
    while(_wire->available())
    {
      //put it in the incoming fifo
      if (!FIFO_Push(&_in_fifo, _wire->read()))
      {
        //returns false if FIFO is full, stop reading
        break;
      }
      //count it for ret
      count++;      
    }
  }
	//return data read qty
	return count;
}


uint16_t CFA039A0NV::recieveDataSPI(uint16_t wait_ms)
{
	//checks interface for incoming data
	//if any present, puts on moduleInFIFO
	uint16_t count = 0;
	//request data from the SPI slave (the CFA module)
	//since we dont know how much data is in the CFA modules outgoing buffer,
	//read 8 bytes at a time until the data ready line goes high
	uint8_t data;
	unsigned long timeout;

	//wait for ready pin to go low indicating data is ready to read
	//to exit immediatley if ready is not low, use recieveData(0)
  if (_spi != NULL)
  {
    timeout = millis() + wait_ms;
    while (digitalRead(_ready_pin) == HIGH)
    {
      if (millis() > timeout)
        //wait timeout
        return 0;
    }
    _spi->beginTransaction(*_spi_settings);
    digitalWrite(_spi_cs_pin, LOW); //slave-select
    while (digitalRead(_ready_pin) == LOW)
    {
      //ready data while the ready pin remains low
      //to get data we need to send data, so just send 0xFF's
      data = _spi->transfer(0xFF);
      //put it in our incoming buffer. getPacket() takes care of
      //removing any trash bytes between packets
      if (!FIFO_Push(&_in_fifo, data))
        //returns false if FIFO is full
        //stop reading
        break;
      //loop again, try to get more data
      count++;
    }
    //done
    digitalWrite(_spi_cs_pin, HIGH); //slave-deselect
    _spi->endTransaction();
  }

	//return data read qty
	return count;
}

uint16_t CFA039A0NV::recieveData(uint16_t wait_ms)
{
  //checks interface for incoming data
	//if any present, puts on moduleInFIFO
  //returns the amount of new data read from the interface
  if (_wire != NULL)
  {
    //get data from I2C
    return recieveDataI2C(wait_ms);
  }
  if (_spi != NULL)
  {
    return recieveDataSPI(wait_ms);
  }
  //done
  return 0;
}

bool CFA039A0NV::getPacket(packet_t *packet)
{
	//return first packet found on input FIFO
	uint16_t	positionCount;
	uint16_t	inQueue;
	uint16_t	i;
  uint8_t   dataBuf;

	//check for enough data in fifo
	inQueue = FIFO_Count(&_in_fifo);
	if (inQueue < PACKET_HEADER_SIZE)
		//needs to have 4 or more bytes to contain valid packet
		return false;

	//check for packets
	for (positionCount = 0; positionCount < inQueue - (PACKET_HEADER_SIZE-1); positionCount++)
	{
    //get command number and packet length from fifo
		FIFO_Peek(&_in_fifo, positionCount, &packet->command);
		FIFO_Peek(&_in_fifo, positionCount + 1, &packet->length);
		//check command & length
		if (
				(packet->command & (PCMD_REPLY | PCMD_REPORT | PCMD_ERROR)) &&
				(packet->length < PACKET_DATA_SIZE+1) &&
				(inQueue > (positionCount + packet->length + PACKET_HEADER_SIZE - 1))
				)
		{
			//valid packet type and length
      //get data
			for (i = 0; i < packet->length; i++)
				FIFO_Peek(&_in_fifo, positionCount + 2 + i, &packet->data[i]);
      //get crc
			FIFO_Peek(&_in_fifo, positionCount + 2 + packet->length + 0, &packet->crc.b[0]);
			FIFO_Peek(&_in_fifo, positionCount + 2 + packet->length + 1, &packet->crc.b[1]);
      //check crc
			if (getCRC((byte*)packet, 2+packet->length) == packet->crc.w)
			{
				//valid crc, is a packet (packet ready to be returned)
				//remove packet (and any bad data before it) from the fifo
        for (i = 0; i < (positionCount + packet->length + PACKET_HEADER_SIZE); i++)
          FIFO_Pop(&_in_fifo, &dataBuf);
	      //debug output
        debugPrintPacket("IN", packet);
				//done
				return true;
			}
		}
	}

	//no packet found, pop off bad bytes (fifo-tail -> (fifo-head - PACKET_MAX_DATA_LEN))
	if (inQueue > PACKET_MAX_SIZE(PACKET_CFA039_DATA_SIZE))
	{
		//quick adjust FIFO length
    for (i = 0; i < (uint16_t)(packet->length + PACKET_HEADER_SIZE); i++)
      FIFO_Pop(&_in_fifo, &dataBuf);
	}

	//done, no packets found
	return false;
}

CFA039A0NV::state_e CFA039A0NV::cmdGetReply(uint8_t command, uint8_t subCommand, packet_t *packet, uint16_t wait_ms)
{
  //wait for incoming reply matching cammand/subcommand
  //set subCommand to 0xFF (255) if not matching a sub command
  //packet must not be null
  unsigned long timeout = millis() + wait_ms;
  while (millis() < timeout)
  {
    //keep checking for more data until the timeout
    if (recieveData(wait_ms/10) != 0)
    {
      //got some data in the fifo, see if there is a packet in it
      if (getPacket(packet) == true)
      {
        //we got a packet, check the command / subcommand
        //top two bits are disregarded
        if ((packet->command & 0x3F) == (command & 0x3F))
        {
          //matched command
          if ((subCommand == 0xFF) || ((packet->length > 0) && (packet->data[0] == subCommand)))
          {
            //no subcommand, or matched subcommand
            //we got the packet we were expecting!
            return CFA_OK;
          }
        }
      }
    }
  }
  if (_debug_enabled)
    Serial.println(F("timeout waiting for reply"));
  return CFA_TIMEOUT;
}

CFA039A0NV::state_e CFA039A0NV::sendPacketAndGetReply(packet_t *packet, uint16_t wait_ms, bool matchSubCommand)
{
  //sends the packet and waits for the reply
  //the reply is returned in "packet"
  state_e ret;
  uint8_t command, subcommand;

  //keep the original packet details for later
  command = packet->command;
  subcommand = packet->data[0];

  //send the packet
  ret = sendPacket(packet);
  if (ret != CFA_OK)
    return ret;

  //setup the timeout
  unsigned long timeout = millis() + wait_ms;

  //wait for the reply packet
  while (millis() < timeout)
  {
    //keep checking for more data until the timeout
    int g = recieveData(wait_ms/10);
    /*Serial.print("GOT BYTES: ");
    Serial.println(g);
    Serial.print("IN FIFO: ");
    for (int i = 0; i < FIFO_Count(&_in_fifo); i++)
    {
      uint8_t d;
      FIFO_Peek(&_in_fifo, i, &d);
      Serial.print(d, HEX);
      Serial.print(" ");
    }
    Serial.println("");*/
    //check for a new packet
    if (getPacket(packet) == true)
    {
      //we got a packet, check the command / subcommand
      //top two bits are disregarded for now
      if ((packet->command & 0x3F) == (command & 0x3F))
      {
        //matched command
        if ((matchSubCommand == false) || (subcommand == packet->data[0]))
        {
          //subcommand match, or match not needed
          if ((packet->command & PCMD_REPLY) == PCMD_REPLY)
            //normal reply
            return CFA_OK;
          if ((packet->command & PCMD_ERROR) == PCMD_ERROR)
            //error reply
            return CFA_CMDERROR;
          //odd, shouldnt happen
          return CFA_ERROR;
        }
      }
      //if we get to here the packet didnt match, so we continue...
    }
  }
  //
  if (_debug_enabled)
    Serial.println(F("timeout waiting for reply"));
  return CFA_TIMEOUT;
}

CFA039A0NV::state_e CFA039A0NV::getReportPacket(PacketReports_t reportType, packet_t *reportPacket, uint16_t waitMS)
{
  //wait for incoming reply matching the reportType or, any type of report if reportType is =0xFF
  //waits for up to waitMS milliseconds, or forever if waitMS==0
  //reportPacket must not be null
  unsigned long timeout = millis() + waitMS;

  while ((millis() < timeout) || (waitMS == 0))
  {
    //keep checking for more data until the timeout
    if (recieveData(waitMS/10) != 0)
    {
      //got some data in the fifo, see if there is a packet in it
      if (getPacket(reportPacket) == true)
      {
        //we got a packet, check if it matches the reportType
        if (
          (reportPacket->command == reportType) ||
          ((reportType == 0xFF) && ((reportPacket->command & 0xC0) == PCMD_REPORT))
          )
        {
          //matched report packet
          return CFA_OK;
        }
      }
    }
  }
  return CFA_TIMEOUT;
}

void CFA039A0NV::clearBuffers(void)
{
  FIFO_Empty(&_in_fifo);
}

void CFA039A0NV::debugPrintPacket(const char *prepend, packet_t *packet)
{
	//write debug packet info to host
  //command/reply/report/error numbers are stripped of the packet type
	uint8_t i;

  if (_debug_enabled == false)
    //do nothing
    return;

  if (prepend != NULL)
    Serial.print(prepend);
  if ((packet->command & PCMD_ERROR) == PCMD_ERROR)
  {
    //error packet
    Serial.print(F(" ERR:")); 
  }
  else if ((packet->command & PCMD_REPORT) == PCMD_REPORT)
  {
    //report packet
    Serial.print(F(" RPT:"));
  }
  else if ((packet->command & PCMD_REPLY) == PCMD_REPLY)
  {
    //reply packet
    Serial.print(F(" RPY:"));
  }
  else
  {
    //command packet
    Serial.print(F(" CMD:"));
  }
  Serial.print(packet->command & 0x3F);
  //
  Serial.print(F(" L:")); Serial.print(packet->length);
  //
  Serial.print(F(" D:"));
	for (i = 0; i < packet->length; i++)
  {
    Serial.print(packet->data[i], DEC);
    Serial.print(F(","));
  }
  //
  Serial.print(F(" CRC:0x"));
  Serial.print(packet->crc.b[0], HEX);
  Serial.print(F(",0x"));
  Serial.print(packet->crc.b[1], HEX);
  //
  Serial.print(F("\n"));
  Serial.flush();
}

//CRC lookup table to avoid bit-shifting loops
//fast, but uses some extra program space
const uint16_t CRCLookupTable[256] =
	{0x00000,0x01189,0x02312,0x0329B,0x04624,0x057AD,0x06536,0x074BF,
	0x08C48,0x09DC1,0x0AF5A,0x0BED3,0x0CA6C,0x0DBE5,0x0E97E,0x0F8F7,
	0x01081,0x00108,0x03393,0x0221A,0x056A5,0x0472C,0x075B7,0x0643E,
	0x09CC9,0x08D40,0x0BFDB,0x0AE52,0x0DAED,0x0CB64,0x0F9FF,0x0E876,
	0x02102,0x0308B,0x00210,0x01399,0x06726,0x076AF,0x04434,0x055BD,
	0x0AD4A,0x0BCC3,0x08E58,0x09FD1,0x0EB6E,0x0FAE7,0x0C87C,0x0D9F5,
	0x03183,0x0200A,0x01291,0x00318,0x077A7,0x0662E,0x054B5,0x0453C,
	0x0BDCB,0x0AC42,0x09ED9,0x08F50,0x0FBEF,0x0EA66,0x0D8FD,0x0C974,
	0x04204,0x0538D,0x06116,0x0709F,0x00420,0x015A9,0x02732,0x036BB,
	0x0CE4C,0x0DFC5,0x0ED5E,0x0FCD7,0x08868,0x099E1,0x0AB7A,0x0BAF3,
	0x05285,0x0430C,0x07197,0x0601E,0x014A1,0x00528,0x037B3,0x0263A,
	0x0DECD,0x0CF44,0x0FDDF,0x0EC56,0x098E9,0x08960,0x0BBFB,0x0AA72,
	0x06306,0x0728F,0x04014,0x0519D,0x02522,0x034AB,0x00630,0x017B9,
	0x0EF4E,0x0FEC7,0x0CC5C,0x0DDD5,0x0A96A,0x0B8E3,0x08A78,0x09BF1,
	0x07387,0x0620E,0x05095,0x0411C,0x035A3,0x0242A,0x016B1,0x00738,
	0x0FFCF,0x0EE46,0x0DCDD,0x0CD54,0x0B9EB,0x0A862,0x09AF9,0x08B70,
	0x08408,0x09581,0x0A71A,0x0B693,0x0C22C,0x0D3A5,0x0E13E,0x0F0B7,
	0x00840,0x019C9,0x02B52,0x03ADB,0x04E64,0x05FED,0x06D76,0x07CFF,
	0x09489,0x08500,0x0B79B,0x0A612,0x0D2AD,0x0C324,0x0F1BF,0x0E036,
	0x018C1,0x00948,0x03BD3,0x02A5A,0x05EE5,0x04F6C,0x07DF7,0x06C7E,
	0x0A50A,0x0B483,0x08618,0x09791,0x0E32E,0x0F2A7,0x0C03C,0x0D1B5,
	0x02942,0x038CB,0x00A50,0x01BD9,0x06F66,0x07EEF,0x04C74,0x05DFD,
	0x0B58B,0x0A402,0x09699,0x08710,0x0F3AF,0x0E226,0x0D0BD,0x0C134,
	0x039C3,0x0284A,0x01AD1,0x00B58,0x07FE7,0x06E6E,0x05CF5,0x04D7C,
	0x0C60C,0x0D785,0x0E51E,0x0F497,0x08028,0x091A1,0x0A33A,0x0B2B3,
	0x04A44,0x05BCD,0x06956,0x078DF,0x00C60,0x01DE9,0x02F72,0x03EFB,
	0x0D68D,0x0C704,0x0F59F,0x0E416,0x090A9,0x08120,0x0B3BB,0x0A232,
	0x05AC5,0x04B4C,0x079D7,0x0685E,0x01CE1,0x00D68,0x03FF3,0x02E7A,
	0x0E70E,0x0F687,0x0C41C,0x0D595,0x0A12A,0x0B0A3,0x08238,0x093B1,
	0x06B46,0x07ACF,0x04854,0x059DD,0x02D62,0x03CEB,0x00E70,0x01FF9,
	0x0F78F,0x0E606,0x0D49D,0x0C514,0x0B1AB,0x0A022,0x092B9,0x08330,
	0x07BC7,0x06A4E,0x058D5,0x0495C,0x03DE3,0x02C6A,0x01EF1,0x00F78
};

uint16_t CFA039A0NV::getCRC(uint8_t *data, uint8_t length)
{
	register uint16_t newCrc = 0xFFFF;
  //calculate the crc
	while (length--)
		newCrc = (newCrc >> 8) ^ CRCLookupTable[(newCrc ^ *data++) & 0xff];
	//make this crc match the one's complement that is sent in the packet
	return (~newCrc);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void CFA039A0NV::FIFO_Init(FIFO_t *FIFO, uint8_t *Buffer, uint16_t Size)
{
	FIFO->Buffer = Buffer;
	FIFO->Length = Size;
	FIFO->Head = 0;
	FIFO->Tail = 0;
}

uint16_t CFA039A0NV::FIFO_Count(FIFO_t const *FIFO)
{
    return FIFO->Head - FIFO->Tail;
}

uint16_t CFA039A0NV::FIFO_Remaining(FIFO_t const *FIFO)
{
	return FIFO->Length - FIFO_Count(FIFO);
}

bool CFA039A0NV::FIFO_Full(FIFO_t const *FIFO)
{
    return (FIFO_Count(FIFO) == FIFO->Length) ? true : false;
}

bool CFA039A0NV::FIFO_Empty(FIFO_t const *FIFO)
{
	return (FIFO->Head == FIFO->Tail) ? true : false;
}

void CFA039A0NV::FIFO_PushCircular(FIFO_t *FIFO, uint8_t Data)
{
	if (FIFO_Full(FIFO))
		//full, remove last byte
		FIFO->Tail++;
	FIFO->Buffer[FIFO->Head % FIFO->Length] = Data;
	FIFO->Head++;
}

bool CFA039A0NV::FIFO_Push(FIFO_t *FIFO, uint8_t Data)
{
#ifndef FIFO_NOCHECKS
	if (FIFO_Full(FIFO))
		return false;
#endif
	FIFO->Buffer[FIFO->Head % FIFO->Length] = Data;
	FIFO->Head++;
    return true;
}

bool CFA039A0NV::FIFO_Pop(FIFO_t *FIFO, uint8_t *Data)
{
#ifndef FIFO_NOCHECKS
	if (FIFO_Empty(FIFO))
		return false;
#endif
	*Data = FIFO->Buffer[FIFO->Tail % FIFO->Length];
	FIFO->Tail++;
    return true;
}

bool CFA039A0NV::FIFO_Peek(FIFO_t *FIFO, uint16_t Position, uint8_t *Data)
{
#ifndef FIFO_NOCHECKS
	if (Position >= FIFO_Count(FIFO))
		return false;
#endif
	*Data = FIFO->Buffer[(FIFO->Tail + Position) % FIFO->Length];
	return true;
}

void CFA039A0NV::FIFO_Flush(FIFO_t *FIFO)
{
	FIFO->Tail = FIFO->Head;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

//command 0
CFA039A0NV::state_e CFA039A0NV::cmdPing(uint8_t *data, uint8_t length)
{
  //returns CMD_OK if the command is sent ok, and the module replies with the
  //correct ping data
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint16_t i;
  state_e ret;
  packet.command = PCMD2_PING;
  memcpy(packet.data, data, length);
  packet.length = length;
  ret = sendPacket(&packet);
  if (ret == CFA_OK)
  {
    //wait for reply packet
    ret = cmdGetReply(PCMD2_PING, 0xFF, &packet, _cmd_reply_timeout);
    if (ret == CFA_OK)
    {
      //check returned packet type
      if (packet.command == (PCMD2_PING | PCMD_REPLY))
      {
        //check data
        for (i = 0; i < packet.length; i++)
          if (data[i] != packet.data[i])
          {
            //not a match
            return CFA_BADDATA;
          }
        //all ok
        return CFA_OK;
      }
      else
        return CFA_CMDERROR;
    }
  }
  return ret;
}

//command 1
CFA039A0NV::state_e CFA039A0NV::cmdGetModuleVersion(char verString[32])
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret = CFA_ERROR;
  packet.command = PCMD2_GET_VER;
  packet.length = 0;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    //got version number
    memcpy(verString, packet.data, packet.length);
    verString[packet.length] = 0; //null terminate string
    //done
    return CFA_OK;
  }
  return ret;
}

CFA039A0NV::state_e CFA039A0NV::cmdGetModuleSerialNumber(char serialString[32])
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret = CFA_ERROR;
  packet.command = PCMD2_GET_VER;
  packet.length = 1;
  packet.data[0] = 1;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    //got serial number
    memcpy(serialString, packet.data, packet.length);
    serialString[packet.length] = 0; //null terminate string
    //done
    return CFA_OK;
  }
  return ret;
}

//command 2
CFA039A0NV::state_e CFA039A0NV::cmdWriteUserFlash(char *data, uint8_t length)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_WRITE_USER_FLASH;
  packet.length = length;
  memcpy(packet.data, data, length);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}

//command 3
CFA039A0NV::state_e CFA039A0NV::cmdReadUserFlash(char *data, uint8_t *length)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_READ_USER_FLASH;
  packet.length = 1;
  packet.data[0] = *length;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    *length = packet.length;
    memcpy(data, packet.data, *length);
  }
  return ret;
}

//command 4
CFA039A0NV::state_e CFA039A0NV::cmdStoreBootState(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_STORE_BOOT_STATE;
  packet.length = 0;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}

//command 5
CFA039A0NV::state_e CFA039A0NV::cmdLoadSavedSettings(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_REBOOT;
  packet.length = 3;
  packet.data[0] = 8; packet.data[1] = 18; packet.data[2] = 99;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}
CFA039A0NV::state_e CFA039A0NV::cmdRestartHost(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_REBOOT;
  packet.length = 3;
  packet.data[0] = 12; packet.data[1] = 28; packet.data[2] = 97;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}  
CFA039A0NV::state_e CFA039A0NV::cmdPowerOffHost(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_REBOOT;
  packet.length = 3;
  packet.data[0] = 3; packet.data[1] = 11; packet.data[2] = 95;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}  
CFA039A0NV::state_e CFA039A0NV::cmdModuleRestart(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_REBOOT;
  packet.length = 3;
  packet.data[0] = 8; packet.data[1] = 25; packet.data[2] = 48;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}
CFA039A0NV::state_e CFA039A0NV::cmdRestoreDefaultsReset(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_REBOOT;
  packet.length = 3;
  packet.data[0] = 10; packet.data[1] = 8; packet.data[2] = 98;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}
CFA039A0NV::state_e CFA039A0NV::cmdRestartIntoBootloader(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_REBOOT;
  packet.length = 3;
  packet.data[0] = 88; packet.data[1] = 207; packet.data[2] = 5;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}

//command 6
CFA039A0NV::state_e CFA039A0NV::cmdClearDisplay(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_CLEAR_LCD;
  packet.length = 0;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}

//command 14
CFA039A0NV::state_e CFA039A0NV::cmdSetDisplayBrightness(uint8_t backlightPercent)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_LCD_AND_KEYPAD_BACKLIGHT;
  packet.length = 1;
  packet.data[0] = (backlightPercent > 100) ? 100 : backlightPercent;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}
CFA039A0NV::state_e CFA039A0NV::cmdSetDisplayAndKeypadBrightness(uint8_t backlightPercent, uint8_t keypadPercent)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_LCD_AND_KEYPAD_BACKLIGHT;
  packet.length = 2;
  packet.data[0] = (backlightPercent > 100) ? 100 : backlightPercent;
  packet.data[1] = (keypadPercent > 100) ? 100 : keypadPercent;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}
CFA039A0NV::state_e CFA039A0NV::cmdSetKeypadAndIndicatorLEDColor(keypad_e keys, indicator_e indicators, uint8_t R, uint8_t G, uint8_t B)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_LCD_AND_KEYPAD_BACKLIGHT;
  packet.length = 5;
  packet.data[0] = (uint8_t)indicators;
  packet.data[1] = (uint8_t)keys;
  packet.data[2] = R;
  packet.data[3] = G;
  packet.data[4] = B;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
} 
CFA039A0NV::state_e CFA039A0NV::cmdSetKeypadAndIndicatorLEDColor(keypad_e keys, indicator_e indicators, uint32_t RGB888)
{
  return cmdSetKeypadAndIndicatorLEDColor(keys, indicators, (RGB888 >> 16) & 0xFF, (RGB888 >> 8) & 0xFF, RGB888 & 0xFF);
}

CFA039A0NV::state_e CFA039A0NV::cmdSetLightingIdleDimming(dimmingeffects_e dimmingEffects, actmonitor_e activityMonitor,
  uint8_t level1TimoutSeconds, uint8_t level1DimPercent, uint8_t level2TimoutSeconds, uint8_t level2DimPercent)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_LCD_AND_KEYPAD_BACKLIGHT;
  packet.length = 6;
  packet.data[0] = (uint8_t)dimmingEffects;
  packet.data[1] = (uint8_t)activityMonitor;
  packet.data[2] = level1TimoutSeconds;
  packet.data[3] = level2TimoutSeconds;
  packet.data[4] = (level1DimPercent > 100) ? 100 : level1DimPercent;
  packet.data[5] = (level2DimPercent > 100) ? 100 : level1DimPercent;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}

//command 23
CFA039A0NV::state_e CFA039A0NV::cmdSetKeypadReporting(keypad_e pressReport, keypad_e releaseReport)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_KEY_REPORTING;
  packet.length = 2;
  packet.data[0] = (uint8_t)pressReport;
  packet.data[1] = (uint8_t)releaseReport;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);
}

//command 24
CFA039A0NV::state_e CFA039A0NV::cmdPollKeypadState(keypad_e *currentState, keypad_e *pressedSinceLastPoll, keypad_e *releasedSinceLastPoll)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD1_READ_KEYPAD_STATE;
  packet.length = 0;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    //got values
    if (currentState != NULL) *currentState = (keypad_e)packet.data[0];
    if (pressedSinceLastPoll != NULL) *pressedSinceLastPoll = (keypad_e)packet.data[1];
    if (releasedSinceLastPoll != NULL) *releasedSinceLastPoll = (keypad_e)packet.data[2];
  }
  return ret;
}

//command 25
CFA039A0NV::state_e CFA039A0NV::cmdSetTouchscreenReporting(touchreport_e reportingOption)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_TOUCH_REPORTING;
  packet.length = 1;
  packet.data[0] = (uint8_t)reportingOption;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

//command 28
CFA039A0NV::state_e CFA039A0NV::cmdATXOptions(atx_e atxOptions, uint8_t pulseLength)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_ATX_POWER_CONTROL;
  packet.length = 2;
  packet.data[0] = (uint8_t)atxOptions;
  packet.data[1] = pulseLength;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

//command 29
CFA039A0NV::state_e CFA039A0NV::cmdWatchdogReset(uint8_t watchdogTimeoutSeconds)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_RESET_WATCHDOG;
  packet.length = 1;
  packet.data[0] = watchdogTimeoutSeconds;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}
CFA039A0NV::state_e CFA039A0NV::cmdWatchdogDisable(void)
{
  return cmdWatchdogReset(0);
}

//command 33
CFA039A0NV::state_e CFA039A0NV::cmdSetInterfaceSerialOptions(bool enableInterface, interfaceoption_e options, serialbaud_e baudRate)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_CONFIGURE_INTERFACE;
  packet.length = 3;
  packet.data[0] = INTERFACE_SERIAL;
  packet.data[1] = (uint8_t)options;
  packet.data[2] = (baudRate > 2) ? SERIALBAUD_115200 : baudRate;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetInterfaceUSBOptions(bool enableInterface, interfaceoption_e options)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_CONFIGURE_INTERFACE;
  packet.length = 2;
  packet.data[0] = INTERFACE_USB;
  packet.data[1] = (uint8_t)options;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetInterfaceSPIOptions(bool enableInterface, interfaceoption_e options, spimode_e spiMode)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_CONFIGURE_INTERFACE;
  packet.length = 4;
  packet.data[0] = INTERFACE_SPI;
  packet.data[1] = (uint8_t)options;
  packet.data[2] = (uint8_t)spiMode;
  packet.data[3] = 0;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetInterfaceI2COptions(bool enableInterface, interfaceoption_e options, uint8_t i2cAddress, i2cspeed_e i2cSpeed)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_CONFIGURE_INTERFACE;
  packet.length = 4;
  packet.data[0] = INTERFACE_SPI;
  packet.data[1] = (uint8_t)options;
  packet.data[2] = (i2cAddress > 0x7F) ? CFA039A0NV_DEFAULT_I2C_ADDRESS : i2cAddress;
  packet.data[3] = (uint8_t)i2cSpeed;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

//command 34


CFA039A0NV::state_e CFA039A0NV::cmdSetGPIOPin(gpiopins_e gpioIndex, uint8_t outPWMPercent, drivemodes_e driveMode)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_GPIO_PIN;
  packet.length = 3;
  packet.data[0] = (uint8_t)gpioIndex;
  packet.data[1] = (outPWMPercent > 100) ? 100 : outPWMPercent;
  packet.data[2] = driveMode | (1<<3); //user mode
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGPIOPinToDefault(gpiopins_e gpioIndex)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_GPIO_PIN;
  packet.length = 3;
  packet.data[0] = (uint8_t)gpioIndex;
  packet.data[1] = 0;
  packet.data[2] = 0; //default mode
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout);  
}

CFA039A0NV::state_e CFA039A0NV::cmdReadGPIOPin(gpiopins_e gpioIndex, bool *currentPinState, bool *fallSinceLastPoll, bool *riseSinceLastPoll, uint8_t *outPWMPercent, drivemodes_e *driveMode, bool *userMode)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_GPIO_PIN;
  packet.length = 1;
  packet.data[0] = (uint8_t)gpioIndex;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    if (packet.length != 4)
      return CFA_CMDERROR;
    if (currentPinState != NULL) *currentPinState = packet.data[1] & 0b001;
    if (fallSinceLastPoll != NULL) *fallSinceLastPoll = packet.data[1] & 0b010;
    if (riseSinceLastPoll != NULL) *riseSinceLastPoll = packet.data[1] & 0b100;
    if (outPWMPercent != NULL) *outPWMPercent = packet.data[2];
    if (driveMode != NULL) *driveMode = (drivemodes_e)(packet.data[3] & 0b111);
    if (userMode != NULL) *userMode = packet.data[3] & 0b1000;
  }
  return ret;
}

CFA039A0NV::state_e CFA039A0NV::cmdReadGPIOPinADC(gpiopins_e gpioIndex, uint16_t *avgSinceLastPollx16, uint16_t *minSinceLastPoll, uint16_t *maxSinceLastPoll)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_GPIO_PIN;
  packet.length = 1;
  packet.data[0] = (uint8_t)gpioIndex;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    if (packet.length != 7)
      return CFA_CMDERROR;
    if (avgSinceLastPollx16 != NULL) *avgSinceLastPollx16 = (packet.data[2] << 8) | packet.data[1];
    if (minSinceLastPoll != NULL) *minSinceLastPoll = (packet.data[4] << 8) | packet.data[3];
    if (maxSinceLastPoll != NULL) *maxSinceLastPoll = (packet.data[6] << 8) | packet.data[5];
  }
  return ret;
}

CFA039A0NV::state_e CFA039A0NV::cmdReadGPIOPinADC(gpiopins_e gpioIndex, float *avgSinceLastPollVolts, float *minSinceLastPollVolts, float *maxSinceLastPollVolts)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_GPIO_PIN;
  packet.length = 1;
  packet.data[0] = (uint8_t)gpioIndex;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout);
  if (ret == CFA_OK)
  {
    if (packet.length != 7)
      return CFA_CMDERROR;
    if (avgSinceLastPollVolts != NULL)
    {
      uint16_t v = (packet.data[2] << 8) | packet.data[1];
      *avgSinceLastPollVolts = (float)v / 16.0 * 3.3 / 4095.0;
    }
    if (minSinceLastPollVolts != NULL)
    {
      uint16_t v = (packet.data[4] << 8) | packet.data[3];
      *minSinceLastPollVolts = (float)v * 3.3 / 4095.0;
    }
    if (maxSinceLastPollVolts != NULL)
    {
      uint16_t v = (packet.data[6] << 8) | packet.data[5];
      *maxSinceLastPollVolts = (float)v * 3.3 / 4095.0;
    }
  }
  return ret;
}

//command 37 - subcommand 0
CFA039A0NV::state_e CFA039A0NV::cmdReadFBSCABCount(uint8_t *fbscabCount)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 1;
  packet.data[0] = SCMD_FBSCAB_QUERY;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 2)
      return CFA_CMDERROR;
    if (fbscabCount != NULL)
      *fbscabCount = packet.data[1];
  }
  return ret;
}
CFA039A0NV::state_e CFA039A0NV::cmdReadFBSCABSerialNumber(uint8_t fbscabIndex, char fbscabSerialNumber[16])
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD_FBSCAB_QUERY;
  packet.data[1] = fbscabIndex;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 18)
      return CFA_CMDERROR;
    memcpy(fbscabSerialNumber, packet.data, 16);
  }
  return ret;
}

//command 37 - subcommand 1
CFA039A0NV::state_e CFA039A0NV::cmdSetFBSCABFanPower(uint8_t fbscabIndex, uint8_t fan1Power, uint8_t fan2Power, uint8_t fan3Power, uint8_t fan4Power)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 6;
  packet.data[0] = SCMD_FBSCAB_FAN;
  packet.data[1] = fbscabIndex;
  packet.data[2] = (fan1Power > 100) ? 100 : fan1Power;
  packet.data[3] = (fan2Power > 100) ? 100 : fan2Power;
  packet.data[4] = (fan3Power > 100) ? 100 : fan3Power;
  packet.data[5] = (fan4Power > 100) ? 100 : fan4Power;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetFBSCABFanPowerAndSettings(uint8_t fbscabIndex, uint8_t fan1Power, uint8_t fan2Power, uint8_t fan3Power, uint8_t fan4Power,
  fanbitmask_e enableFailSafe, uint8_t failSafeTimeout,
  uint8_t fan1GlitchDelay, uint8_t fan2GlitchDelay, uint8_t fan3GlitchDelay, uint8_t fan4GlitchDelay)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 12;
  packet.data[0] = SCMD_FBSCAB_FAN;
  packet.data[1] = fbscabIndex;
  packet.data[2] = (fan1Power > 100) ? 100 : fan1Power;
  packet.data[3] = (fan2Power > 100) ? 100 : fan2Power;
  packet.data[4] = (fan3Power > 100) ? 100 : fan3Power;
  packet.data[5] = (fan4Power > 100) ? 100 : fan4Power;
  packet.data[6] = (uint8_t)enableFailSafe;
  packet.data[7] = failSafeTimeout;
  packet.data[8] = fan1GlitchDelay;
  packet.data[9] = fan2GlitchDelay;
  packet.data[10] = fan3GlitchDelay;
  packet.data[11] = fan4GlitchDelay;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

//command 37 - subcommand 2
uint16_t calcFanRPM(uint8_t cycles, uint8_t timerLSB, uint8_t timerMSB)
{
    uint16_t rpm = 0;
    uint16_t timer_ticks = timerLSB | (timerMSB << 8);
    if ((cycles < 3) || (timerLSB == 0xFF))
    {
      //fan stopped or invalid value
      rpm = 0;
    }
    else if (cycles < 4)
    {
      //fan too slow, return an RPM value of 1
      rpm = 1;
    }
    else
    {
      //calculate RPM
      //pulses_per_revolution = 2; //specific to each fan, most commonly 2
      //fan_rpm = ((27692308L / pulses_per_revolution) * (cycles - 3)) / (float)timer_ticks;  
      rpm = ((27692308L / 2) * (cycles - 3)) / timer_ticks;
    }
    return rpm;
}

CFA039A0NV::state_e CFA039A0NV::cmdReadFanTach(uint8_t fbscabIndex, uint16_t *fan1RPM, uint16_t *fan2RPM, uint16_t *fan3RPM, uint16_t *fan4RPM)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD_FBSCAB_FAN_RPM;
  packet.data[1] = fbscabIndex;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if ((packet.length != 14) || (packet.data[1] != fbscabIndex))
      return CFA_CMDERROR;
    if (fan1RPM != NULL) *fan1RPM = calcFanRPM(packet.data[2], packet.data[3], packet.data[4]);
    if (fan2RPM != NULL) *fan2RPM = calcFanRPM(packet.data[5], packet.data[6], packet.data[7]);
    if (fan3RPM != NULL) *fan3RPM = calcFanRPM(packet.data[8], packet.data[9], packet.data[10]);
    if (fan4RPM != NULL) *fan4RPM = calcFanRPM(packet.data[11], packet.data[12], packet.data[13]);
  }
  return ret;
}

CFA039A0NV::state_e CFA039A0NV::cmdReadDOWTemperature(uint8_t fbscabIndex, uint8_t dowSensorIndex, uint16_t *tempX16)
{
  //returns the temperature *16, or UINT16_MAX if invalid
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD_FBSCAB_TEMP_VALUE;
  packet.data[1] = fbscabIndex;
  packet.data[2] = dowSensorIndex;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if ((packet.length != 5) || (packet.data[1] != fbscabIndex) || (packet.data[2] != dowSensorIndex))
      return CFA_CMDERROR;
    //check CRC
    if (packet.data[4] & 0b11000000)
    {
      //one or both of the two top bits were set, which means
      //the sensor wasnt present, no data, or crc check failed
      if (tempX16 != NULL) *tempX16 = UINT16_MAX;
    }
    //calc temperature
    uint16_t raw = (packet.data[3] | (packet.data[4] << 8)) & 0x3F;
    if (tempX16 != NULL)
      *tempX16 = raw;
  }

  return ret;  
}

CFA039A0NV::state_e CFA039A0NV::cmdReadDOWTemperature(uint8_t fbscabIndex, uint8_t dowSensorIndex, float *temperature)
{
  //returns the temperature, or FLT_MAX if invalid
  state_e ret;
  uint16_t tempraw;
  ret = cmdReadDOWTemperature(fbscabIndex, dowSensorIndex, &tempraw);
  if ((ret != CFA_OK) || (tempraw == UINT16_MAX))
  {
    if (temperature != NULL) *temperature = FLT_MAX;
    return ret;
  }
  if (temperature != NULL)
    *temperature = tempraw / 16.0f;    
  return ret;  
}

CFA039A0NV::state_e CFA039A0NV::cmdFBSCABResetAndSearch(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 1;
  packet.data[0] = SCMD_FBSCAB_RESET;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFBSCABSetAutoFanControl(uint8_t fbscabIndex, uint8_t fanIndex, bool afcEnabled, bool fanOffUnderMinPower,
  uint8_t afcResponsiveness, uint8_t dowTempIndex, int8_t targetTemperature, uint8_t minFanPower, uint8_t maxFanPower)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FBSCAB_PORTAL;
  packet.length = 8;
  packet.data[0] = SCMD_FBSCAB_AUTO_FAN_CONT;
  packet.data[1] = fbscabIndex;
  packet.data[2] = (fanIndex > 3) ? 3 : fanIndex;
  packet.data[3] = ((uint8_t)afcEnabled << 0) | ((uint8_t)fanOffUnderMinPower << 1) |
    (((afcResponsiveness > 15) ? 15 : afcResponsiveness) << 4);
  packet.data[4] = (dowTempIndex > 15) ? 15 : dowTempIndex;
  if (targetTemperature < -40) targetTemperature = -40;
  if (targetTemperature > 127) targetTemperature = 127;
  packet.data[5] = (uint8_t)(targetTemperature + 128);
  packet.data[6] = (minFanPower > 99) ? 99 : minFanPower;
  packet.data[7] = (maxFanPower > 100) ? 100 : maxFanPower;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

//command 39 group
CFA039A0NV::state_e CFA039A0NV::cmdFileOpen(fileopenmode_e openMode, char *fileName)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t namelen = strlen(fileName);
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2 + namelen;
  packet.data[0] = SCMD_FILE_FILE;
  packet.data[1] = (uint8_t)openMode;
  memcpy(&packet.data[2], fileName, namelen);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFileClose(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD_FILE_FILE;
  packet.data[1] = FILEOPEN_CLOSE;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFileSeek(uint32_t seekLocation)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 5;
  packet.data[0] = SCMD_FILE_SEEK;
  packet.data[1] = seekLocation & 0xFF;
  packet.data[2] = (seekLocation >> 8) & 0xFF;
  packet.data[3] = (seekLocation >> 16) & 0xFF;
  packet.data[4] = (seekLocation >> 24) & 0xFF;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFileRead(uint8_t *data, uint8_t *length)
{
  //length specifies amount of data to read, and
  //also returns the amount of data that was read
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD_FILE_READ;
  packet.data[1] = *length;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    *length = packet.length;
    if (*length != 0)
      memcpy(data, packet.data, *length);
  }
  return ret;
}

CFA039A0NV::state_e CFA039A0NV::cmdFileWrite(uint8_t *data, uint8_t length)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FILE_PORTAL;
  if (length > 123) length = 123;
  packet.length = 1 + length;
  packet.data[0] = SCMD_FILE_WRITE;
  memcpy(&packet.data[1], data, length);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFileDelete(char *filename)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(filename);
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 1 + len;
  packet.data[0] = SCMD_FILE_DELETE;
  memcpy(&packet.data[1], filename, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFileCopy(char *sourceFilename, char *destFilename)
{
  state_e ret;
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(sourceFilename);
  //set source file name
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2 + len;
  packet.data[0] = SCMD_FILE_COPY;
  packet.data[1] = 0;
  memcpy(&packet.data[2], sourceFilename, len);
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret != CFA_OK)
    //no go
    return ret;
  //set destination file name
  len = strlen(destFilename);
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2 + len;
  packet.data[0] = SCMD_FILE_COPY;
  packet.data[1] = 1;
  memcpy(&packet.data[2], destFilename, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdFilesystemFormat(filesystem_e filesystem)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD_FILE_FORMATFS;
  if (filesystem > 1) return CFA_CMDERROR;
  packet.data[1] = (uint8_t)filesystem;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdReadStorageSize(filesystem_e storage, uint32_t *totalCapacity, uint32_t *availableSpace)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD2_FILE_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD_FILE_SIZESTATS;
  packet.data[1] = (uint8_t)storage;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.data[1] != (uint8_t)storage)
      return CFA_CMDERROR;
    if (totalCapacity != NULL)
      *totalCapacity =  (uint32_t)packet.data[2] | ((uint32_t)packet.data[3] << 8) | ((uint32_t)packet.data[4] << 16) | ((uint32_t)packet.data[5] << 24);
    if (availableSpace != NULL)
      *availableSpace =  (uint32_t)packet.data[6] | ((uint32_t)packet.data[7] << 8) | ((uint32_t)packet.data[8] << 16) | ((uint32_t)packet.data[9] << 24);        
  }
  return ret;
}

//command 40 group
CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsOptions(bool manualFrameUpdate)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_GFX_OPTIONS;
  packet.data[1] = (uint8_t)manualFrameUpdate;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdGraphicsManualFrameUpdate(void)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 1;
  packet.data[0] = SCMD3_GFX_BUFFERFLUSH;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsBackgroundColor(uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_BACKGROUNDCOLOR;
  SETPKTDATA_UINT16(1, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdRemoveGraphicsObject(uint8_t objID)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_GFX_OBJ_REMOVE;
  packet.data[1] = objID;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

#warning CHECK SIGNED LOCATIONS WORK!

CFA039A0NV::state_e CFA039A0NV::cmdMoveGraphicsObject(uint8_t objID, int16_t X, int16_t Y)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 6;
  packet.data[0] = SCMD3_GFX_OBJ_LOCATION;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, X);
  SETPKTDATA_UINT16(4, Y);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsObjectTouchReporting(uint8_t objID, touchreport_e touchOption)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_OBJ_TOUCHREPORTING;
  packet.data[1] = objID;
  packet.data[2] = (uint8_t)touchOption;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsObjectZIndex(uint8_t objID, uint8_t zIndex)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_OBJ_ZINDEX;
  packet.data[1] = objID;
  packet.data[2] = zIndex;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsObjectFontSlot(uint8_t objID, uint8_t fontSlot)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_OBJ_FONTSLOT;
  packet.data[1] = objID;
  packet.data[2] = fontSlot;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsObjectFontSize(uint8_t objID, uint8_t fontSize)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_OBJ_FONTSIZE;
  packet.data[1] = objID;
  packet.data[2] = fontSize;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsObjectCornerRadius(uint8_t objID, uint8_t cornerRadius)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_OBJ_CORNERRADIUS;
  packet.data[1] = objID;
  packet.data[2] = cornerRadius;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsObjectOpacity(uint8_t objID, uint8_t opacityValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_GFX_OBJ_OPACITY;
  packet.data[1] = objID;
  packet.data[2] = opacityValue;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsFillAColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_FILLACOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsBorderAColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_BORDERACOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsControlAColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_CONTROLACOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsTextAColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_TEXTACOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsFillBColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_FILLBCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsBorderBColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_BORDERBCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsControlBColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_CONTROLBCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsTextBColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_TEXTBCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsFillCColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_FILLCCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsBorderCColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_BORDERCCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsControlCColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_CONTROLCCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdSetGraphicsTextCColor(uint8_t objID, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_GFX_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_GFX_OBJ_TEXTCCOLOR;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

//command 41 group

CFA039A0NV::state_e CFA039A0NV::cmdVideoFileLoad(uint8_t objID, char *filename, int16_t X, int16_t Y,
  bool startPlaying,
  uint8_t zIndex, touchreport_e touchReportingMode, uint16_t videoStartFrame, uint16_t videoEndFrame,
  uint8_t playLoops)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t slen = strlen(filename);
  packet.command = PCMD3_VIDEO_PORTAL;
  packet.length = 14 + slen;
  packet.data[0] = SCMD3_VIDEO_LOAD;
  packet.data[1] = objID;
  packet.data[2] = (uint8_t)startPlaying;
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  packet.data[7] = zIndex;
  packet.data[8] = (uint8_t)touchReportingMode;
  SETPKTDATA_UINT16(9, videoStartFrame);
  SETPKTDATA_UINT16(11, videoEndFrame);
  packet.data[13] = playLoops;
  memcpy(&packet.data[14], filename, slen);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdVideoPlay(uint8_t objID)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_VIDEO_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_VIDEO_CONTROL;
  packet.data[1] = objID;
  packet.data[2] = 1;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdVideoPauseToggle(uint8_t objID)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_VIDEO_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_VIDEO_CONTROL;
  packet.data[1] = objID;
  packet.data[2] = 2;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdVideoControl(uint8_t objID, uint16_t videoStartFrame, uint16_t videoEndFrame, uint8_t playLoops)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_VIDEO_PORTAL;
  packet.length = 8;
  packet.data[0] = SCMD3_VIDEO_LOAD;
  packet.data[1] = objID;
  packet.data[2] = 0;
  SETPKTDATA_UINT16(3, videoStartFrame);
  SETPKTDATA_UINT16(5, videoEndFrame);
  packet.data[7] = playLoops;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

//command 42 group

CFA039A0NV::state_e CFA039A0NV::cmdImageFileLoad(uint8_t objID, const char *filename,
  int16_t X, int16_t Y,
  bool disabledState, bool isButton, bool toggleButton, bool toggleStartDown, bool enableStateChangeReportPackets, 
  uint8_t zIndex, touchreport_e touchReportingMode)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t slen = strlen(filename);
  packet.command = PCMD3_IMAGE_PORTAL;
  packet.length = 9 + slen;
  packet.data[0] = SCMD3_IMAGE_LOAD;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0) | ((uint8_t)isButton << 1) | ((uint8_t)toggleButton << 2) |
    ((uint8_t)toggleStartDown << 3) | ((uint8_t)enableStateChangeReportPackets << 4);
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  packet.data[7] = zIndex;
  packet.data[8] = (uint8_t)touchReportingMode;
  memcpy(&packet.data[9], filename, slen);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdImageChangeOptions(uint8_t objID,
  bool disabledState, bool isButton, bool toggleButton, bool toggleStartDown, bool enableStateChangeReportPackets)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_IMAGE_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_IMAGE_LOAD;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0) | ((uint8_t)isButton << 1) | ((uint8_t)toggleButton << 2) |
    ((uint8_t)toggleStartDown << 3) | ((uint8_t)enableStateChangeReportPackets << 4);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdImageFileLoadDownState(uint8_t objID, const char *filename)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t slen = strlen(filename);
  packet.command = PCMD3_IMAGE_PORTAL;
  packet.length = 2 + slen;
  packet.data[0] = SCMD3_IMAGE_LOADDOWN;
  packet.data[1] = objID;
  memcpy(&packet.data[2], filename, slen);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdImageFileLoadDisabledState(uint8_t objID, const char *filename)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t slen = strlen(filename);
  packet.command = PCMD3_IMAGE_PORTAL;
  packet.length = 2 + slen;
  packet.data[0] = SCMD3_IMAGE_LOADDISABLED;
  packet.data[1] = objID;
  memcpy(&packet.data[2], filename, slen);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdImageLoadFromHost(void)
{
  //not implemented in this library (yet)
  return CFA_ERROR;
}

CFA039A0NV::state_e CFA039A0NV::cmdImageSetButtonState(uint8_t objID, bool setDownState)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_IMAGE_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_IMAGE_BUTTONSTATE;
  packet.data[1] = objID;
  packet.data[2] = (uint8_t)setDownState;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdImageReadButtonState(uint8_t objID, bool *currentState, bool *pressSinceLastPoll, bool *releasedSinceLastPoll)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD3_IMAGE_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_IMAGE_BUTTONSTATE;
  packet.data[1] = objID;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 3)
      return CFA_CMDERROR;
    if (currentState != NULL) *currentState = packet.data[2] & 0b001;
    if (pressSinceLastPoll != NULL) *pressSinceLastPoll = packet.data[2] & 0b010;
    if (releasedSinceLastPoll != NULL) *releasedSinceLastPoll = packet.data[2] & 0b100;
  }
  return ret;
}

//command 43 group

CFA039A0NV::state_e CFA039A0NV::cmdTTFLoadFont(uint8_t fontSlot, const char *filename, bool deferredLoading)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t slen = strlen(filename);
  packet.command = PCMD3_TTFONT_PORTAL;
  packet.length = 3 + slen;
  packet.data[0] = SCMD3_TTFONT_LOAD;
  packet.data[1] = fontSlot;
  packet.data[2] = (uint8_t)deferredLoading << 1;
  memcpy(&packet.data[3], filename, slen);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdTTFUnloadFont(uint8_t fontSlot)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_TTFONT_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_TTFONT_UNLOAD;
  packet.data[1] = fontSlot;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);  
}

CFA039A0NV::state_e CFA039A0NV::cmdTTFGetTextDimensions(uint8_t objID, const char *text,
  uint16_t *renderedTextWidth, uint16_t *renderedTextHeight)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  uint8_t len = strlen(text);
  packet.command = PCMD3_TTFONT_PORTAL;
  packet.length = 9 + len;
  packet.data[0] = SCMD3_TTFONT_NEWTEXT;
  packet.data[1] = objID;
  packet.data[2] = 0;
  SETPKTDATA_UINT16(3, 0);
  SETPKTDATA_UINT16(5, 0);
  packet.data[7] = 1;
  packet.data[8] = 0;
  memcpy(&packet.data[9], text, len);
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 5)
      return CFA_CMDERROR;
    if (renderedTextWidth != NULL) *renderedTextWidth = packet.data[1] | ((uint16_t)packet.data[2] << 8);
    if (renderedTextHeight != NULL) *renderedTextHeight = packet.data[3] | ((uint16_t)packet.data[4] << 8);
  }
  return ret;
}

CFA039A0NV::state_e CFA039A0NV::cmdTTFNewText(uint8_t objID, const char *text,
  int16_t X, int16_t Y,
  justify_e justifyText, uint8_t zIndex, touchreport_e touchReporting,
  uint16_t *renderedTextWidth, uint16_t *renderedTextHeight)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  uint8_t len = strlen(text);
  packet.command = PCMD3_TTFONT_PORTAL;
  packet.length = 9 + len;
  packet.data[0] = SCMD3_TTFONT_NEWTEXT;
  packet.data[1] = objID;
  if (justifyText == JUSTIFY_CENTER)
    packet.data[2] = (1<<3);
  else if (justifyText == JUSTIFY_RIGHT)
    packet.data[2] = (1<<4);
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  packet.data[7] = zIndex;
  packet.data[8] = (uint8_t)touchReporting;
  memcpy(&packet.data[9], text, len);
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 5)
      return CFA_CMDERROR;
    if (renderedTextWidth != NULL) *renderedTextWidth = packet.data[1] | ((uint16_t)packet.data[2] << 8);
    if (renderedTextHeight != NULL) *renderedTextHeight = packet.data[3] | ((uint16_t)packet.data[4] << 8);
  }
  return ret;
}

#warning add function defaults

CFA039A0NV::state_e CFA039A0NV::cmdTTFTextChangeOptions(uint8_t objID, const char *text,
  justify_e justifyText,
  uint16_t *renderedTextWidth, uint16_t *renderedTextHeight)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  uint8_t len = strlen(text);
  packet.command = PCMD3_TTFONT_PORTAL;
  packet.length = 3 + len;
  packet.data[0] = SCMD3_TTFONT_SETTEXT;
  packet.data[1] = objID;
  if (justifyText == JUSTIFY_CENTER)
    packet.data[2] = (1<<3);
  else if (justifyText == JUSTIFY_RIGHT)
    packet.data[2] = (1<<4);
  memcpy(&packet.data[3], text, len);
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 5)
      return CFA_CMDERROR;
    if (renderedTextWidth != NULL) *renderedTextWidth = packet.data[1] | ((uint16_t)packet.data[2] << 8);
    if (renderedTextHeight != NULL) *renderedTextHeight = packet.data[3] | ((uint16_t)packet.data[4] << 8);
  }
  return ret;
}

//command 44 group

CFA039A0NV::state_e CFA039A0NV::cmdSketchNewSurface(uint8_t objID,
  int16_t X, int16_t Y,
  uint16_t width, uint16_t height,
  bool backgroundFill, uint8_t zIndex, touchreport_e touchReporting)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_SKETCH_PORTAL;
  packet.length = 13;
  packet.data[0] = SCMD3_SKETCH_NEW;
  packet.data[1] = objID;
  packet.data[2] = (uint8_t)backgroundFill;
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  SETPKTDATA_UINT16(7, width);
  SETPKTDATA_UINT16(9, height);
  packet.data[11] = zIndex;
  packet.data[12] = (uint8_t)touchReporting;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdSketchDrawLine(uint8_t objID,
  uint16_t xStart, uint16_t yStart, uint16_t xEnd, uint16_t yEnd,
  uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_SKETCH_PORTAL;
  packet.length = 13;
  packet.data[0] = SCMD3_SKETCH_LINE;
  packet.data[1] = objID;
  packet.data[2] = 0;
  SETPKTDATA_UINT16(3, xStart);
  SETPKTDATA_UINT16(5, yStart);
  SETPKTDATA_UINT16(7, xEnd);
  SETPKTDATA_UINT16(9, yEnd);
  SETPKTDATA_UINT16(11, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdSketchDrawRectangle(uint8_t objID,
  uint16_t xStart, uint16_t yStart, uint16_t xEnd, uint16_t yEnd,
  uint8_t cornerRadius,
  bool fillWithColor, uint16_t fillRGB565, bool drawBorder, uint16_t borderRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_SKETCH_PORTAL;
  packet.length = 16;
  packet.data[0] = SCMD3_SKETCH_RECT;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)fillWithColor << 0) | ((uint8_t)drawBorder << 1);
  SETPKTDATA_UINT16(3, xStart);
  SETPKTDATA_UINT16(5, yStart);
  SETPKTDATA_UINT16(7, xEnd);
  SETPKTDATA_UINT16(9, yEnd);
  packet.data[11] = cornerRadius;
  SETPKTDATA_UINT16(12, fillRGB565);
  SETPKTDATA_UINT16(14, borderRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

  CFA039A0NV::state_e CFA039A0NV::cmdSketchDrawCircle(uint8_t objID,
  uint16_t centerX, uint16_t centerY, uint16_t radius,
  bool fillWithColor, uint16_t fillRGB565, bool drawBorder, uint16_t borderRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_SKETCH_PORTAL;
  packet.length = 13;
  packet.data[0] = SCMD3_SKETCH_CIRCLE;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)fillWithColor << 0) | ((uint8_t)drawBorder << 1);
  SETPKTDATA_UINT16(3, centerX);
  SETPKTDATA_UINT16(5, centerY);
  SETPKTDATA_UINT16(7, radius);
  SETPKTDATA_UINT16(9, fillRGB565);
  SETPKTDATA_UINT16(11, borderRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdSketchDrawPixel(uint8_t objID, uint16_t X, uint16_t Y, uint16_t colorRGB565)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_SKETCH_PORTAL;
  packet.length = 13;
  packet.data[0] = SCMD3_SKETCH_PIXEL;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, X);
  SETPKTDATA_UINT16(4, Y);
  SETPKTDATA_UINT16(6, colorRGB565);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

//command 47 group

CFA039A0NV::state_e CFA039A0NV::cmdButtonNew(uint8_t objID, const char *buttonLabel,
  int16_t X, int16_t Y, uint16_t width, uint16_t height,
  bool disabledState, bool isToggleButton, bool toggleStartDown, bool enableStateChangeReportPackets, 
  uint8_t zIndex, touchreport_e touchReporting)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(buttonLabel);
  packet.command = PCMD3_BUTTON_PORTAL;
  packet.length = 13 + len;
  packet.data[0] = SCMD3_BUTTON_NEW;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0) | ((uint8_t)isToggleButton << 1)
    | ((uint8_t)toggleStartDown << 2) | ((uint8_t)enableStateChangeReportPackets << 4);
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  SETPKTDATA_UINT16(7, width);
  SETPKTDATA_UINT16(9, height);
  packet.data[11] = zIndex;
  packet.data[12] = (uint8_t)touchReporting;
  memcpy(&packet.data[13], buttonLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdButtonSetUpStateLabel(uint8_t objID, const char *buttonLabel)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(buttonLabel);
  packet.command = PCMD3_BUTTON_PORTAL;
  packet.length = 2 + len;
  packet.data[0] = SCMD3_BUTTON_UP;
  packet.data[1] = objID;
  memcpy(&packet.data[2], buttonLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdButtonSetDownStateLabel(uint8_t objID, const char *buttonLabel)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(buttonLabel);
  packet.command = PCMD3_BUTTON_PORTAL;
  packet.length = 2 + len;
  packet.data[0] = SCMD3_BUTTON_DOWN;
  packet.data[1] = objID;
  memcpy(&packet.data[2], buttonLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdButtonSetDisabledStateLabel(uint8_t objID, const char *buttonLabel)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(buttonLabel);
  packet.command = PCMD3_BUTTON_PORTAL;
  packet.length = 2 + len;
  packet.data[0] = SCMD3_BUTTON_DISABLED;
  packet.data[1] = objID;
  memcpy(&packet.data[2], buttonLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdButtonSetState(uint8_t objID, bool downState)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_BUTTON_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_BUTTON_STATE;
  packet.data[1] = objID;
  packet.data[2] = (uint8_t)downState;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdImageReadState(uint8_t objID, bool *currentState, bool *pressSinceLastPoll, bool *releasedSinceLastPoll)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD3_BUTTON_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_BUTTON_STATE;
  packet.data[1] = objID;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if (packet.length != 3)
      return CFA_CMDERROR;
    if (currentState != NULL) *currentState = packet.data[2] & 0b001;
    if (pressSinceLastPoll != NULL) *pressSinceLastPoll = packet.data[2] & 0b010;
    if (releasedSinceLastPoll != NULL) *releasedSinceLastPoll = packet.data[2] & 0b100;
  }
  return ret;
}

//command 48 group

CFA039A0NV::state_e CFA039A0NV::cmdSliderNew(uint8_t objID, const char *textLabel,
    int16_t X, int16_t Y, uint16_t width, uint16_t height,
    int16_t minValue, int16_t maxValue, int16_t startValue,
    bool disabledState, bool showValue, justify_e valueLocation, bool noBackgroundFill, bool enableStateChangeReportPackets, 
    uint8_t zIndex, touchreport_e touchReporting)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(textLabel);
  packet.command = PCMD3_SLIDER_PORTAL;
  packet.length = 19 + len;
  packet.data[0] = SCMD3_SLIDER_NEW;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0);
  if (showValue)
  {
    if (valueLocation == JUSTIFY_LEFT)
      packet.data[2] |= (1<<1);
    if (valueLocation == JUSTIFY_RIGHT)
      packet.data[2] |= (1<<2);        
  }
  packet.data[2] |= ((uint8_t)noBackgroundFill << 3) | ((uint8_t)enableStateChangeReportPackets << 5);
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  SETPKTDATA_UINT16(7, width);
  SETPKTDATA_UINT16(9, height);
  packet.data[11] = zIndex;
  packet.data[12] = (uint8_t)touchReporting;
  SETPKTDATA_UINT16(13, minValue);
  SETPKTDATA_UINT16(15, maxValue);
  SETPKTDATA_UINT16(17, startValue);  
  memcpy(&packet.data[19], textLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdSliderSetValue(uint8_t objID, int16_t newValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_SLIDER_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_SLIDER_VALUE;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, newValue);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdSliderReadValue(uint8_t objID, int16_t *currentValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD3_SLIDER_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_SLIDER_VALUE;
  packet.data[1] = objID;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if ((packet.length != 4) || (packet.data[1] != objID))
      return CFA_CMDERROR;
    if (currentValue != NULL) *currentValue = (int16_t)((uint16_t)packet.data[2] | ((uint16_t)packet.data[3] << 8));
  }
  return ret;
}

//command 49 group

CFA039A0NV::state_e CFA039A0NV::cmdNumberEditNew(uint8_t objID, const char *textLabel,
    int16_t X, int16_t Y, uint16_t width, uint16_t height,
    int16_t minValue, int16_t maxValue, int16_t startValue,
    bool disabledState, justify_e labelPosition, bool noBackgroundFill, bool isPercentage, bool enableStateChangeReportPackets, 
    uint8_t zIndex, touchreport_e touchReporting)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(textLabel);
  packet.command = PCMD3_NUMEDIT_PORTAL;
  packet.length = 19 + len;
  packet.data[0] = SCMD3_NUMEDIT_NEW;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0) | ((uint8_t)noBackgroundFill << 1);
  if (labelPosition == JUSTIFY_RIGHT)
    packet.data[2] |= (1<<2);
  packet.data[2] |= ((uint8_t)isPercentage << 3) | ((uint8_t)enableStateChangeReportPackets << 5);
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  SETPKTDATA_UINT16(7, width);
  SETPKTDATA_UINT16(9, height);
  packet.data[11] = zIndex;
  packet.data[12] = (uint8_t)touchReporting;
  SETPKTDATA_UINT16(13, minValue);
  SETPKTDATA_UINT16(15, maxValue);
  SETPKTDATA_UINT16(17, startValue);  
  memcpy(&packet.data[19], textLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdNumberEditSetValue(uint8_t objID, int16_t newValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_NUMEDIT_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_NUMEDIT_VALUE;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, newValue);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdNumberEditReadValue(uint8_t objID, int16_t *currentValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD3_NUMEDIT_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_NUMEDIT_VALUE;
  packet.data[1] = objID;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if ((packet.length != 4) || (packet.data[1] != objID))
      return CFA_CMDERROR;
    if (currentValue != NULL) *currentValue = (int16_t)((uint16_t)packet.data[2] | ((uint16_t)packet.data[3] << 8));
  }
  return ret;
}

//command 50 group
CFA039A0NV::state_e CFA039A0NV::cmdCheckboxNew(uint8_t objID, const char *textLabel,
    int16_t X, int16_t Y, uint16_t width, uint16_t height,
    checkbox_e startState, 
    bool tickStateEnabled, bool crossStateEnabled, bool disabledState, justify_e labelPosition,
    bool noBackgroundFill, bool enableStateChangeReportPackets, 
    uint8_t zIndex, touchreport_e touchReporting)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(textLabel);
  packet.command = PCMD3_CHECKBOX_PORTAL;
  packet.length = 14 + len;
  packet.data[0] = SCMD3_CHECKBOX_NEW;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0) | ((uint8_t)noBackgroundFill << 1);
  if (labelPosition == JUSTIFY_RIGHT)
    packet.data[2] |= (1<<2);
  packet.data[2] |= ((uint8_t)tickStateEnabled << 3) | ((uint8_t)crossStateEnabled << 4) | 
    ((uint8_t)enableStateChangeReportPackets << 6);
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  SETPKTDATA_UINT16(7, width);
  SETPKTDATA_UINT16(9, height);
  packet.data[11] = zIndex;
  packet.data[12] = (uint8_t)touchReporting;
  packet.data[13] = (uint8_t)startState;
  memcpy(&packet.data[14], textLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdCheckboxSetValue(uint8_t objID, checkbox_e newState)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_CHECKBOX_PORTAL;
  packet.length = 3;
  packet.data[0] = SCMD3_CHECKBOX_VALUE;
  packet.data[1] = objID;
  packet.data[2] = (uint8_t)newState;
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdCheckboxReadValue(uint8_t objID, checkbox_e *currentState)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD3_CHECKBOX_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_CHECKBOX_VALUE;
  packet.data[1] = objID;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if ((packet.length != 3) || (packet.data[1] != objID))
      return CFA_CMDERROR;
    if (currentState != NULL) *currentState = (checkbox_e)packet.data[2];
  }
  return ret;
}

//command 51 group

CFA039A0NV::state_e CFA039A0NV::cmdProgressBarNew(uint8_t objID, const char *textLabel,
    int16_t X, int16_t Y, uint16_t width, uint16_t height,
    int16_t minValue, int16_t maxValue, int16_t startValue,
    bool disabledState, justify_e labelPosition, bool noBackgroundFill, bool isPercentage,
    uint8_t zIndex, touchreport_e touchReporting)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  uint8_t len = strlen(textLabel);
  packet.command = PCMD3_PROGRESSBAR_PORTAL;
  packet.length = 19 + len;
  packet.data[0] = SCMD3_PROGRESSBAR_NEW;
  packet.data[1] = objID;
  packet.data[2] = ((uint8_t)disabledState << 0) | ((uint8_t)noBackgroundFill << 1) | ((uint8_t)isPercentage << 2);
  if (labelPosition == JUSTIFY_LEFT)
    packet.data[2] |= (1<<3);
  if (labelPosition == JUSTIFY_RIGHT)
    packet.data[2] |= (1<<4);      
  SETPKTDATA_UINT16(3, X);
  SETPKTDATA_UINT16(5, Y);
  SETPKTDATA_UINT16(7, width);
  SETPKTDATA_UINT16(9, height);
  packet.data[11] = zIndex;
  packet.data[12] = (uint8_t)touchReporting;
  SETPKTDATA_UINT16(13, minValue);
  SETPKTDATA_UINT16(15, maxValue);
  SETPKTDATA_UINT16(17, startValue);  
  memcpy(&packet.data[19], textLabel, len);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdProgressBarSetValue(uint8_t objID, int16_t newValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  packet.command = PCMD3_PROGRESSBAR_PORTAL;
  packet.length = 4;
  packet.data[0] = SCMD3_PROGRESSBAR_VALUE;
  packet.data[1] = objID;
  SETPKTDATA_UINT16(2, newValue);
  return sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
}

CFA039A0NV::state_e CFA039A0NV::cmdProgressBarReadValue(uint8_t objID, int16_t *currentValue)
{
#ifndef CFA039A_USE_STATIC_PACKET
  packet_t packet;
#endif
  state_e ret;
  packet.command = PCMD3_PROGRESSBAR_PORTAL;
  packet.length = 2;
  packet.data[0] = SCMD3_PROGRESSBAR_VALUE;
  packet.data[1] = objID;
  ret = sendPacketAndGetReply(&packet, _cmd_reply_timeout, true);
  if (ret == CFA_OK)
  {
    if ((packet.length != 4) || (packet.data[1] != objID))
      return CFA_CMDERROR;
    if (currentValue != NULL) *currentValue = (int16_t)((uint16_t)packet.data[2] | ((uint16_t)packet.data[3] << 8));
  }
  return ret;
}
