#ifndef NOTE_SERIAL_HPP
#define NOTE_SERIAL_HPP

#include <stddef.h>
#include <stdint.h>

class NoteSerial
{
public:

    virtual ~NoteSerial(void) {}

    /**************************************************************************/
    /*!
        @brief  Determines if the Notecard Serial port has data available.
        @return The number of bytes available to read.
    */
    /**************************************************************************/
    virtual size_t available(void) = 0;

    /**************************************************************************/
    /*!
        @brief  Read a byte from the Notecard Serial port.
        @return A single character byte.
    */
    /**************************************************************************/
    virtual char receive(void) = 0;

    /**************************************************************************/
    /*!
        @brief  Resets the serial port.
        @return `true` if the Serial port is available.
    */
    /**************************************************************************/
    virtual bool reset(void) = 0;

    /**************************************************************************/
    /*!
        @brief  Writes a buffer to the Notecard Serial port.
        @param    buffer
                  The bytes to write.
        @param    size
                  The number of bytes to write.
        @param    flush
                  Use `true` to flush to Serial.
        @return The number of bytes transmitted.
    */
    /**************************************************************************/
    virtual size_t transmit(uint8_t * buffer, size_t size, bool flush) = 0;
};

/******************************************************************************/
/*!
    @brief Creates a NoteSerial instance

    Helper function to abstract, create and maintain a single instance
    of the NoteSerial interface implementation, as required by the underlying
    `note-c` library.

    @param[in] serial_parameters
               Pointer to the parameters required to instantiate
               the platform specific UART implementation.
*/
/******************************************************************************/
template<typename T> NoteSerial * make_note_serial (T & serial_parameters);
NoteSerial * make_note_serial (nullptr_t);

#endif // NOTE_SERIAL_HPP
