/*
  GBALib_UltrasonicSensor.cpp - HC-SR04 Ultrasonic Sensor library.
  Copyright (c) 2024 Graziano Blasilli.
  
  This library provides a simple interface to read the HC-SR04 Ultrasonic Sensor inputs.
*/

#include "GBALib_UltrasonicSensor.h"

// Constructor to initialize the UltrasonicSensor object and set pin modes
UltrasonicSensor::UltrasonicSensor(int triggerPin, int echoPin) {
  this->triggerPin = triggerPin;
  this->echoPin = echoPin;
  
  pinMode(triggerPin, OUTPUT); // Set the trigger pin as an output
  pinMode(echoPin, INPUT);     // Set the echo pin as an input
}

/**
 * Reads the distance in centimeters using the ultrasonic sensor.
 * 
 * @return The calculated distance in centimeters.
 */
int UltrasonicSensor::readDistance() {
  // Ensure the trigger pin is initially low
  digitalWrite(triggerPin, LOW);
  delayMicroseconds(2); // Wait for 2 microseconds for signal stability

  // Send a 10-microsecond pulse to the trigger pin to generate the ultrasonic wave
  digitalWrite(triggerPin, HIGH);
  delayMicroseconds(10); // 10-microsecond pulse
  digitalWrite(triggerPin, LOW);

  // Measure the duration of the echo signal (time the signal stays high)
  long duration = pulseIn(echoPin, HIGH);

  // Calculate the distance in centimeters
  // Formula: distance = (duration * speed of sound) / 2
  // Speed of sound in air = 0.034 cm/microsecond
  int distanceCentimeter = duration * 0.034 / 2;

  // Return the calculated distance
  return distanceCentimeter;
}

/**
 * Executes `n` subsequent readings and returns the average distance.
 * This helps to reduce noise and provides a more stable measurement.
 * 
 * @param n The number of readings to average.
 * @return The average distance in centimeters.
 */
int UltrasonicSensor::readDistance(int n) {
  if (n <= 0) {
    return 0;  // Return 0 if an invalid number of readings is provided
  }

  long totalDistance = 0;  // Variable to store the sum of distances

  // Perform `n` readings and accumulate the total distance
  for (int i = 0; i < n; i++) {
    totalDistance += readDistance();  // Call the single reading method
    delay(50);  // Small delay between readings to ensure sensor stability
  }

  // Calculate and return the average distance
  return totalDistance / n;
}
