#ifndef UNIVERSAL_DEBUG_H
#define UNIVERSAL_DEBUG_H

#include <HardwareSerial.h>
#include <stdarg.h>
#include <Arduino.h> // Include Arduino core for Print class definition and String support

// --- Configuration Defaults (Can be overridden in the .ino file BEFORE including this header) ---
// Define the default pin numbers for SoftwareSerial (standard Arduino boards)
#ifndef BT_RX_PIN
#define BT_RX_PIN 10 
#endif
#ifndef BT_TX_PIN
#define BT_TX_PIN 11
#endif

// Control Bluetooth output here: Define this in your .ino file to enable/disable easily
// #define DEBUG_BLUETOOTH_ENABLED 1 

// --- PLATFORM DETECTION LOGIC ---
// Check if user forced ESP32, or if compiler natively detects it
#if defined(FORCE_ESP32) || defined(ESP32)
  #define IS_ESP32_PLATFORM 1 // Use 1 for preprocessor #if checks
#else
  #define IS_ESP32_PLATFORM 0
#endif
// --------------------------------

// 1. --- Conditional Includes and Instance Definitions ---
#if IS_ESP32_PLATFORM
  #include <BluetoothSerial.h>
  BluetoothSerial SerialBT; 
  #define HAS_BUILTIN_BT 1
#else
  #include <SoftwareSerial.h>
  SoftwareSerial BTserial(BT_RX_PIN, BT_TX_PIN); 
  #define HAS_BUILTIN_BT 0
#endif

// A buffer large enough for most debug messages
#define DEBUG_BUFFER_SIZE 128

class UniversalDebug : public Print { // Inherit publicly from Print
public:
    UniversalDebug(HardwareSerial& hwSerial) : _hwSerial(hwSerial) {}

    void begin(long baudRate, const char* btName = "Arduino_Debug") {
        _hwSerial.begin(baudRate);

        #if DEBUG_BLUETOOTH_ENABLED
            #if HAS_BUILTIN_BT
                SerialBT.begin(btName);
            #else
                BTserial.begin(baudRate);
            #endif
        #endif
    }

    /**
     * @brief Required override for the Print base class. Handles all standard Debug.print() calls.
     */
    size_t write(uint8_t character) override {
        _hwSerial.write(character);

        #if DEBUG_BLUETOOTH_ENABLED
            // Use #if here for COMPILE-TIME exclusion of the unused variable
            #if HAS_BUILTIN_BT
                SerialBT.write(character);
            #else
                BTserial.write(character);
            #endif
        #endif
        return 1; // Return 1 byte written
    }

    /**
     * @brief Function for printf-style formatting with a newline at the end.
     */
    void printlnFormat(const char* format, ...) {
        char buffer[DEBUG_BUFFER_SIZE];
        va_list args;
        va_start(args, format);
        vsnprintf(buffer, sizeof(buffer), format, args);
        va_end(args);

        // Use the inherited print function which uses write() internally
        this->print(buffer);
        this->println(); // Add standard newline using inherited function
    }

    /**
     * @brief Function for printf-style formatting without a newline. Alias for printf().
     */
    void printFormat(const char* format, ...) {
        char buffer[DEBUG_BUFFER_SIZE];
        va_list args;
        va_start(args, format);
        vsnprintf(buffer, sizeof(buffer), format, args);
        va_end(args);
        
        this->print(buffer);
    }
    
    // Alias printf() to printFormat() for familiarity
    void printf(const char* format, ...) {
        // Calls the implementation above
        char buffer[DEBUG_BUFFER_SIZE];
        va_list args;
        va_start(args, format);
        vsnprintf(buffer, sizeof(buffer), format, args);
        va_end(args);
        
        this->print(buffer);
    }

    // Note: Standard Debug.print(int), Debug.print(float), Debug.println(String) 
    // functions are automatically inherited from the 'Print' base class.

private:
    HardwareSerial& _hwSerial;
};

#endif // UNIVERSAL_DEBUG_H
