/**
 * Basicmicro Library Example: GETSTREAMS (146) - MCP Only
 *
 * Demonstrates reading the communication parameters for all "Stream"
 * channels on MCP series controllers. This command returns the total number
 * of configured streams and their detailed settings.
 *
 * !!! IMPORTANT: This command is marked as "MCP only". It may not work on
 * Sabertooth or Kangaroo controllers. The meaning of the returned parameters
 * depends heavily on the 'streamType' and the specific MCP model. Consult your
 * controller's documentation for detailed parameter definitions. !!!
 *
 * This example uses HardwareSerial (Serial1) for communication with the controller
 * and HardwareSerial (Serial) for debugging output. Adjust serial ports and
 * controller address as needed for your setup.
 */

#include <Arduino.h>
#include <Basicmicro.h>

// Define the serial port for communication with the motor controller
// On boards like Mega, Due, Leonardo, etc., use Serial1, Serial2, Serial3.
//#define CONTROLLER_SERIAL   Serial1

// Optional: Use SoftwareSerial on AVR boards if HardwareSerial is not available
#include <SoftwareSerial.h>
#define RX_PIN 10 // Connect to controller's TX pin
#define TX_PIN 11 // Connect to controller's RX pin
SoftwareSerial controllerSerial_SW(RX_PIN, TX_PIN);
#define CONTROLLER_SERIAL   controllerSerial_SW // Use this define instead of Serial1

// Define the address of your motor controller
#define MOTOR_ADDRESS       128

// Define the library's internal read timeout in microseconds
#define LIBRARY_READ_TIMEOUT 10000

// Instantiate the Basicmicro library object
// If using SoftwareSerial, uncomment the #define above and use controllerSerial_SW
Basicmicro controller(&CONTROLLER_SERIAL, LIBRARY_READ_TIMEOUT);

// Buffer to store the stream configurations.
// Choose a size large enough for the maximum number of streams your controller has.
// The library header doesn't specify a max, but 2-4 is a common number of streams.
#define MAX_STREAMS 4
Basicmicro::StreamConfig streamConfigurations[MAX_STREAMS];


void setup() {
  // Initialize debug serial port
  Serial.begin(9600);
  while (!Serial && millis() < 5000); // Wait for Serial port to connect (useful on some boards)

  Serial.println("Basicmicro GETSTREAMS Example (MCP Only)");
  Serial.print("Connecting to controller on ");
  // Print the name of the serial port being used (if possible)
  #if defined(CONTROLLER_SERIAL) && !defined(RX_PIN) // Check if using HardwareSerial and not SoftwareSerial
    Serial.println("Hardware Serial");
  #elif defined(RX_PIN) // Check if SoftwareSerial pins are defined using RX_PIN
    Serial.println("Software Serial");
  #else
    Serial.println("Unknown Serial type");
  #endif


  // Initialize the communication serial port for the controller
  controller.begin(9600); // Ensure baud rate matches your controller
  delay(100); // Short delay to let the controller initialize after power-up or serial init

  Serial.println("Attempting to read stream configurations...");
  Serial.println("NOTE: This command is for MCP controllers.");
}

void loop() {
  // Variable to store the number of streams reported by the controller
  uint8_t stream_count = 0; // Initialize to 0

  // Attempt to read the stream configurations
  // This command is typically only supported by MCP series controllers.
  // The function returns true on success, false on failure.
  // The number of streams is stored in stream_count, and the configurations
  // are stored in the streamConfigurations array (up to MAX_STREAMS).
  bool success = controller.GetStreams(MOTOR_ADDRESS, stream_count, streamConfigurations, MAX_STREAMS);

  if (success) {
    Serial.println("GETSTREAMS command successful.");
    Serial.print("Reported Stream Count: "); Serial.println(stream_count);
    Serial.println("Current Stream Configurations:");

    // Iterate up to the reported count (or MAX_STREAMS if reported count is larger)
    uint8_t display_count = min(stream_count, MAX_STREAMS);
    for (uint8_t i = 0; i < display_count; i++) {
        Serial.print("  Stream Index "); Serial.print(i); Serial.println(":");
        Serial.print("    type: "); Serial.println(streamConfigurations[i].type);
        Serial.print("    baudrate: "); Serial.println(streamConfigurations[i].baudrate);
        Serial.print("    timeout: "); Serial.println(streamConfigurations[i].timeout);
    }
     if (stream_count > MAX_STREAMS) {
        Serial.print("(Only displaying first "); Serial.print(MAX_STREAMS); Serial.println(" configurations due to buffer size)");
    }
    Serial.println("NOTE: Interpretation of these values is specific to your controller model.");

  } else {
    Serial.println("GETSTREAMS command failed.");
    Serial.println("Check wiring, power, address, baud rate, and controller model.");
    Serial.println("This command may not be supported by your controller.");
  }

  // Wait a few seconds before reading again
  delay(5000); // Reading settings infrequently is fine
}