#include "EEPROMHandler.h"

EEPROMHandler::EEPROMHandler(uint8_t deviceAddress, int baseAddr)
    : deviceAddress(deviceAddress), baseAddress(baseAddr) {}

void EEPROMHandler::saveInt(int address, int value) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.write(value >> 8);
    Wire.write(value & 0xFF);
    Wire.endTransmission();
    delay(5);
}

int EEPROMHandler::readInt(int address) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.endTransmission();

    Wire.requestFrom(deviceAddress, (uint8_t)2);
    int value = (Wire.read() << 8) | Wire.read();
    return value;
}

void EEPROMHandler::validateInt(int address, int minValue, int maxValue, int defaultValue) {
    int value = readInt(address);
    if (value < minValue || value > maxValue) {
        saveInt(address, defaultValue);
    }
}

void EEPROMHandler::validateInt(int address, int minValue, int maxValue) {
    int value = readInt(address);
    if (value < minValue || value > maxValue) {
        int defaultValue = (minValue + maxValue) / 2;
        saveInt(address, defaultValue);
    }
}

void EEPROMHandler::saveLong(int address, long value) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    for (int i = 3; i >= 0; i--) {
        Wire.write((value >> (i * 8)) & 0xFF);
    }
    Wire.endTransmission();
    delay(5);
}

long EEPROMHandler::readLong(int address) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.endTransmission();

    Wire.requestFrom(deviceAddress, (uint8_t)4);
    long value = 0;
    for (int i = 3; i >= 0; i--) {
        value |= ((long)Wire.read() << (i * 8));
    }
    return value;
}

void EEPROMHandler::validateLong(int address, long minValue, long maxValue, long defaultValue) {
    long value = readLong(address);
    if (value < minValue || value > maxValue) {
        saveLong(address, defaultValue);
    }
}

void EEPROMHandler::validateLong(int address, long minValue, long maxValue) {
    long value = readLong(address);
    if (value < minValue || value > maxValue) {
        long defaultValue = (minValue + maxValue) / 2;
        saveLong(address, defaultValue);
    }
}

void EEPROMHandler::saveFloat(int address, float value) {
    long intValue;
    memcpy(&intValue, &value, sizeof(float));
    saveLong(address, intValue);
}

float EEPROMHandler::readFloat(int address) {
    long intValue = readLong(address);
    float value;
    memcpy(&value, &intValue, sizeof(float));
    return value;
}

void EEPROMHandler::validateFloat(int address, float minValue, float maxValue, float defaultValue) {
    float value = readFloat(address);
    if (value < minValue || value > maxValue) {
        saveFloat(address, defaultValue);
    }
}

void EEPROMHandler::validateFloat(int address, float minValue, float maxValue) {
    float value = readFloat(address);
    if (value < minValue || value > maxValue) {
        float defaultValue = (minValue + maxValue) / 2.0f;
        saveFloat(address, defaultValue);
    }
}

void EEPROMHandler::saveDouble(int address, double value) {
    long intValue;
    memcpy(&intValue, &value, sizeof(double));
    saveLong(address, intValue);
}

double EEPROMHandler::readDouble(int address) {
    long intValue = readLong(address);
    double value;
    memcpy(&value, &intValue, sizeof(double));
    return value;
}

void EEPROMHandler::validateDouble(int address, double minValue, double maxValue, double defaultValue) {
    double value = readDouble(address);
    if (value < minValue || value > maxValue) {
        saveDouble(address, defaultValue);
    }
}

void EEPROMHandler::validateDouble(int address, double minValue, double maxValue) {
    double value = readDouble(address);
    if (value < minValue || value > maxValue) {
        double defaultValue = (minValue + maxValue) / 2.0;
        saveDouble(address, defaultValue);
    }
}

void EEPROMHandler::saveChar(int address, char value) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.write(value);
    Wire.endTransmission();
    delay(5);
}

char EEPROMHandler::readChar(int address) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.endTransmission();

    Wire.requestFrom(deviceAddress, (uint8_t)1);
    return Wire.read();
}

void EEPROMHandler::validateChar(int address, char defaultValue) {
    char value = readChar(address);
    if (value < 32 || value > 126) { // Printable ASCII range
        saveChar(address, defaultValue);
    }
}

void EEPROMHandler::validateChar(int address) {
    char value = readChar(address);
    if (value < 32 || value > 126) { // Printable ASCII range
        saveChar(address, 'A'); // Default to 'A'
    }
}

void EEPROMHandler::saveBool(int address, bool value) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.write(value);
    Wire.endTransmission();
    delay(5);
}

bool EEPROMHandler::readBool(int address) {
    Wire.beginTransmission(deviceAddress);
    Wire.write((baseAddress + address) >> 8);
    Wire.write((baseAddress + address) & 0xFF);
    Wire.endTransmission();

    Wire.requestFrom(deviceAddress, (uint8_t)1);
    return Wire.read();
}

void EEPROMHandler::validateBool(int address, bool defaultValue) {
    bool value = readBool(address);
    if (value != 0 && value != 1) {
        saveBool(address, defaultValue);
    }
}

void EEPROMHandler::validateBool(int address) {
    bool value = readBool(address);
    if (value != 0 && value != 1) {
        saveBool(address, false); // Default to false
    }
}

void EEPROMHandler::saveString(int address, const String &value) {
    int len = value.length();
    saveInt(address, len);
    for (int i = 0; i < len; i++) {
        saveChar(address + sizeof(int) + i, value[i]);
    }
}

String EEPROMHandler::readString(int address) {
    int len = readInt(address);
    String value = "";
    for (int i = 0; i < len; i++) {
        value += readChar(address + sizeof(int) + i);
    }
    return value;
}

void EEPROMHandler::validateString(int address, int maxLength, const String &defaultValue) {
    String value = readString(address);
    if (value.length() == 0 || value.length() > maxLength) {
        saveString(address, defaultValue);
    }
}

void EEPROMHandler::validateString(int address, int maxLength) {
    String value = readString(address);
    if (value.length() == 0 || value.length() > maxLength) {
        String defaultValue = "Default";
        saveString(address, defaultValue);
    }
}

void EEPROMHandler::saveBytes(int startAddress, const uint8_t* data, int length) {
    for (int i = 0; i < length; i++) {
        saveChar(startAddress + i, data[i]);
    }
}

void EEPROMHandler::readBytes(int startAddress, uint8_t* data, int length) {
    for (int i = 0; i < length; i++) {
        data[i] = readChar(startAddress + i);
    }
}

void EEPROMHandler::validateBytes(int address, const uint8_t* defaultData, int length) {
    uint8_t* data = new uint8_t[length];
    readBytes(address, data, length);
    bool isValid = true;
    for (int i = 0; i < length; i++) {
        if (data[i] != defaultData[i]) {
            isValid = false;
            break;
        }
    }
    if (!isValid) {
        saveBytes(address, defaultData, length);
    }
    delete[] data;
}

void EEPROMHandler::validateBytes(int address, int length) {
    uint8_t* data = new uint8_t[length];
    readBytes(address, data, length);
    bool isValid = true;
    for (int i = 0; i < length; i++) {
        if (data[i] < 0 || data[i] > 255) {
            isValid = false;
            break;
        }
    }
    if (!isValid) {
        uint8_t* defaultData = new uint8_t[length];
        for (int i = 0; i < length; i++) {
            defaultData[i] = 0;
        }
        saveBytes(address, defaultData, length);
        delete[] defaultData;
    }
    delete[] data;
}

void EEPROMHandler::saveIntArray(int startAddress, int values[], int size) {
    for (int i = 0; i < size; i++) {
        saveInt(startAddress + i * sizeof(int), values[i]);
    }
}

void EEPROMHandler::readIntArray(int startAddress, int values[], int size) {
    for (int i = 0; i < size; i++) {
        values[i] = readInt(startAddress + i * sizeof(int));
    }
}

void EEPROMHandler::validateIntArray(int startAddress, int values[], int size, int defaultValues[]) {
    readIntArray(startAddress, values, size);
    bool isValid = true;
    for (int i = 0; i < size; i++) {
        if (values[i] < 0 || values[i] > 100) { // Example range check
            isValid = false;
            break;
        }
    }
    if (!isValid) {
        saveIntArray(startAddress, defaultValues, size);
    }
}

void EEPROMHandler::validateIntArray(int startAddress, int values[], int size) {
    readIntArray(startAddress, values, size);
    bool isValid = true;
    for (int i = 0; i < size; i++) {
        if (values[i] < 0 || values[i] > 100) { // Example range check
            isValid = false;
            break;
        }
    }
    if (!isValid) {
        int* defaultValues = new int[size];
        for (int i = 0; i < size; i++) {
            defaultValues[i] = 50; // Example default value
        }
        saveIntArray(startAddress, defaultValues, size);
        delete[] defaultValues;
    }
}

void EEPROMHandler::clearEEPROM(int length) {
    for (int i = 0; i < length; i++) {
        saveChar(i, 0);
    }
}