#ifndef DWIN_UNI_HMI_H
#define DWIN_UNI_HMI_H

#include <Arduino.h>
#include "DwinDefines.h"

// Callback function type (Triggered when a button is pressed or data received)
typedef void (*DwinEventCallback)(uint16_t address, uint16_t data);

class Dwin {
  private:
    Stream *_serial;        // Serial port object
    uint8_t _rxBuffer[255]; // Receive buffer
    int _rxIndex;           // Buffer index
    
    // Page offset for multi-language support (e.g. TR=0, EN=10)
    uint16_t _pageOffset; 

    // Event Listener Variables
    DwinEventCallback _eventCallback;
    
    // Async parsing state machine
    enum ParseState { STATE_IDLE, STATE_HEADER1, STATE_HEADER2, STATE_LEN, STATE_CMD };
    ParseState _state;
    uint8_t _frameLen;
    uint8_t _frameCmd;

    // --- Helper (Private) Functions ---
    uint16_t makeWord(uint8_t hb, uint8_t lb);
    void sendCommand(uint8_t cmd, uint16_t address, uint8_t *data, uint8_t dataLen);
    void setPageDirect(uint16_t pageID); // Internal use for direct page switching

  public:
    // Constructor
    Dwin();

    // Initialization
    void begin(Stream *serialPort, long baudRate = 115200);

    // --- BASIC ACCESS (WRITE/READ) ---
    void writeVP(uint16_t address, uint16_t value); // Write 16-bit
    void writeVP(uint16_t address, uint32_t value); // Write 32-bit (Long)
    void writeVP(uint16_t address, float value);    // Write Float
    void readVP(uint16_t address, uint8_t wordLen); // Request Read (Response via listener)

    // --- SYSTEM & PAGE ---
    void setPage(uint16_t pageID); // Switches page (Adds language offset automatically)
    uint16_t getPage();            // Requests current page ID
    void resetHMI();               // Resets the display

    // --- MULTI-LANGUAGE SUPPORT ---
    // Method 1: Page Offset (e.g. Set 0 for TR, 10 for EN)
    void setLanguageOffset(uint16_t offset);
    
    // Method 2: Native ICL Switching (e.g. 32.ICL for TR, 33.ICL for EN)
    void setLanguageICL(uint8_t iclID);

    // --- UI APPEARANCE ---
    void setBrightness(uint8_t level); // 0-100
    void setBuzzer(uint16_t durationMs); // Beep duration in ms
    
    // Change Text Color (Requires SP Address)
    // textControlSP: The SP address assigned to the Text Display control
    void setTextColor(uint16_t textControlSP, uint16_t color);

    // --- LISTENER (MUST BE CALLED IN LOOP) ---
    void listen();
    void attachCallback(DwinEventCallback func);
};

#endif