/*
  Dwin.cpp - Universal DWIN DGUS II Library Implementation
  Created by Askin Keles, December 3, 2025.
  Released into the public domain.
*/

#include "DWIN_UNI_HMI.h"

// Constructor
Dwin::Dwin() {
    _state = STATE_IDLE;
    _rxIndex = 0;
    _eventCallback = NULL;
    _pageOffset = 0; // Default language offset is 0
}

// Initialization
void Dwin::begin(Stream *serialPort, long baudRate) {
    _serial = serialPort;
    // Note: In Arduino, .begin(baud) is usually called in setup() by the user.
    // We just take the stream pointer.
}

// --- HELPER FUNCTIONS ---

// Combine two bytes into a 16-bit word
uint16_t Dwin::makeWord(uint8_t hb, uint8_t lb) {
    return (hb << 8) | lb;
}

// Standard DWIN Packet Sender
// Structure: Header + Len + Cmd + Address + Data
void Dwin::sendCommand(uint8_t cmd, uint16_t address, uint8_t *data, uint8_t dataLen) {
    // Frame structure: [5A A5] [Len] [Cmd] [AddrH AddrL] [Data...]
    // Len = Cmd(1) + Addr(2) + DataLen
    uint8_t frameLen = 1 + 2 + dataLen;
    
    // Header
    _serial->write(DWIN_HEADER_1);
    _serial->write(DWIN_HEADER_2);
    
    // Length
    _serial->write(frameLen);
    
    // Command (0x82 Write or 0x83 Read)
    _serial->write(cmd);
    
    // Address (Big Endian - MSB First)
    _serial->write((uint8_t)(address >> 8));
    _serial->write((uint8_t)(address & 0xFF));
    
    // Data (If any)
    if (dataLen > 0 && data != NULL) {
        _serial->write(data, dataLen);
    }
}

// --- BASIC ACCESS ---

// Write 16-bit Integer
void Dwin::writeVP(uint16_t address, uint16_t value) {
    uint8_t data[2];
    data[0] = (uint8_t)(value >> 8);   // High Byte
    data[1] = (uint8_t)(value & 0xFF); // Low Byte
    
    sendCommand(CMD_WRITE_VP, address, data, 2);
}

// Write 32-bit Integer (Long)
void Dwin::writeVP(uint16_t address, uint32_t value) {
    uint8_t data[4];
    data[0] = (uint8_t)((value >> 24) & 0xFF);
    data[1] = (uint8_t)((value >> 16) & 0xFF);
    data[2] = (uint8_t)((value >> 8) & 0xFF);
    data[3] = (uint8_t)(value & 0xFF);
    
    sendCommand(CMD_WRITE_VP, address, data, 4);
}

// Write Float
void Dwin::writeVP(uint16_t address, float value) {
    union { float f; uint8_t b[4]; } u;
    u.f = value;
    // DWIN expects Big-Endian, ESP32 is Little-Endian. We must reverse bytes.
    uint8_t data[4] = { u.b[3], u.b[2], u.b[1], u.b[0] };
    sendCommand(CMD_WRITE_VP, address, data, 4);
}

// Request Data Read
void Dwin::readVP(uint16_t address, uint8_t wordLen) {
    uint8_t len = wordLen; // Number of words to read
    // For Read command, data part contains the length to read
    sendCommand(CMD_READ_VP, address, &len, 1);
}

// --- SYSTEM CONTROLS ---

// Change Page (Relative to Language Offset)
void Dwin::setPage(uint16_t pageID) {
    uint16_t targetPage = pageID + _pageOffset;
    setPageDirect(targetPage);
}

// Change Page (Direct Address)
void Dwin::setPageDirect(uint16_t pageID) {
    uint8_t data[4] = {0x5A, 0x01, (uint8_t)(pageID >> 8), (uint8_t)(pageID & 0xFF)};
    sendCommand(CMD_WRITE_VP, ADDR_PAGE_SWITCH, data, 4);
}

uint16_t Dwin::getPage() {
    // Sends a read request. Response is handled in listen().
    readVP(ADDR_PIC_NOW, 1);
    return 0; 
}

void Dwin::resetHMI() {
    // Writing 0x55AA 0x5AA5 to address 0x0004 resets the system.
    uint8_t data[4] = {0x55, 0xAA, 0x5A, 0xA5};
    sendCommand(CMD_WRITE_VP, ADDR_SYS_RESET, data, 4);
}

// --- LANGUAGE & UI ---

// Multi-language Method 1: Page Offset
void Dwin::setLanguageOffset(uint16_t offset) {
    _pageOffset = offset;
}

// Multi-language Method 2: ICL File Switch
void Dwin::setLanguageICL(uint8_t iclID) {
    uint8_t data[4] = {0x5A, 0x00, 0x00, iclID};
    sendCommand(CMD_WRITE_VP, ADDR_ICL_SWITCH, data, 4);
}

void Dwin::setBrightness(uint8_t level) {
    uint8_t data[4] = {0x00, 0x00, 0x00, level}; // Level 0-0x64
    sendCommand(CMD_WRITE_VP, ADDR_BRIGHTNESS, data, 4);
}

void Dwin::setBuzzer(uint16_t durationMs) {
    uint8_t unitTime = durationMs / 10;
    if (unitTime == 0 && durationMs > 0) unitTime = 1;
    
    uint8_t data[2] = {0x00, unitTime};
    sendCommand(CMD_WRITE_VP, ADDR_BUZZER, data, 2);
}

void Dwin::setTextColor(uint16_t spAddress, uint16_t color) {
    // Color is located at SP address + 3rd word offset
    writeVP(spAddress + 0x03, color);
}

// --- LISTENER & PARSER ---

void Dwin::listen() {
    while (_serial->available()) {
        uint8_t inByte = _serial->read();

        switch (_state) {
            case STATE_IDLE:
                if (inByte == DWIN_HEADER_1) _state = STATE_HEADER1;
                break;
            
            case STATE_HEADER1:
                if (inByte == DWIN_HEADER_2) _state = STATE_HEADER2;
                else _state = STATE_IDLE; 
                break;

            case STATE_HEADER2:
                _frameLen = inByte;
                _rxIndex = 0;
                _state = STATE_LEN;
                break;

            case STATE_LEN:
                _frameCmd = inByte;
                _state = STATE_CMD;
                break;

            case STATE_CMD:
                // Buffer the data
                if (_rxIndex < sizeof(_rxBuffer)) {
                    _rxBuffer[_rxIndex++] = inByte;
                }
                
                // Packet Complete? (FrameLen includes Cmd, so -1)
                if (_rxIndex >= _frameLen - 1) {
                    _state = STATE_IDLE; // Reset state
                    
                    // If command is 0x83 (Read/Return)
                    if (_frameCmd == CMD_READ_VP) {
                        // _rxBuffer structure: [VP_H] [VP_L] [Len] [Data_H] [Data_L] ...
                        if (_rxIndex >= 4) {
                            uint16_t vpAddr = makeWord(_rxBuffer[0], _rxBuffer[1]);
                            uint8_t dataWordLen = _rxBuffer[2]; // Number of words
                            
                            // Trigger callback if user attached one
                            if (_eventCallback != NULL && dataWordLen > 0) {
                                // Currently returning only the first word.
                                uint16_t dataVal = makeWord(_rxBuffer[3], _rxBuffer[4]);
                                _eventCallback(vpAddr, dataVal);
                            }
                        }
                    }
                }
                break;
        }
    }
}

void Dwin::attachCallback(DwinEventCallback func) {
    _eventCallback = func;
}