/*
  BasicTest.ino - Example for DWIN DGUS HMI Library
  This example demonstrates:
  1. Connecting to the screen
  2. Writing values (Integer, Long, Float)
  3. Listening for touch events (Callback)
  4. Multi-language handling
*/

#include <Arduino.h>
#include <DWIN_UNI_HMI.h>

// --- Pin Definitions ---
// For ESP32:
#define RX_PIN 16
#define TX_PIN 17
// For Arduino Uno (SoftwareSerial):
// #include <SoftwareSerial.h>
// SoftwareSerial mySerial(10, 11); // RX, TX

// Create Dwin Object
Dwin hmi;

// --- Callback Function ---
// This function is called automatically when a button is pressed on the HMI
void onHmiEvent(uint16_t address, uint16_t data) {
  Serial.print("Event Address: 0x");
  Serial.print(address, HEX);
  Serial.print(" Data: ");
  Serial.println(data);

  // Example: If button at 0x1000 is pressed
  if (address == 0x1000) {
    Serial.println("Button Pressed! Changing Page...");
    hmi.setPage(1); // Go to Page 1 (Plus language offset)
  }
}

void setup() {
  Serial.begin(115200);
  Serial.println("DWIN HMI Demo Starting...");

  // Initialize HMI Serial
  // ESP32 Hardware Serial 2:
  Serial2.begin(115200, SERIAL_8N1, RX_PIN, TX_PIN);
  hmi.begin(&Serial2, 115200);

  // For Arduino Uno:
  // mySerial.begin(115200);
  // hmi.begin(&mySerial, 115200);

  // Attach the event listener
  hmi.attachCallback(onHmiEvent);

  // --- Test Functions ---
  
  // 1. System Settings
  hmi.setBrightness(80);      // Set Brightness to 80%
  hmi.setBuzzer(50);          // Beep for 500ms
  
  // 2. Language Settings
  // Assume: Page 0-9 is Turkish, 10-19 is English
  hmi.setLanguageOffset(0);   // Set to Turkish
  // hmi.setLanguageOffset(10); // Set to English

  // 3. Write Values
  hmi.writeVP(0x2000, (uint16_t)1234);        // Write Integer
  hmi.writeVP(0x2002, (uint32_t)987654); // Write Long
  hmi.writeVP(0x2004, 3.1415f);     // Write Float

  // 4. Change Text Color (Example)
  // Assume Text Display control is at SP Address 0x5000
  hmi.setTextColor(0x5000, COLOR_RED);
}

void loop() {
  // MUST CALL THIS continuously to receive touch events
  hmi.listen();
}