# Universal DWIN DGUS II Library

A robust, event-driven, and platform-agnostic Arduino library for **DWIN DGUS II** HMI displays. Designed for professional applications requiring reliable communication, multi-language support, and dynamic UI control.

Compatible with **ESP32, STM32, Arduino AVR**, and any microcontroller supporting the `Stream` class.

## 🚀 Features

* **Event-Driven Architecture:** Non-blocking `listen()` function with callback support. No more `delay()` loops waiting for screen response.
* **Platform Agnostic:** Works on HardwareSerial (ESP32/STM32) and SoftwareSerial (AVR).
* **Multi-Language Support:** Built-in management for multi-language projects via Page Offset or Native ICL switching.
* **Complete Data Types:** Read/Write support for 16-bit Integer, 32-bit Long, and Float (IEEE 754).
* **System Controls:** Easy APIs for Page Switching, Brightness, Buzzer, and RTC.
* **Dynamic UI:** Change Text Colors and other SP (Description Pointer) attributes at runtime.

## 📦 Installation

### Method 1: Via Arduino Library Manager (Recommended)
1.  Open **Arduino IDE**.
2.  Go to `Sketch` -> `Include Library` -> `Manage Libraries...` (or click the Library icon on the left).
3.  Search for **"DWIN_UNI_HMI"**.
4.  Click **Install**.

### Method 2: Manual Installation (ZIP)
1.  Download this repository as a ZIP file.
2.  Open Arduino IDE.
3.  Go to `Sketch` -> `Include Library` -> `Add .ZIP Library`.
4.  Select the downloaded ZIP file.

## 🔌 Wiring (Example for ESP32)

| DWIN HMI | ESP32 | Description |
| :--- | :--- | :--- |
| **TX** | GPIO 16 (RX2) | HMI Transmit -> MCU Receive |
| **RX** | GPIO 17 (TX2) | HMI Receive <- MCU Transmit |
| **VCC** | 5V / 3.3V | Power Supply |
| **GND** | GND | Ground (Common) |

> **Note:** Ensure your DWIN screen matches the logic voltage of your MCU (3.3V vs 5V). Use a level shifter if necessary.

## 💻 Usage Example

```cpp
#include <DWIN_UNI_HMI.h>

// ESP32 Serial2 Pins
#define RX_PIN 16
#define TX_PIN 17

Dwin hmi;

// Callback function to handle touch events
void onEvent(uint16_t address, uint16_t data) {
  if (address == 0x1000) {
    // Button at 0x1000 pressed
    hmi.setPage(1); 
  }
}

void setup() {
  Serial2.begin(115200, SERIAL_8N1, RX_PIN, TX_PIN);
  hmi.begin(&Serial2, 115200);
  
  hmi.attachCallback(onEvent); // Register callback
  
  hmi.setBrightness(100);      // Set max brightness
}

void loop() {
  hmi.listen(); // Keep listening for events
}


📚 API Reference
Core
begin(Stream *port, long baud): Initialize library.

listen(): Process incoming data. Call this in loop().

attachCallback(callbackFunc): Register a function to handle touch events.

Data Read/Write
writeVP(addr, value): Write 16-bit integer.

writeVP(addr, longVal): Write 32-bit long.

writeVP(addr, floatVal): Write Float number.

readVP(addr, len): Request data (Response comes via callback).

System & UI
setPage(pageID): Switch page (Language offset aware).

setBrightness(0-100): Adjust backlight.

setBuzzer(timeMs): Beep the buzzer.

setTextColor(spAddr, color): Change text color dynamically.

Multi-Language
setLanguageOffset(offset): Adds an offset to setPage calls (e.g., 0 for EN, 10 for TR).

setLanguageICL(iclID): Switches the background ICL file (0xDE command).

📄 License
This library is open-source and available under the MIT License.