/*
 What this code does
- Initializes WiFi, syncs time using NTP, and connects to MQTT.
- Reads DHT22 sensor values periodically (every 5 seconds → miscommented as 1 sec).
- Stores sensor readings (temperature, humidity, timestamp) in a buffer (up to 200 readings).
- Supports two batch modes:
  • Daily batch → publishes all buffered readings to MQTT at the specified time (cron format).
  • Periodic batch → can be enabled for frequent sending (currently disabled).
- Uses cron-style scheduling (esp_cron) instead of delay-based timing.
- Buffers readings to reduce network traffic.
- MQTT reconnect logic included, closes connection after publishing.

 Required Libraries (install via Arduino Library Manager)
- DHT sensor library by Adafruit
- PubSubClient
- esp-cron
- WiFi (built-in for ESP32)
*/

#include <Arduino.h>
#include <WiFi.h>            // WiFi connectivity
#include <PubSubClient.h>    // MQTT communication
#include "DHT.h"             // DHT sensor handling
#ifdef __cplusplus
extern "C" {
#endif
  #include <esp_cron.h>
#ifdef __cplusplus
}
#endif
// -------------------- Configuration --------------------
#define DHTPIN  4        // GPIO pin for DHT sensor
#define DHTTYPE DHT22    // Sensor type: DHT11 or DHT22
#define WIFI_SSID ""     // Change SSID here
#define WIFI_PASS ""     // Change password here
#define WIFI_CHANNEL 6   // Optional: speeds up WiFi connection
#define MQTT_SERVER "broker.hivemq.com"
#define MQTT_PORT 1883
#define MQTT_TOPIC_PERIODIC "esp32/dht/periodic"
#define MQTT_TOPIC_DAILY    "esp32/dht/daily"

// ---------------- Globals ----------------
WiFiClient espClient;
PubSubClient mqttClient(espClient);
DHT dht(DHTPIN, DHTTYPE);

// ---------------- Data Buffer ----------------
struct DHTReading {
  float temperature;
  float humidity;
  unsigned long timestamp;
};

#define MAX_READINGS 200
DHTReading readings[MAX_READINGS];
int readingCount = 0;

// ---------------- Flags ----------------
volatile bool readFlag = false;
volatile bool sendFlag = false;
volatile bool dailyBatchFlag = false;

// ---------------- Cron Callbacks ----------------
static void readJob(cron_job *job) { readFlag = true; }
static void sendJob(cron_job *job) { sendFlag = true; }
static void dailyBatchJob(cron_job *job) {
  dailyBatchFlag = true;
  Serial.println("[CRON] Daily batch trigger reached.");
}

// ---------------- MQTT Reconnect ----------------
unsigned long lastReconnectAttempt = 0;
bool reconnectMQTT() {
  Serial.print("[MQTT] Attempting connection...");
  bool ok = mqttClient.connect("ESP32Client");
  if (ok) {
    Serial.println(" success.");
    mqttClient.subscribe(MQTT_TOPIC_PERIODIC);
    mqttClient.subscribe(MQTT_TOPIC_DAILY);
  } else {
    Serial.printf(" failed (rc=%d)\n", mqttClient.state());
  }
  return ok;
}

// ---------------- Helpers ----------------
void setupWiFi() {
  Serial.printf("Connecting to WiFi %s", WIFI_SSID);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println(" connected.");
}

void setupMQTT() {
  mqttClient.setServer(MQTT_SERVER, MQTT_PORT);
  while (!mqttClient.connected()) {
    Serial.print("Connecting to MQTT...");
    if (mqttClient.connect("ESP32Client")) {
      Serial.println(" connected.");
    } else {
      Serial.printf(" failed, rc=%d retrying in 2 seconds\n", mqttClient.state());
      delay(2000);
    }
  }
}

void syncNTPTime() {
  configTime(0, 0, "pool.ntp.org", "time.nist.gov");
  setenv("TZ", "CET-1CEST,M3.5.0/2,M10.5.0/3", 1);  // Germany CET/CEST
  tzset();

  Serial.print("Waiting for NTP time sync");
  struct tm timeinfo;
  while (!getLocalTime(&timeinfo)) {
    Serial.print(".");
    delay(500);
  }
  Serial.println(" done.");

  char buf[64];
  strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", &timeinfo);
  Serial.printf("Current local time (Germany): %s\n", buf);
}

void publishBatch(const char *topic, const char *label) {
  if (readingCount == 0) {
    Serial.printf("[%s] No readings to send.\n", label);
    return;
  }

  if (!mqttClient.connected()) {
    if (!reconnectMQTT()) {
      Serial.println("[WARN] MQTT not connected, skipping publish.");
      return;
    }
  }

  String json = "[";
  for (int i = 0; i < readingCount; i++) {
    json += String("{\"t\":") + readings[i].temperature +
            ",\"h\":" + readings[i].humidity +
            ",\"ts\":" + readings[i].timestamp + "}";
    if (i < readingCount - 1) json += ",";
  }
  json += "]";

  mqttClient.publish(topic, json.c_str());
  Serial.printf("[%s] Published %d readings.\n", label, readingCount);
  readingCount = 0;

  mqttClient.disconnect();  // Close connection after publish
}

// ---------------- Arduino Setup ----------------
void setup() {
  Serial.begin(115200);
  dht.begin();

  setupWiFi();
  syncNTPTime();
  setupMQTT();

  cron_job_create("*/5 * * * * *", readJob, NULL);     // Every 5 seconds
  // cron_job_create("*/10 * * * * *", sendJob, NULL); // Optional periodic sending
  cron_job_create("0 10 12 * * *", dailyBatchJob, NULL); // Daily at 12:10:00
  cron_start();

  Serial.println("Cron-based DHT batch MQTT started.");
}

// ---------------- Main Loop ----------------
void loop() {
  if (WiFi.status() != WL_CONNECTED) return;

  mqttClient.loop();

  if (readFlag) {
    readFlag = false;
    float t = dht.readTemperature();
    float h = dht.readHumidity();
    if (!isnan(t) && !isnan(h)) {
      if (readingCount < MAX_READINGS) {
        readings[readingCount++] = {t, h, millis()};
      }
      Serial.printf("[READ] T=%.2f°C H=%.2f%% stored=%d\r\n", t, h, readingCount);
    }
  }

  if (sendFlag) {
    sendFlag = false;
    publishBatch(MQTT_TOPIC_PERIODIC, "PERIODIC SEND");
  }

  if (dailyBatchFlag) {
    dailyBatchFlag = false;
    publishBatch(MQTT_TOPIC_DAILY, "DAILY SEND");
  }

  delay(50);
}
