/*
 What this code does
- Initializes WiFi and MQTT connection.
- Reads DHT22 sensor values every 1 second.
- Stores sensor values (temperature, humidity, and timestamp) in an array.
- Every 5 seconds, it converts stored readings into JSON format and publishes them to an MQTT broker.
- Uses cron-style scheduling (esp_cron) instead of delay-based timing.
- Buffers readings instead of sending each individually to minimize network usage.

 Required Libraries (install via Arduino Library Manager)
- DHT sensor library by Adafruit
- PubSubClient
- esp-cron
- WiFi (built-in for ESP32)
*/

#include <Arduino.h>
#include <WiFi.h>          // WiFi connectivity
#include <PubSubClient.h>  // MQTT communication
#include "DHT.h"           // DHT sensor handling
#include "esp_cron.h"      // Cron-style task scheduling

// -------------------- Configuration --------------------
#define DHTPIN  4          // GPIO pin for DHT sensor
#define DHTTYPE DHT22      // DHT11 or DHT22
#define WIFI_SSID ""       // Change SSID here
#define WIFI_PASS ""       // Change password here
#define WIFI_CHANNEL 6     // Optional: speeds up WiFi connection
#define MQTT_SERVER "broker.hivemq.com" // Change MQTT broker
#define MQTT_PORT 1883                  // Change MQTT port
#define MQTT_TOPIC "esp32/dht/batch"    // Change MQTT topic

// -------------------- Globals --------------------
WiFiClient espClient;
PubSubClient mqttClient(espClient);
DHT dht(DHTPIN, DHTTYPE);

unsigned long lastReconnectAttempt = 0;

bool reconnectMQTT() {
  if (mqttClient.connect("ESP32Client")) {
    Serial.println("MQTT reconnected.");
    mqttClient.subscribe(MQTT_TOPIC);
  }
  return mqttClient.connected();
}

struct DHTReading {
  float temperature;
  float humidity;
  unsigned long timestamp;
};

#define MAX_READINGS 20
DHTReading readings[MAX_READINGS];
int readingCount = 0;

volatile bool readFlag = false;
volatile bool sendFlag = false;

static void readJob(cron_job *job) {
  readFlag = true;
}

static void sendJob(cron_job *job) {
  sendFlag = true;
}

// -------------------- Setup helpers --------------------
void setupWiFi() {
  Serial.print("Connecting to WiFi...");
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println(" connected.");
}

void setupMQTT() {
  mqttClient.setServer(MQTT_SERVER, MQTT_PORT);
  while (!mqttClient.connected()) {
    Serial.print("Connecting to MQTT...");
    if (mqttClient.connect("ESP32Client")) {
      Serial.println(" connected.");
    } else {
      Serial.print(" failed, rc=");
      Serial.print(mqttClient.state());
      Serial.println(" retrying in 2 seconds");
      delay(2000);
    }
  }
}

// -------------------- Arduino Setup/Loop --------------------
void setup() {
  Serial.begin(115200);
  dht.begin();
  setupWiFi();
  setupMQTT();

  esp_timer_init();

  // Read every 1 second
  cron_job_create("*/1 * * * * *", readJob, NULL);

  // Send every 5 seconds
  cron_job_create("*/5 * * * * *", sendJob, NULL);

  cron_start();
  Serial.println("Cron-based DHT batch MQTT started.");
}

void loop() {
  mqttClient.loop();

  if (!mqttClient.connected()) {
    unsigned long now = millis();
    if (now - lastReconnectAttempt > 5000) {
      lastReconnectAttempt = now;
      reconnectMQTT();
    }
  }

  if (readFlag) {
    readFlag = false;
    float t = dht.readTemperature();
    float h = dht.readHumidity();
    if (!isnan(t) && !isnan(h)) {
      readings[readingCount++] = {t, h, millis()};
      Serial.printf("[READ] T=%.2f°C H=%.2f%% stored=%d\r\n", t, h, readingCount);
    }
  }

  if (sendFlag) {
    sendFlag = false;
    if (mqttClient.connected() && readingCount > 0) {
      String json = "[";
      for (int i = 0; i < readingCount; i++) {
        json += String("{\"t\":") + readings[i].temperature +
                ",\"h\":" + readings[i].humidity +
                ",\"ts\":" + readings[i].timestamp + "}";
        if (i < readingCount - 1) json += ",";
      }
      json += "]";

      mqttClient.publish(MQTT_TOPIC, json.c_str());
      Serial.printf("[SEND] Published %d readings\r\n", readingCount);
      readingCount = 0;
    } else {
      Serial.println("[SEND] No readings to send.");
    }
  }

  delay(50);
}
