/*
    This file is part of the Arduino_RouterBridge library.

    Copyright (c) 2025 Arduino SA

    This Source Code Form is subject to the terms of the Mozilla Public
    License, v. 2.0. If a copy of the MPL was not distributed with this
    file, You can obtain one at http://mozilla.org/MPL/2.0/.

*/

#pragma once

#ifndef BRIDGE_TCP_CLIENT_H
#define BRIDGE_TCP_CLIENT_H

#define TCP_CONNECT_METHOD          "tcp/connect"
#define TCP_CLOSE_METHOD            "tcp/close"
#define TCP_WRITE_METHOD            "tcp/write"
#define TCP_READ_METHOD             "tcp/read"

#include <api/RingBuffer.h>
#include <api/Client.h>
#include "bridge.h"

#define DEFAULT_TCP_CLIENT_BUF_SIZE    512

template<size_t BufferSize=DEFAULT_TCP_CLIENT_BUF_SIZE>
class BridgeTCPClient final: public Client {

    BridgeClass* bridge;
    uint32_t connection_id{};
    RingBufferN<BufferSize> temp_buffer;
    struct k_mutex client_mutex{};
    bool _connected = false;

public:
    explicit BridgeTCPClient(BridgeClass& bridge): bridge(&bridge) {}

    bool begin() {
        k_mutex_init(&client_mutex);
        if (!(*bridge)) {
            return bridge->begin();
        }
        return true;
    }

    int connect(IPAddress ip, uint16_t port) override {
        return connect(ip.toString().c_str(), port);
    }

    int connect(const char *host, uint16_t port) override {

        String send_buffer = host;
        send_buffer += ":";
        send_buffer += String(port);

        k_mutex_lock(&client_mutex, K_FOREVER);

        const bool resp = bridge->call(TCP_CONNECT_METHOD, connection_id, send_buffer);

        if (!resp) {
            _connected = false;
            k_mutex_unlock(&client_mutex);
            return -1;
        }
        _connected = true;

        k_mutex_unlock(&client_mutex);

        return 0;
    }

    size_t write(uint8_t c) override {
        return write(&c, 1);
    }

    size_t write(const uint8_t *buf, size_t size) override {
        String send_buffer;

        for (size_t i = 0; i < size; ++i) {
            send_buffer += static_cast<char>(buf[i]);
        }

        size_t written;
        const bool ret = bridge->call(TCP_WRITE_METHOD, written, send_buffer);
        if (ret) {
            return written;
        }

        return 0;
    }

    int available() override {
        k_mutex_lock(&client_mutex, K_FOREVER);
        const int size = temp_buffer.availableForStore();
        if (size > 0) _read(size);
        const int available = temp_buffer.available();
        k_mutex_unlock(&client_mutex);
        return available;
    }

    int read() override {
        uint8_t c;
        read(&c, 1);
        return c;
    }

    int read(uint8_t *buf, size_t size) override {
        k_mutex_lock(&client_mutex, K_FOREVER);
        int i = 0;
        while (temp_buffer.available() && i < size) {
            buf[i++] = temp_buffer.read_char();
        }
        k_mutex_unlock(&client_mutex);
        return i;
    }

    int peek() override {
        k_mutex_lock(&client_mutex, K_FOREVER);
        if (temp_buffer.available()) {
            k_mutex_unlock(&client_mutex);
            return temp_buffer.peek();
        }
        k_mutex_unlock(&client_mutex);
        return -1;
    }

    void flush() override {
        // No-op: flush is implemented for Client subclasses using an output buffer
    }

    void stop() override {
        k_mutex_lock(&client_mutex, K_FOREVER);
        String msg;
        const bool resp = bridge->call(TCP_CLOSE_METHOD, msg, connection_id);
        if (resp) {
            _connected = false;
        }
        k_mutex_unlock(&client_mutex);
    }

    uint8_t connected() override {
        if (_connected) return 1;
        return 0;
    }

    operator bool() override {
        return available() || connected();
    }

private:
    void _read(size_t size) {

        if (size == 0 || !_connected) return;

        k_mutex_lock(&client_mutex, K_FOREVER);

        MsgPack::arr_t<uint8_t> message;
        const bool ret = bridge->call(TCP_READ_METHOD, message, size);

        if (ret) {
            for (size_t i = 0; i < message.size(); ++i) {
                temp_buffer.store_char(static_cast<char>(message[i]));
            }
        }

        if (bridge->get_last_client_error().code > NO_ERR) {
            _connected = false;
        }

        k_mutex_unlock(&client_mutex);
    }
    
};


#endif //BRIDGE_TCP_CLIENT_H
