package testsuite

import (
	"encoding/hex"
	"encoding/json"
	"fmt"
	"io"
	"os"
	"testing"

	"github.com/arduino/go-paths-helper"
	"github.com/stretchr/testify/require"
)

// UploadSketchAndGetRPCAndDebugPorts uploads the sketch to the board and
// returns the RPC and debug ports. The sketch is supposed to be
// located in the current directory.
func UploadSketchAndGetRPCAndDebugPorts(t *testing.T) (rpc string, debug string) {
	sketchName := t.Name()
	fqbn, _, uploadPort := getFQBNAndPorts(t)
	{
		// Copy serial_ports.h file to the sketch directory
		serialPortHeader := []byte(fmt.Sprintln("// This file is automatically generated from ../serial_ports.h. DO NOT EDIT."))
		serialPortsH, err := paths.New("serial_ports.h").ReadFile()
		require.NoError(t, err)
		err = paths.New(sketchName, "serial_ports.h").WriteFile(append(serialPortHeader, serialPortsH...))
		require.NoError(t, err)

		// Upload the sketch
		cli, err := paths.NewProcess(nil, "arduino-cli", "compile", "--fqbn", fqbn, "--library", "../..", "-u", "-p", uploadPort, sketchName)
		require.NoError(t, err)
		cli.RedirectStderrTo(os.Stderr)
		cli.RedirectStdoutTo(os.Stdout)
		require.NoError(t, cli.Run())
	}

	// Get the rpc and debug ports
	fqbn2, rpcPort, debugPort := getFQBNAndPorts(t)
	require.Equal(t, fqbn, fqbn2, "FQBN mismatch between upload and run ports: %s != %s", fqbn, fqbn2)
	return rpcPort, debugPort
}

// getFQBNAndPorts retrieves the FQBN of the board under test and the
// corresponding upload and RPC ports. Debugging messages will be output
// in the upload port, the RPC communication will be done on the RPC port.
// If the board do not have a second serial port, the RPC port will be
// assigned to a USB-2-Serial dongle/converter if found.
func getFQBNAndPorts(t *testing.T) (fqbn string, rpcPort string, uploadPort string) {
	cli, err := paths.NewProcess(nil, "arduino-cli", "board", "list", "--json")
	require.NoError(t, err)
	out, _, err := cli.RunAndCaptureOutput(t.Context())
	require.NoError(t, err)
	var cliResult struct {
		DetectedPorts []struct {
			MatchingBoards []struct {
				Fqbn string `json:"fqbn"`
			} `json:"matching_boards"`
			Port struct {
				Address    string `json:"address"`
				Properties struct {
					Vid string `json:"vid"`
					Pid string `json:"pid"`
				} `json:"properties"`
			} `json:"port"`
		} `json:"detected_ports"`
	}
	require.NoError(t, json.Unmarshal(out, &cliResult))
	checkFQBN := func(boardFQBN string) {
		if fqbn != boardFQBN {
			fqbn = boardFQBN
			uploadPort = ""
			rpcPort = ""
		}
	}
	for _, port := range cliResult.DetectedPorts {
		for _, board := range port.MatchingBoards {
			if board.Fqbn == "arduino:mbed_giga:giga" {
				checkFQBN(board.Fqbn)
				uploadPort = port.Port.Address
			}
			if board.Fqbn == "arduino:samd:arduino_zero_edbg" {
				checkFQBN("arduino:samd:arduino_zero_native")
				rpcPort = port.Port.Address
			}
			if board.Fqbn == "arduino:samd:arduino_zero_native" {
				checkFQBN(board.Fqbn)
				uploadPort = port.Port.Address
			}
			if board.Fqbn == "arduino:mbed_nano:nanorp2040connect" {
				checkFQBN(board.Fqbn)
				uploadPort = port.Port.Address
			}
		}
	}
	if rpcPort == "" {
		for _, port := range cliResult.DetectedPorts {
			if port.Port.Properties.Vid == "0x0483" && port.Port.Properties.Pid == "0x374B" {
				rpcPort = port.Port.Address
			}
			if port.Port.Properties.Vid == "0x1A86" && port.Port.Properties.Pid == "0x55D4" {
				rpcPort = port.Port.Address
			}
		}
	}
	require.NotEmpty(t, uploadPort, "Upload port not found")
	require.NotEmpty(t, rpcPort, "Debug port not found")
	return fqbn, rpcPort, uploadPort
}

// Expect checks that the input stream returns the expected
// string. It reads the input stream until it has read the
// expected number of bytes. It is used to check the output
// of the Arduino board.
func Expect(t *testing.T, in io.Reader, expected string) {
	buff := make([]byte, len(expected))
	read := 0
	for read < len(expected) {
		n, err := in.Read(buff[read:])
		require.NoError(t, err)
		read += n
	}
	require.Equal(t, expected, string(buff))
}

// DebugStream is a wrapper around io.ReadWriteCloser that logs the data
// read and written to the stream in hex format.
// It is used to debug the communication with the Arduino board.
type DebugStream struct {
	Upstream io.ReadWriteCloser
	Portname string
}

func (d *DebugStream) Read(p []byte) (n int, err error) {
	n, err = d.Upstream.Read(p)
	if err != nil {
		fmt.Printf("%s READ  ERROR: %v\n", d.Portname, err)
	} else {
		fmt.Printf("%s READ  << %s\n", d.Portname, hex.EncodeToString(p[:n]))
	}
	return n, err
}

func (d *DebugStream) Write(p []byte) (n int, err error) {
	n, err = d.Upstream.Write(p)
	if err != nil {
		fmt.Printf("%s WRITE ERROR: %v\n", d.Portname, err)
	} else {
		fmt.Printf("%s WRITE >> %s\n", d.Portname, hex.EncodeToString(p[:n]))
	}
	return n, err
}

func (d *DebugStream) Close() error {
	err := d.Upstream.Close()
	fmt.Printf("%s CLOSE", d.Portname)
	if err != nil {
		fmt.Printf(" (ERROR: %v)", err)
	}
	fmt.Println()
	return err
}
