/*
    This file is part of the Arduino_CloudUtils library.

    Copyright (c) 2024 Arduino SA

    This Source Code Form is subject to the terms of the Mozilla Public
    License, v. 2.0. If a copy of the MPL was not distributed with this
    file, You can obtain one at http://mozilla.org/MPL/2.0/.
*/

#include <catch2/catch_test_macros.hpp>
#include <crc/crc16.h>

SCENARIO("Test the crc16 function for provisioning protocol") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x1A14);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xC2E3);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x00);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xF078);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x34F4);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with CCIT_ZERO") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8208);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x3063);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x00);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x00);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x50C6);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with ARC") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xBB0E);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0140);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0,  0x8005, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x00);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x8005, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x00);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x8005, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xADF1);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with AUG CCITT") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x1D0F, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xB336);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x1D0F, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFCFF);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0,  0x1021, 0x1D0F, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x1D0F);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x1021, 0x1D0F, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xCC9C);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x1021, 0x1D0F, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x28F5);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with BUYPASS") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xB059);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x000A);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0,  0x8005, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0000);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x8005, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0000);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x8005, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xD135);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with CCITT_FALSE") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8C18);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xD193);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE1F0);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x52B0);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with CDMA2000") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xC867, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xBD24);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xC867, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFBA2);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0xC867, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xC867, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x6B6C);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xC867, 0xFFFF, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x22C2);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with DDS 110") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x800D, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xB289);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x800D, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8E09);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x8005, 0x800D, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x800D);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x800D, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8E03);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x800D, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8E47);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with DECT R") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0001, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x834F);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0001, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0E9A);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x0589, 0x0, 0x0001, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0001);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0001, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0001);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0001, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE582);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with DECT X") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x834E);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0E9B);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x0589, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0000);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0000);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x0589, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE583);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with DNP") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE674);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xA51D);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x3D65, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xC126);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with EN 13757") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE8BC);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data),  0x3D65, 0x0, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xB850);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x3D65, 0x0, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x3D65, 0x0, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xB736);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with GENIBUS") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x73E7);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x2E6C);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0xFFFF, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x1E0F);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0xFFFF, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xAD4F);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with MAXIM") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x44F1);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFEBF);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x8005, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0x0, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x520E);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with MCRF4XX") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE5EB);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x3D1C);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0F87);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xCB0B);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with RIELLO") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xB2AA, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x89A3);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xB2AA, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xAB2F);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0xB2AA, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x554D);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xB2AA, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x99B4);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xB2AA, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xE08A);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with T10 DIF") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8BB7, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x7892);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8BB7, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x176E);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x8BB7, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8BB7, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8BB7, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x84C8);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with TELEDISK") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xA097, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x530D);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xA097, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x412E);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0xA097, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xA097, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0xA097, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x7BA6);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with TMS37157") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x89EC, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x4AAF);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x89EC, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xB7AC);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0x89EC, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x3791);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x89EC, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8537);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x89EC, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x3143);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with USB") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x5FF1);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xBE00);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x8005, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xBF40);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0xFFFF, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xBB56);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with A") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xC6C6, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xDA28);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xC6C6, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x6365);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0xC6C6, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x6363);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xC6C6, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x51FE);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0xC6C6, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x1DFC);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with KERMIT") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xED9B);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x329B);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xA54B);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with MODBUS") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xA00E);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x41FF);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x8005, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0xFFFF);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x40BF);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x8005, 0xFFFF, 0x0, true, true);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x44A9);
    }
  }

  /****************************************************************************/
}

SCENARIO("Test the crc16 function with XMODEM") {
  /****************************************************************************/

  WHEN("Calculate crc16 numbers 0-5")
  {
    uint8_t data[] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x8208);
    }
  }

  WHEN("Calculate crc16 single byte")
  {
    uint8_t data[] = {0x03};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x3063);
    }
  }

  WHEN("Calculate crc16 empty data")
  {
    uint8_t data;
    uint16_t crc = arduino::crc16::calculate(&data, 0, 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 byte 0x00")
  {
    uint8_t data[] = {0x00};
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x0);
    }
  }

  WHEN("Calculate crc16 random data for provisioning protocol")
  {
    uint8_t data[] = {0xDA, 0x00, 0x01, 0x20, 0x01, 0x8E, 0x6F, 0x46, 0x41, 0x53, 0x54, 0x57,
                    0x45, 0x42, 0x2D, 0x46, 0x42, 0x45, 0x31, 0x34, 0x31, 0x32, 0x38, 0x26, 0x72, 0x56, 0x6F, 0x64,
                    0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x41, 0x33, 0x37, 0x35, 0x30, 0x35, 0x38, 0x34, 0x32, 0x38,
                    0x2D, 0x6D, 0x56, 0x6F, 0x64, 0x61, 0x66, 0x6F, 0x6E, 0x65, 0x2D, 0x57, 0x69, 0x46, 0x69, 0x38,
                    0x2E, 0x78, 0x1B, 0x44, 0x49, 0x52, 0x45, 0x43, 0x54, 0x2D, 0x38, 0x30, 0x2D, 0x48, 0x50, 0x20,
                    0x4F, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4A, 0x65, 0x74, 0x20, 0x33, 0x38, 0x33, 0x30, 0x38, 0x5C,
                    0x6F, 0x69, 0x6C, 0x69, 0x61, 0x64, 0x62, 0x6F, 0x78, 0x2D, 0x36, 0x30, 0x32, 0x38, 0x43, 0x35,
                    0x38, 0x52, 0x6F, 0x69, 0x50, 0x68, 0x6F, 0x6E, 0x65, 0x20, 0x64, 0x69, 0x20, 0x4C, 0x75, 0x61,
                    0x6E, 0x61, 0x38, 0x56, 0x70, 0x57, 0x69, 0x6E, 0x64, 0x33, 0x20, 0x48, 0x55, 0x42, 0x2D, 0x37,
                    0x45, 0x34, 0x37, 0x30, 0x43, 0x38, 0x46};
    
    uint16_t crc = arduino::crc16::calculate(data, sizeof(data), 0x1021, 0x0, 0x0, false, false);
    THEN("The crc16 is calculated correctly")
    {
      REQUIRE(crc == 0x50C6);
    }
  }

  /****************************************************************************/
}
