/*
    This file is part of the Arduino_AlvikCarrier library.

    Copyright (c) 2024 Arduino SA

    This Source Code Form is subject to the terms of the Mozilla Public
    License, v. 2.0. If a copy of the MPL was not distributed with this
    file, You can obtain one at http://mozilla.org/MPL/2.0/.
    
*/

#include "Arduino_APDS9999.h"

Arduino_APDS9999::Arduino_APDS9999(TwoWire & wire):_wire(wire){

}

int Arduino_APDS9999::begin(){
  _wire.begin();
  if (getID()!=APDS9999_DEFAULT_ID){
    return 0;
  }
  writeRegister(APDS9999_MAIN_CTRL,APDS9999_SOFT_RST);
  delay(500);
  writeRegister(APDS9999_MAIN_CTRL, 0x00);
  return 1;
}

void Arduino_APDS9999::end(){
  _wire.end();
}

void Arduino_APDS9999::writeRegister(uint8_t reg, uint8_t value) {
  _wire.beginTransmission(APDS9999_ADDRESS);
  _wire.write(reg);
  _wire.write(value);
  _wire.endTransmission();
}

uint8_t Arduino_APDS9999::readRegister(uint8_t reg) {
  _wire.beginTransmission(APDS9999_ADDRESS);
  _wire.write(reg);
  _wire.endTransmission();
  
  _wire.requestFrom(APDS9999_ADDRESS, 1);
  if (_wire.available()) {
    return _wire.read();
  }
  return 0;
}

uint32_t Arduino_APDS9999::readRegister24(uint8_t reg) {
  _wire.beginTransmission(APDS9999_ADDRESS);
  _wire.write(reg);
  _wire.endTransmission();

  _wire.requestFrom(APDS9999_ADDRESS, 3);
  if (_wire.available() >= 3) {
    uint32_t value = _wire.read();
    value |= (_wire.read() << 8);
    value |= (_wire.read() << 16);
    return value;
  }
  return 0;
}

//--------------------------------------------------------------------------//
//                              Sensor Data                                 //
//--------------------------------------------------------------------------//


void Arduino_APDS9999::enableLightSensor(){
  uint8_t ctrl = readRegister(APDS9999_MAIN_CTRL);
  writeRegister(APDS9999_MAIN_CTRL, ctrl | APDS9999_LS_ENABLE);
  delay(20);
}

void Arduino_APDS9999::disableLightSensor(){
  uint8_t ctrl = readRegister(APDS9999_MAIN_CTRL);
  writeRegister(APDS9999_MAIN_CTRL, ctrl & ~APDS9999_LS_ENABLE);
}

void Arduino_APDS9999::enableColorSensor(){
  enableLightSensor();
  uint8_t ctrl = readRegister(APDS9999_MAIN_CTRL);
  writeRegister(APDS9999_MAIN_CTRL, ctrl | APDS9999_CS_ENABLE);
  delay(20);
}

void Arduino_APDS9999::disableColorSensor(){
  uint8_t ctrl = readRegister(APDS9999_MAIN_CTRL);
  writeRegister(APDS9999_MAIN_CTRL, ctrl & ~APDS9999_CS_ENABLE);
}

void Arduino_APDS9999::enableProximitySensor(){
  uint8_t ctrl = readRegister(APDS9999_MAIN_CTRL);
  writeRegister(APDS9999_MAIN_CTRL, ctrl | APDS9999_PS_ENABLE);
  delay(20);
}

void Arduino_APDS9999::disableProximitySensor(){
  uint8_t ctrl = readRegister(APDS9999_MAIN_CTRL);
  writeRegister(APDS9999_MAIN_CTRL, ctrl & ~APDS9999_PS_ENABLE);
}

void Arduino_APDS9999::setGain(const uint8_t gain){
  uint8_t ctrl = readRegister(APDS9999_LS_GAIN);
  writeRegister(APDS9999_LS_GAIN, (ctrl & 0xF8) | gain);
}

void Arduino_APDS9999::setLSResolution(const uint8_t resolution){
  uint8_t ctrl = readRegister(APDS9999_LS_MEAS_R);
  writeRegister(APDS9999_LS_MEAS_R, (ctrl & 0x8F) | (resolution<<4));
}

void Arduino_APDS9999::setLSRate(const uint8_t rate){
  uint8_t ctrl = readRegister(APDS9999_LS_MEAS_R);
  writeRegister(APDS9999_LS_MEAS_R, (ctrl & 0xF8) | rate);
}

void Arduino_APDS9999::setPSResolution(const uint8_t resolution){
  uint8_t ctrl = readRegister(APDS9999_PS_MEAS_R);
  writeRegister(APDS9999_PS_MEAS_R, (ctrl & 0xE7) | (resolution<<3));
}

void Arduino_APDS9999::setPSRate(const uint8_t rate){
  uint8_t ctrl = readRegister(APDS9999_PS_MEAS_R);
  writeRegister(APDS9999_PS_MEAS_R, (ctrl & 0xF8) | rate);
}


int Arduino_APDS9999::getIR(){
  return readRegister24(APDS9999_IR_DATA);
}

int Arduino_APDS9999::getRed(){
  return readRegister24(APDS9999_RED_DATA);
}

int Arduino_APDS9999::getGreen(){
  return readRegister24(APDS9999_GREEN_DATA);
}

int Arduino_APDS9999::getBlue(){
  return readRegister24(APDS9999_BLUE_DATA);
}

void Arduino_APDS9999::readColor(int & red, int & green, int & blue){
  red = getRed();
  green = getGreen();
  blue = getBlue();
}

void Arduino_APDS9999::readColor(int & red, int & green, int & blue, int & ir){
  red = getRed();
  green = getGreen();
  blue = getBlue();
  ir = getIR();
}

int Arduino_APDS9999::getProximity(){
  uint8_t lsb = readRegister(APDS9999_PS_DATA_L);
  uint8_t msb =readRegister(APDS9999_PS_DATA_H);

  bool overflow = msb & 0x08;  // Check overflow bit in PS_DATA_1

  if (overflow) {
    return -1;
  }

  return (msb & 0x07) << 8 | lsb;
}

uint8_t Arduino_APDS9999::getID(){
  return readRegister(APDS9999_ID);
}



