/*
  This file is part of the ArduinoIoTCloud library.

  Copyright (c) 2019 Arduino SA

  This Source Code Form is subject to the terms of the Mozilla Public
  License, v. 2.0. If a copy of the MPL was not distributed with this
  file, You can obtain one at http://mozilla.org/MPL/2.0/.
*/

/******************************************************************************
  INCLUDE
 ******************************************************************************/

#include <ArduinoIoTCloud.h>

/******************************************************************************
  CTOR/DTOR
 ******************************************************************************/

ArduinoIoTCloudClass::ArduinoIoTCloudClass()
: _connection{nullptr}
#if NETWORK_CONFIGURATOR_ENABLED
, _configurator{nullptr}
#endif
, _time_service(TimeService)
, _thing_id{"xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"}
, _lib_version{AIOT_CONFIG_LIB_VERSION}
, _device_id{"xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"}
, _cloud_event_callback{nullptr}
{

}

/******************************************************************************
  PUBLIC MEMBER FUNCTIONS
 ******************************************************************************/

void ArduinoIoTCloudClass::push()
{
  requestUpdateForAllProperties(getThingPropertyContainer());
}

bool ArduinoIoTCloudClass::setTimestamp(String const & prop_name, unsigned long const timestamp)
{
  Property * p = getProperty(getThingPropertyContainer(), prop_name);

  if (p == nullptr)
    return false;

  p->setTimestamp(timestamp);

  return true;
}

void ArduinoIoTCloudClass::addCallback(ArduinoIoTCloudEvent const event, OnCloudEventCallback callback)
{
  _cloud_event_callback[static_cast<size_t>(event)] = callback;
}

/* The following methods are used for non-LoRa boards */
Property& ArduinoIoTCloudClass::addPropertyReal(bool& property, String name, Permission const permission)
{
  return addPropertyReal(property, name, -1, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(float& property, String name, Permission const permission)
{
  return addPropertyReal(property, name, -1, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(int& property, String name, Permission const permission)
{
  return addPropertyReal(property, name, -1, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(unsigned int& property, String name, Permission const permission)
{
  return addPropertyReal(property, name, -1, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(String& property, String name, Permission const permission)
{
  return addPropertyReal(property, name, -1, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(Property& property, String name, Permission const permission)
{
  return addPropertyReal(property, name, -1, permission);
}

/* The following methods are used for both LoRa and non-Lora boards */
Property& ArduinoIoTCloudClass::addPropertyReal(bool& property, String name, int tag, Permission const permission)
{
  Property* p = new CloudWrapperBool(property);
  return addPropertyReal(*p, name, tag, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(float& property, String name, int tag, Permission const permission)
{
  Property* p = new CloudWrapperFloat(property);
  return addPropertyReal(*p, name, tag, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(int& property, String name, int tag, Permission const permission)
{
  Property* p = new CloudWrapperInt(property);
  return addPropertyReal(*p, name, tag, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(unsigned int& property, String name, int tag, Permission const permission)
{
  Property* p = new CloudWrapperUnsignedInt(property);
  return addPropertyReal(*p, name, tag, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(String& property, String name, int tag, Permission const permission)
{
  Property* p = new CloudWrapperString(property);
  return addPropertyReal(*p, name, tag, permission);
}
Property& ArduinoIoTCloudClass::addPropertyReal(Property& property, String name, int tag, Permission const permission)
{
  return addPropertyToContainer(getThingPropertyContainer(), property, name, permission, tag);
}

/* The following methods are deprecated but still used for non-LoRa boards */
void ArduinoIoTCloudClass::addPropertyReal(bool& property, String name, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperBool(property);
  addPropertyRealInternal(*p, name, -1, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(float& property, String name, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperFloat(property);
  addPropertyRealInternal(*p, name, -1, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(int& property, String name, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperInt(property);
  addPropertyRealInternal(*p, name, -1, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(unsigned int& property, String name, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperUnsignedInt(property);
  addPropertyRealInternal(*p, name, -1, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(String& property, String name, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperString(property);
  addPropertyRealInternal(*p, name, -1, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(Property& property, String name, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  addPropertyRealInternal(property, name, -1, permission_type, seconds, fn, minDelta, synFn);
}

/* The following methods are deprecated but still used for both LoRa and non-LoRa boards */
void ArduinoIoTCloudClass::addPropertyReal(bool& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperBool(property);
  addPropertyRealInternal(*p, name, tag, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(float& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperFloat(property);
  addPropertyRealInternal(*p, name, tag, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(int& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperInt(property);
  addPropertyRealInternal(*p, name, tag, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(unsigned int& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperUnsignedInt(property);
  addPropertyRealInternal(*p, name, tag, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(String& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Property* p = new CloudWrapperString(property);
  addPropertyRealInternal(*p, name, tag, permission_type, seconds, fn, minDelta, synFn);
}
void ArduinoIoTCloudClass::addPropertyReal(Property& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  addPropertyRealInternal(property, name, tag, permission_type, seconds, fn, minDelta, synFn);
}

void ArduinoIoTCloudClass::addPropertyRealInternal(Property& property, String name, int tag, permissionType permission_type, long seconds, void(*fn)(void), float minDelta, void(*synFn)(Property & property))
{
  Permission permission = Permission::ReadWrite;
  if (permission_type == READ) {
    permission = Permission::Read;
  } else if (permission_type == WRITE) {
    permission = Permission::Write;
  } else {
    permission = Permission::ReadWrite;
  }

  if (seconds == ON_CHANGE) {
    addPropertyToContainer(getThingPropertyContainer(), property, name, permission, tag).publishOnChange(minDelta, Property::DEFAULT_MIN_TIME_BETWEEN_UPDATES_MILLIS).onUpdate(fn).onSync(synFn);
  } else {
    addPropertyToContainer(getThingPropertyContainer(), property, name, permission, tag).publishEvery(seconds).onUpdate(fn).onSync(synFn);
  }
}

/******************************************************************************
  PROTECTED MEMBER FUNCTIONS
 ******************************************************************************/

void ArduinoIoTCloudClass::execCloudEventCallback(ArduinoIoTCloudEvent const event)
{
  OnCloudEventCallback callback = _cloud_event_callback[static_cast<size_t>(event)];
  if (callback) {
    (*callback)();
  }
}

__attribute__((weak)) void setDebugMessageLevel(int const /* level */)
{
  /* do nothing */
}
