/*
 * Copyright (c) 2018 Arduino SA. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining 
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be 
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef _BEAR_SSL_TRUST_ANCHORS_H_
#define _BEAR_SSL_TRUST_ANCHORS_H_

#include "bearssl/bearssl_ssl.h"

// The following was created by running the BearSSL "brssl" tool in the 
// extras/TrustAnchors directory:
//
//   brssl ta *.cer
//
//Reading file 'AmazonRootCA1.cer': 1 trust anchor
//Reading file 'BaltimoreCyberTrust.cer': 1 trust anchor
//Reading file 'COMODO RSA Certification Authority.cer': 1 trust anchor
//Reading file 'DigiCertGlobalRootCA.cer': 1 trust anchor
//Reading file 'DigiCertGlobalRootG2.cer': 1 trust anchor
//Reading file 'Digicert_Root.cer': 1 trust anchor
//Reading file 'EnTrust.cer': 1 trust anchor
//Reading file 'GeoTrustPrimaryCertificationAuthority-G3.cer': 1 trust anchor
//Reading file 'GlobalSignRootR1.cer': 1 trust anchor
//Reading file 'GoDaddyRootCertificateAuthority-G2.cer': 1 trust anchor
//Reading file 'ISRGRootX1.cer': 1 trust anchor
//Reading file 'MicrosoftRSARootCertificateAuthority2017.cer': 1 trust anchor
//Reading file 'VeriSign.cer': 1 trust anchor

static const unsigned char TA0_DN[] = {
        0x30, 0x39, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x0F, 0x30, 0x0D, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x06, 0x41, 0x6D, 0x61, 0x7A, 0x6F, 0x6E, 0x31, 0x19, 0x30, 0x17,
        0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x10, 0x41, 0x6D, 0x61, 0x7A, 0x6F,
        0x6E, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41, 0x20, 0x31
};

static const unsigned char TA0_RSA_N[] = {
        0xB2, 0x78, 0x80, 0x71, 0xCA, 0x78, 0xD5, 0xE3, 0x71, 0xAF, 0x47, 0x80,
        0x50, 0x74, 0x7D, 0x6E, 0xD8, 0xD7, 0x88, 0x76, 0xF4, 0x99, 0x68, 0xF7,
        0x58, 0x21, 0x60, 0xF9, 0x74, 0x84, 0x01, 0x2F, 0xAC, 0x02, 0x2D, 0x86,
        0xD3, 0xA0, 0x43, 0x7A, 0x4E, 0xB2, 0xA4, 0xD0, 0x36, 0xBA, 0x01, 0xBE,
        0x8D, 0xDB, 0x48, 0xC8, 0x07, 0x17, 0x36, 0x4C, 0xF4, 0xEE, 0x88, 0x23,
        0xC7, 0x3E, 0xEB, 0x37, 0xF5, 0xB5, 0x19, 0xF8, 0x49, 0x68, 0xB0, 0xDE,
        0xD7, 0xB9, 0x76, 0x38, 0x1D, 0x61, 0x9E, 0xA4, 0xFE, 0x82, 0x36, 0xA5,
        0xE5, 0x4A, 0x56, 0xE4, 0x45, 0xE1, 0xF9, 0xFD, 0xB4, 0x16, 0xFA, 0x74,
        0xDA, 0x9C, 0x9B, 0x35, 0x39, 0x2F, 0xFA, 0xB0, 0x20, 0x50, 0x06, 0x6C,
        0x7A, 0xD0, 0x80, 0xB2, 0xA6, 0xF9, 0xAF, 0xEC, 0x47, 0x19, 0x8F, 0x50,
        0x38, 0x07, 0xDC, 0xA2, 0x87, 0x39, 0x58, 0xF8, 0xBA, 0xD5, 0xA9, 0xF9,
        0x48, 0x67, 0x30, 0x96, 0xEE, 0x94, 0x78, 0x5E, 0x6F, 0x89, 0xA3, 0x51,
        0xC0, 0x30, 0x86, 0x66, 0xA1, 0x45, 0x66, 0xBA, 0x54, 0xEB, 0xA3, 0xC3,
        0x91, 0xF9, 0x48, 0xDC, 0xFF, 0xD1, 0xE8, 0x30, 0x2D, 0x7D, 0x2D, 0x74,
        0x70, 0x35, 0xD7, 0x88, 0x24, 0xF7, 0x9E, 0xC4, 0x59, 0x6E, 0xBB, 0x73,
        0x87, 0x17, 0xF2, 0x32, 0x46, 0x28, 0xB8, 0x43, 0xFA, 0xB7, 0x1D, 0xAA,
        0xCA, 0xB4, 0xF2, 0x9F, 0x24, 0x0E, 0x2D, 0x4B, 0xF7, 0x71, 0x5C, 0x5E,
        0x69, 0xFF, 0xEA, 0x95, 0x02, 0xCB, 0x38, 0x8A, 0xAE, 0x50, 0x38, 0x6F,
        0xDB, 0xFB, 0x2D, 0x62, 0x1B, 0xC5, 0xC7, 0x1E, 0x54, 0xE1, 0x77, 0xE0,
        0x67, 0xC8, 0x0F, 0x9C, 0x87, 0x23, 0xD6, 0x3F, 0x40, 0x20, 0x7F, 0x20,
        0x80, 0xC4, 0x80, 0x4C, 0x3E, 0x3B, 0x24, 0x26, 0x8E, 0x04, 0xAE, 0x6C,
        0x9A, 0xC8, 0xAA, 0x0D
};

static const unsigned char TA0_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA1_DN[] = {
        0x30, 0x5A, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x49, 0x45, 0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x09, 0x42, 0x61, 0x6C, 0x74, 0x69, 0x6D, 0x6F, 0x72, 0x65, 0x31,
        0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x0A, 0x43, 0x79,
        0x62, 0x65, 0x72, 0x54, 0x72, 0x75, 0x73, 0x74, 0x31, 0x22, 0x30, 0x20,
        0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x19, 0x42, 0x61, 0x6C, 0x74, 0x69,
        0x6D, 0x6F, 0x72, 0x65, 0x20, 0x43, 0x79, 0x62, 0x65, 0x72, 0x54, 0x72,
        0x75, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74
};

static const unsigned char TA1_RSA_N[] = {
        0xA3, 0x04, 0xBB, 0x22, 0xAB, 0x98, 0x3D, 0x57, 0xE8, 0x26, 0x72, 0x9A,
        0xB5, 0x79, 0xD4, 0x29, 0xE2, 0xE1, 0xE8, 0x95, 0x80, 0xB1, 0xB0, 0xE3,
        0x5B, 0x8E, 0x2B, 0x29, 0x9A, 0x64, 0xDF, 0xA1, 0x5D, 0xED, 0xB0, 0x09,
        0x05, 0x6D, 0xDB, 0x28, 0x2E, 0xCE, 0x62, 0xA2, 0x62, 0xFE, 0xB4, 0x88,
        0xDA, 0x12, 0xEB, 0x38, 0xEB, 0x21, 0x9D, 0xC0, 0x41, 0x2B, 0x01, 0x52,
        0x7B, 0x88, 0x77, 0xD3, 0x1C, 0x8F, 0xC7, 0xBA, 0xB9, 0x88, 0xB5, 0x6A,
        0x09, 0xE7, 0x73, 0xE8, 0x11, 0x40, 0xA7, 0xD1, 0xCC, 0xCA, 0x62, 0x8D,
        0x2D, 0xE5, 0x8F, 0x0B, 0xA6, 0x50, 0xD2, 0xA8, 0x50, 0xC3, 0x28, 0xEA,
        0xF5, 0xAB, 0x25, 0x87, 0x8A, 0x9A, 0x96, 0x1C, 0xA9, 0x67, 0xB8, 0x3F,
        0x0C, 0xD5, 0xF7, 0xF9, 0x52, 0x13, 0x2F, 0xC2, 0x1B, 0xD5, 0x70, 0x70,
        0xF0, 0x8F, 0xC0, 0x12, 0xCA, 0x06, 0xCB, 0x9A, 0xE1, 0xD9, 0xCA, 0x33,
        0x7A, 0x77, 0xD6, 0xF8, 0xEC, 0xB9, 0xF1, 0x68, 0x44, 0x42, 0x48, 0x13,
        0xD2, 0xC0, 0xC2, 0xA4, 0xAE, 0x5E, 0x60, 0xFE, 0xB6, 0xA6, 0x05, 0xFC,
        0xB4, 0xDD, 0x07, 0x59, 0x02, 0xD4, 0x59, 0x18, 0x98, 0x63, 0xF5, 0xA5,
        0x63, 0xE0, 0x90, 0x0C, 0x7D, 0x5D, 0xB2, 0x06, 0x7A, 0xF3, 0x85, 0xEA,
        0xEB, 0xD4, 0x03, 0xAE, 0x5E, 0x84, 0x3E, 0x5F, 0xFF, 0x15, 0xED, 0x69,
        0xBC, 0xF9, 0x39, 0x36, 0x72, 0x75, 0xCF, 0x77, 0x52, 0x4D, 0xF3, 0xC9,
        0x90, 0x2C, 0xB9, 0x3D, 0xE5, 0xC9, 0x23, 0x53, 0x3F, 0x1F, 0x24, 0x98,
        0x21, 0x5C, 0x07, 0x99, 0x29, 0xBD, 0xC6, 0x3A, 0xEC, 0xE7, 0x6E, 0x86,
        0x3A, 0x6B, 0x97, 0x74, 0x63, 0x33, 0xBD, 0x68, 0x18, 0x31, 0xF0, 0x78,
        0x8D, 0x76, 0xBF, 0xFC, 0x9E, 0x8E, 0x5D, 0x2A, 0x86, 0xA7, 0x4D, 0x90,
        0xDC, 0x27, 0x1A, 0x39
};

static const unsigned char TA1_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA2_DN[] = {
        0x30, 0x81, 0x85, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
        0x13, 0x02, 0x47, 0x42, 0x31, 0x1B, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04,
        0x08, 0x13, 0x12, 0x47, 0x72, 0x65, 0x61, 0x74, 0x65, 0x72, 0x20, 0x4D,
        0x61, 0x6E, 0x63, 0x68, 0x65, 0x73, 0x74, 0x65, 0x72, 0x31, 0x10, 0x30,
        0x0E, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x07, 0x53, 0x61, 0x6C, 0x66,
        0x6F, 0x72, 0x64, 0x31, 0x1A, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x11, 0x43, 0x4F, 0x4D, 0x4F, 0x44, 0x4F, 0x20, 0x43, 0x41, 0x20,
        0x4C, 0x69, 0x6D, 0x69, 0x74, 0x65, 0x64, 0x31, 0x2B, 0x30, 0x29, 0x06,
        0x03, 0x55, 0x04, 0x03, 0x13, 0x22, 0x43, 0x4F, 0x4D, 0x4F, 0x44, 0x4F,
        0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69,
        0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F,
        0x72, 0x69, 0x74, 0x79
};

static const unsigned char TA2_RSA_N[] = {
        0x91, 0xE8, 0x54, 0x92, 0xD2, 0x0A, 0x56, 0xB1, 0xAC, 0x0D, 0x24, 0xDD,
        0xC5, 0xCF, 0x44, 0x67, 0x74, 0x99, 0x2B, 0x37, 0xA3, 0x7D, 0x23, 0x70,
        0x00, 0x71, 0xBC, 0x53, 0xDF, 0xC4, 0xFA, 0x2A, 0x12, 0x8F, 0x4B, 0x7F,
        0x10, 0x56, 0xBD, 0x9F, 0x70, 0x72, 0xB7, 0x61, 0x7F, 0xC9, 0x4B, 0x0F,
        0x17, 0xA7, 0x3D, 0xE3, 0xB0, 0x04, 0x61, 0xEE, 0xFF, 0x11, 0x97, 0xC7,
        0xF4, 0x86, 0x3E, 0x0A, 0xFA, 0x3E, 0x5C, 0xF9, 0x93, 0xE6, 0x34, 0x7A,
        0xD9, 0x14, 0x6B, 0xE7, 0x9C, 0xB3, 0x85, 0xA0, 0x82, 0x7A, 0x76, 0xAF,
        0x71, 0x90, 0xD7, 0xEC, 0xFD, 0x0D, 0xFA, 0x9C, 0x6C, 0xFA, 0xDF, 0xB0,
        0x82, 0xF4, 0x14, 0x7E, 0xF9, 0xBE, 0xC4, 0xA6, 0x2F, 0x4F, 0x7F, 0x99,
        0x7F, 0xB5, 0xFC, 0x67, 0x43, 0x72, 0xBD, 0x0C, 0x00, 0xD6, 0x89, 0xEB,
        0x6B, 0x2C, 0xD3, 0xED, 0x8F, 0x98, 0x1C, 0x14, 0xAB, 0x7E, 0xE5, 0xE3,
        0x6E, 0xFC, 0xD8, 0xA8, 0xE4, 0x92, 0x24, 0xDA, 0x43, 0x6B, 0x62, 0xB8,
        0x55, 0xFD, 0xEA, 0xC1, 0xBC, 0x6C, 0xB6, 0x8B, 0xF3, 0x0E, 0x8D, 0x9A,
        0xE4, 0x9B, 0x6C, 0x69, 0x99, 0xF8, 0x78, 0x48, 0x30, 0x45, 0xD5, 0xAD,
        0xE1, 0x0D, 0x3C, 0x45, 0x60, 0xFC, 0x32, 0x96, 0x51, 0x27, 0xBC, 0x67,
        0xC3, 0xCA, 0x2E, 0xB6, 0x6B, 0xEA, 0x46, 0xC7, 0xC7, 0x20, 0xA0, 0xB1,
        0x1F, 0x65, 0xDE, 0x48, 0x08, 0xBA, 0xA4, 0x4E, 0xA9, 0xF2, 0x83, 0x46,
        0x37, 0x84, 0xEB, 0xE8, 0xCC, 0x81, 0x48, 0x43, 0x67, 0x4E, 0x72, 0x2A,
        0x9B, 0x5C, 0xBD, 0x4C, 0x1B, 0x28, 0x8A, 0x5C, 0x22, 0x7B, 0xB4, 0xAB,
        0x98, 0xD9, 0xEE, 0xE0, 0x51, 0x83, 0xC3, 0x09, 0x46, 0x4E, 0x6D, 0x3E,
        0x99, 0xFA, 0x95, 0x17, 0xDA, 0x7C, 0x33, 0x57, 0x41, 0x3C, 0x8D, 0x51,
        0xED, 0x0B, 0xB6, 0x5C, 0xAF, 0x2C, 0x63, 0x1A, 0xDF, 0x57, 0xC8, 0x3F,
        0xBC, 0xE9, 0x5D, 0xC4, 0x9B, 0xAF, 0x45, 0x99, 0xE2, 0xA3, 0x5A, 0x24,
        0xB4, 0xBA, 0xA9, 0x56, 0x3D, 0xCF, 0x6F, 0xAA, 0xFF, 0x49, 0x58, 0xBE,
        0xF0, 0xA8, 0xFF, 0xF4, 0xB8, 0xAD, 0xE9, 0x37, 0xFB, 0xBA, 0xB8, 0xF4,
        0x0B, 0x3A, 0xF9, 0xE8, 0x43, 0x42, 0x1E, 0x89, 0xD8, 0x84, 0xCB, 0x13,
        0xF1, 0xD9, 0xBB, 0xE1, 0x89, 0x60, 0xB8, 0x8C, 0x28, 0x56, 0xAC, 0x14,
        0x1D, 0x9C, 0x0A, 0xE7, 0x71, 0xEB, 0xCF, 0x0E, 0xDD, 0x3D, 0xA9, 0x96,
        0xA1, 0x48, 0xBD, 0x3C, 0xF7, 0xAF, 0xB5, 0x0D, 0x22, 0x4C, 0xC0, 0x11,
        0x81, 0xEC, 0x56, 0x3B, 0xF6, 0xD3, 0xA2, 0xE2, 0x5B, 0xB7, 0xB2, 0x04,
        0x22, 0x52, 0x95, 0x80, 0x93, 0x69, 0xE8, 0x8E, 0x4C, 0x65, 0xF1, 0x91,
        0x03, 0x2D, 0x70, 0x74, 0x02, 0xEA, 0x8B, 0x67, 0x15, 0x29, 0x69, 0x52,
        0x02, 0xBB, 0xD7, 0xDF, 0x50, 0x6A, 0x55, 0x46, 0xBF, 0xA0, 0xA3, 0x28,
        0x61, 0x7F, 0x70, 0xD0, 0xC3, 0xA2, 0xAA, 0x2C, 0x21, 0xAA, 0x47, 0xCE,
        0x28, 0x9C, 0x06, 0x45, 0x76, 0xBF, 0x82, 0x18, 0x27, 0xB4, 0xD5, 0xAE,
        0xB4, 0xCB, 0x50, 0xE6, 0x6B, 0xF4, 0x4C, 0x86, 0x71, 0x30, 0xE9, 0xA6,
        0xDF, 0x16, 0x86, 0xE0, 0xD8, 0xFF, 0x40, 0xDD, 0xFB, 0xD0, 0x42, 0x88,
        0x7F, 0xA3, 0x33, 0x3A, 0x2E, 0x5C, 0x1E, 0x41, 0x11, 0x81, 0x63, 0xCE,
        0x18, 0x71, 0x6B, 0x2B, 0xEC, 0xA6, 0x8A, 0xB7, 0x31, 0x5C, 0x3A, 0x6A,
        0x47, 0xE0, 0xC3, 0x79, 0x59, 0xD6, 0x20, 0x1A, 0xAF, 0xF2, 0x6A, 0x98,
        0xAA, 0x72, 0xBC, 0x57, 0x4A, 0xD2, 0x4B, 0x9D, 0xBB, 0x10, 0xFC, 0xB0,
        0x4C, 0x41, 0xE5, 0xED, 0x1D, 0x3D, 0x5E, 0x28, 0x9D, 0x9C, 0xCC, 0xBF,
        0xB3, 0x51, 0xDA, 0xA7, 0x47, 0xE5, 0x84, 0x53
};

static const unsigned char TA2_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA3_DN[] = {
        0x30, 0x61, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x0C, 0x44, 0x69, 0x67, 0x69, 0x43, 0x65, 0x72, 0x74, 0x20, 0x49,
        0x6E, 0x63, 0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13,
        0x10, 0x77, 0x77, 0x77, 0x2E, 0x64, 0x69, 0x67, 0x69, 0x63, 0x65, 0x72,
        0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55,
        0x04, 0x03, 0x13, 0x17, 0x44, 0x69, 0x67, 0x69, 0x43, 0x65, 0x72, 0x74,
        0x20, 0x47, 0x6C, 0x6F, 0x62, 0x61, 0x6C, 0x20, 0x52, 0x6F, 0x6F, 0x74,
        0x20, 0x43, 0x41
};

static const unsigned char TA3_RSA_N[] = {
        0xE2, 0x3B, 0xE1, 0x11, 0x72, 0xDE, 0xA8, 0xA4, 0xD3, 0xA3, 0x57, 0xAA,
        0x50, 0xA2, 0x8F, 0x0B, 0x77, 0x90, 0xC9, 0xA2, 0xA5, 0xEE, 0x12, 0xCE,
        0x96, 0x5B, 0x01, 0x09, 0x20, 0xCC, 0x01, 0x93, 0xA7, 0x4E, 0x30, 0xB7,
        0x53, 0xF7, 0x43, 0xC4, 0x69, 0x00, 0x57, 0x9D, 0xE2, 0x8D, 0x22, 0xDD,
        0x87, 0x06, 0x40, 0x00, 0x81, 0x09, 0xCE, 0xCE, 0x1B, 0x83, 0xBF, 0xDF,
        0xCD, 0x3B, 0x71, 0x46, 0xE2, 0xD6, 0x66, 0xC7, 0x05, 0xB3, 0x76, 0x27,
        0x16, 0x8F, 0x7B, 0x9E, 0x1E, 0x95, 0x7D, 0xEE, 0xB7, 0x48, 0xA3, 0x08,
        0xDA, 0xD6, 0xAF, 0x7A, 0x0C, 0x39, 0x06, 0x65, 0x7F, 0x4A, 0x5D, 0x1F,
        0xBC, 0x17, 0xF8, 0xAB, 0xBE, 0xEE, 0x28, 0xD7, 0x74, 0x7F, 0x7A, 0x78,
        0x99, 0x59, 0x85, 0x68, 0x6E, 0x5C, 0x23, 0x32, 0x4B, 0xBF, 0x4E, 0xC0,
        0xE8, 0x5A, 0x6D, 0xE3, 0x70, 0xBF, 0x77, 0x10, 0xBF, 0xFC, 0x01, 0xF6,
        0x85, 0xD9, 0xA8, 0x44, 0x10, 0x58, 0x32, 0xA9, 0x75, 0x18, 0xD5, 0xD1,
        0xA2, 0xBE, 0x47, 0xE2, 0x27, 0x6A, 0xF4, 0x9A, 0x33, 0xF8, 0x49, 0x08,
        0x60, 0x8B, 0xD4, 0x5F, 0xB4, 0x3A, 0x84, 0xBF, 0xA1, 0xAA, 0x4A, 0x4C,
        0x7D, 0x3E, 0xCF, 0x4F, 0x5F, 0x6C, 0x76, 0x5E, 0xA0, 0x4B, 0x37, 0x91,
        0x9E, 0xDC, 0x22, 0xE6, 0x6D, 0xCE, 0x14, 0x1A, 0x8E, 0x6A, 0xCB, 0xFE,
        0xCD, 0xB3, 0x14, 0x64, 0x17, 0xC7, 0x5B, 0x29, 0x9E, 0x32, 0xBF, 0xF2,
        0xEE, 0xFA, 0xD3, 0x0B, 0x42, 0xD4, 0xAB, 0xB7, 0x41, 0x32, 0xDA, 0x0C,
        0xD4, 0xEF, 0xF8, 0x81, 0xD5, 0xBB, 0x8D, 0x58, 0x3F, 0xB5, 0x1B, 0xE8,
        0x49, 0x28, 0xA2, 0x70, 0xDA, 0x31, 0x04, 0xDD, 0xF7, 0xB2, 0x16, 0xF2,
        0x4C, 0x0A, 0x4E, 0x07, 0xA8, 0xED, 0x4A, 0x3D, 0x5E, 0xB5, 0x7F, 0xA3,
        0x90, 0xC3, 0xAF, 0x27
};

static const unsigned char TA3_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA4_DN[] = {
        0x30, 0x61, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x0C, 0x44, 0x69, 0x67, 0x69, 0x43, 0x65, 0x72, 0x74, 0x20, 0x49,
        0x6E, 0x63, 0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13,
        0x10, 0x77, 0x77, 0x77, 0x2E, 0x64, 0x69, 0x67, 0x69, 0x63, 0x65, 0x72,
        0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x20, 0x30, 0x1E, 0x06, 0x03, 0x55,
        0x04, 0x03, 0x13, 0x17, 0x44, 0x69, 0x67, 0x69, 0x43, 0x65, 0x72, 0x74,
        0x20, 0x47, 0x6C, 0x6F, 0x62, 0x61, 0x6C, 0x20, 0x52, 0x6F, 0x6F, 0x74,
        0x20, 0x47, 0x32
};

static const unsigned char TA4_RSA_N[] = {
        0xBB, 0x37, 0xCD, 0x34, 0xDC, 0x7B, 0x6B, 0xC9, 0xB2, 0x68, 0x90, 0xAD,
        0x4A, 0x75, 0xFF, 0x46, 0xBA, 0x21, 0x0A, 0x08, 0x8D, 0xF5, 0x19, 0x54,
        0xC9, 0xFB, 0x88, 0xDB, 0xF3, 0xAE, 0xF2, 0x3A, 0x89, 0x91, 0x3C, 0x7A,
        0xE6, 0xAB, 0x06, 0x1A, 0x6B, 0xCF, 0xAC, 0x2D, 0xE8, 0x5E, 0x09, 0x24,
        0x44, 0xBA, 0x62, 0x9A, 0x7E, 0xD6, 0xA3, 0xA8, 0x7E, 0xE0, 0x54, 0x75,
        0x20, 0x05, 0xAC, 0x50, 0xB7, 0x9C, 0x63, 0x1A, 0x6C, 0x30, 0xDC, 0xDA,
        0x1F, 0x19, 0xB1, 0xD7, 0x1E, 0xDE, 0xFD, 0xD7, 0xE0, 0xCB, 0x94, 0x83,
        0x37, 0xAE, 0xEC, 0x1F, 0x43, 0x4E, 0xDD, 0x7B, 0x2C, 0xD2, 0xBD, 0x2E,
        0xA5, 0x2F, 0xE4, 0xA9, 0xB8, 0xAD, 0x3A, 0xD4, 0x99, 0xA4, 0xB6, 0x25,
        0xE9, 0x9B, 0x6B, 0x00, 0x60, 0x92, 0x60, 0xFF, 0x4F, 0x21, 0x49, 0x18,
        0xF7, 0x67, 0x90, 0xAB, 0x61, 0x06, 0x9C, 0x8F, 0xF2, 0xBA, 0xE9, 0xB4,
        0xE9, 0x92, 0x32, 0x6B, 0xB5, 0xF3, 0x57, 0xE8, 0x5D, 0x1B, 0xCD, 0x8C,
        0x1D, 0xAB, 0x95, 0x04, 0x95, 0x49, 0xF3, 0x35, 0x2D, 0x96, 0xE3, 0x49,
        0x6D, 0xDD, 0x77, 0xE3, 0xFB, 0x49, 0x4B, 0xB4, 0xAC, 0x55, 0x07, 0xA9,
        0x8F, 0x95, 0xB3, 0xB4, 0x23, 0xBB, 0x4C, 0x6D, 0x45, 0xF0, 0xF6, 0xA9,
        0xB2, 0x95, 0x30, 0xB4, 0xFD, 0x4C, 0x55, 0x8C, 0x27, 0x4A, 0x57, 0x14,
        0x7C, 0x82, 0x9D, 0xCD, 0x73, 0x92, 0xD3, 0x16, 0x4A, 0x06, 0x0C, 0x8C,
        0x50, 0xD1, 0x8F, 0x1E, 0x09, 0xBE, 0x17, 0xA1, 0xE6, 0x21, 0xCA, 0xFD,
        0x83, 0xE5, 0x10, 0xBC, 0x83, 0xA5, 0x0A, 0xC4, 0x67, 0x28, 0xF6, 0x73,
        0x14, 0x14, 0x3D, 0x46, 0x76, 0xC3, 0x87, 0x14, 0x89, 0x21, 0x34, 0x4D,
        0xAF, 0x0F, 0x45, 0x0C, 0xA6, 0x49, 0xA1, 0xBA, 0xBB, 0x9C, 0xC5, 0xB1,
        0x33, 0x83, 0x29, 0x85
};

static const unsigned char TA4_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA5_DN[] = {
        0x30, 0x6C, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x0C, 0x44, 0x69, 0x67, 0x69, 0x43, 0x65, 0x72, 0x74, 0x20, 0x49,
        0x6E, 0x63, 0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13,
        0x10, 0x77, 0x77, 0x77, 0x2E, 0x64, 0x69, 0x67, 0x69, 0x63, 0x65, 0x72,
        0x74, 0x2E, 0x63, 0x6F, 0x6D, 0x31, 0x2B, 0x30, 0x29, 0x06, 0x03, 0x55,
        0x04, 0x03, 0x13, 0x22, 0x44, 0x69, 0x67, 0x69, 0x43, 0x65, 0x72, 0x74,
        0x20, 0x48, 0x69, 0x67, 0x68, 0x20, 0x41, 0x73, 0x73, 0x75, 0x72, 0x61,
        0x6E, 0x63, 0x65, 0x20, 0x45, 0x56, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20,
        0x43, 0x41
};

static const unsigned char TA5_RSA_N[] = {
        0xC6, 0xCC, 0xE5, 0x73, 0xE6, 0xFB, 0xD4, 0xBB, 0xE5, 0x2D, 0x2D, 0x32,
        0xA6, 0xDF, 0xE5, 0x81, 0x3F, 0xC9, 0xCD, 0x25, 0x49, 0xB6, 0x71, 0x2A,
        0xC3, 0xD5, 0x94, 0x34, 0x67, 0xA2, 0x0A, 0x1C, 0xB0, 0x5F, 0x69, 0xA6,
        0x40, 0xB1, 0xC4, 0xB7, 0xB2, 0x8F, 0xD0, 0x98, 0xA4, 0xA9, 0x41, 0x59,
        0x3A, 0xD3, 0xDC, 0x94, 0xD6, 0x3C, 0xDB, 0x74, 0x38, 0xA4, 0x4A, 0xCC,
        0x4D, 0x25, 0x82, 0xF7, 0x4A, 0xA5, 0x53, 0x12, 0x38, 0xEE, 0xF3, 0x49,
        0x6D, 0x71, 0x91, 0x7E, 0x63, 0xB6, 0xAB, 0xA6, 0x5F, 0xC3, 0xA4, 0x84,
        0xF8, 0x4F, 0x62, 0x51, 0xBE, 0xF8, 0xC5, 0xEC, 0xDB, 0x38, 0x92, 0xE3,
        0x06, 0xE5, 0x08, 0x91, 0x0C, 0xC4, 0x28, 0x41, 0x55, 0xFB, 0xCB, 0x5A,
        0x89, 0x15, 0x7E, 0x71, 0xE8, 0x35, 0xBF, 0x4D, 0x72, 0x09, 0x3D, 0xBE,
        0x3A, 0x38, 0x50, 0x5B, 0x77, 0x31, 0x1B, 0x8D, 0xB3, 0xC7, 0x24, 0x45,
        0x9A, 0xA7, 0xAC, 0x6D, 0x00, 0x14, 0x5A, 0x04, 0xB7, 0xBA, 0x13, 0xEB,
        0x51, 0x0A, 0x98, 0x41, 0x41, 0x22, 0x4E, 0x65, 0x61, 0x87, 0x81, 0x41,
        0x50, 0xA6, 0x79, 0x5C, 0x89, 0xDE, 0x19, 0x4A, 0x57, 0xD5, 0x2E, 0xE6,
        0x5D, 0x1C, 0x53, 0x2C, 0x7E, 0x98, 0xCD, 0x1A, 0x06, 0x16, 0xA4, 0x68,
        0x73, 0xD0, 0x34, 0x04, 0x13, 0x5C, 0xA1, 0x71, 0xD3, 0x5A, 0x7C, 0x55,
        0xDB, 0x5E, 0x64, 0xE1, 0x37, 0x87, 0x30, 0x56, 0x04, 0xE5, 0x11, 0xB4,
        0x29, 0x80, 0x12, 0xF1, 0x79, 0x39, 0x88, 0xA2, 0x02, 0x11, 0x7C, 0x27,
        0x66, 0xB7, 0x88, 0xB7, 0x78, 0xF2, 0xCA, 0x0A, 0xA8, 0x38, 0xAB, 0x0A,
        0x64, 0xC2, 0xBF, 0x66, 0x5D, 0x95, 0x84, 0xC1, 0xA1, 0x25, 0x1E, 0x87,
        0x5D, 0x1A, 0x50, 0x0B, 0x20, 0x12, 0xCC, 0x41, 0xBB, 0x6E, 0x0B, 0x51,
        0x38, 0xB8, 0x4B, 0xCB
};

static const unsigned char TA5_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA6_DN[] = {
        0x30, 0x81, 0xB0, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
        0x13, 0x02, 0x55, 0x53, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04,
        0x0A, 0x13, 0x0D, 0x45, 0x6E, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2C, 0x20,
        0x49, 0x6E, 0x63, 0x2E, 0x31, 0x39, 0x30, 0x37, 0x06, 0x03, 0x55, 0x04,
        0x0B, 0x13, 0x30, 0x77, 0x77, 0x77, 0x2E, 0x65, 0x6E, 0x74, 0x72, 0x75,
        0x73, 0x74, 0x2E, 0x6E, 0x65, 0x74, 0x2F, 0x43, 0x50, 0x53, 0x20, 0x69,
        0x73, 0x20, 0x69, 0x6E, 0x63, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74,
        0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x72, 0x65, 0x66, 0x65, 0x72, 0x65,
        0x6E, 0x63, 0x65, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55, 0x04, 0x0B,
        0x13, 0x16, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x30, 0x36, 0x20, 0x45,
        0x6E, 0x74, 0x72, 0x75, 0x73, 0x74, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E,
        0x31, 0x2D, 0x30, 0x2B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x24, 0x45,
        0x6E, 0x74, 0x72, 0x75, 0x73, 0x74, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20,
        0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F,
        0x6E, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79
};

static const unsigned char TA6_RSA_N[] = {
        0xB6, 0x95, 0xB6, 0x43, 0x42, 0xFA, 0xC6, 0x6D, 0x2A, 0x6F, 0x48, 0xDF,
        0x94, 0x4C, 0x39, 0x57, 0x05, 0xEE, 0xC3, 0x79, 0x11, 0x41, 0x68, 0x36,
        0xED, 0xEC, 0xFE, 0x9A, 0x01, 0x8F, 0xA1, 0x38, 0x28, 0xFC, 0xF7, 0x10,
        0x46, 0x66, 0x2E, 0x4D, 0x1E, 0x1A, 0xB1, 0x1A, 0x4E, 0xC6, 0xD1, 0xC0,
        0x95, 0x88, 0xB0, 0xC9, 0xFF, 0x31, 0x8B, 0x33, 0x03, 0xDB, 0xB7, 0x83,
        0x7B, 0x3E, 0x20, 0x84, 0x5E, 0xED, 0xB2, 0x56, 0x28, 0xA7, 0xF8, 0xE0,
        0xB9, 0x40, 0x71, 0x37, 0xC5, 0xCB, 0x47, 0x0E, 0x97, 0x2A, 0x68, 0xC0,
        0x22, 0x95, 0x62, 0x15, 0xDB, 0x47, 0xD9, 0xF5, 0xD0, 0x2B, 0xFF, 0x82,
        0x4B, 0xC9, 0xAD, 0x3E, 0xDE, 0x4C, 0xDB, 0x90, 0x80, 0x50, 0x3F, 0x09,
        0x8A, 0x84, 0x00, 0xEC, 0x30, 0x0A, 0x3D, 0x18, 0xCD, 0xFB, 0xFD, 0x2A,
        0x59, 0x9A, 0x23, 0x95, 0x17, 0x2C, 0x45, 0x9E, 0x1F, 0x6E, 0x43, 0x79,
        0x6D, 0x0C, 0x5C, 0x98, 0xFE, 0x48, 0xA7, 0xC5, 0x23, 0x47, 0x5C, 0x5E,
        0xFD, 0x6E, 0xE7, 0x1E, 0xB4, 0xF6, 0x68, 0x45, 0xD1, 0x86, 0x83, 0x5B,
        0xA2, 0x8A, 0x8D, 0xB1, 0xE3, 0x29, 0x80, 0xFE, 0x25, 0x71, 0x88, 0xAD,
        0xBE, 0xBC, 0x8F, 0xAC, 0x52, 0x96, 0x4B, 0xAA, 0x51, 0x8D, 0xE4, 0x13,
        0x31, 0x19, 0xE8, 0x4E, 0x4D, 0x9F, 0xDB, 0xAC, 0xB3, 0x6A, 0xD5, 0xBC,
        0x39, 0x54, 0x71, 0xCA, 0x7A, 0x7A, 0x7F, 0x90, 0xDD, 0x7D, 0x1D, 0x80,
        0xD9, 0x81, 0xBB, 0x59, 0x26, 0xC2, 0x11, 0xFE, 0xE6, 0x93, 0xE2, 0xF7,
        0x80, 0xE4, 0x65, 0xFB, 0x34, 0x37, 0x0E, 0x29, 0x80, 0x70, 0x4D, 0xAF,
        0x38, 0x86, 0x2E, 0x9E, 0x7F, 0x57, 0xAF, 0x9E, 0x17, 0xAE, 0xEB, 0x1C,
        0xCB, 0x28, 0x21, 0x5F, 0xB6, 0x1C, 0xD8, 0xE7, 0xA2, 0x04, 0x22, 0xF9,
        0xD3, 0xDA, 0xD8, 0xCB
};

static const unsigned char TA6_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA7_DN[] = {
        0x30, 0x81, 0x98, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
        0x13, 0x02, 0x55, 0x53, 0x31, 0x16, 0x30, 0x14, 0x06, 0x03, 0x55, 0x04,
        0x0A, 0x13, 0x0D, 0x47, 0x65, 0x6F, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20,
        0x49, 0x6E, 0x63, 0x2E, 0x31, 0x39, 0x30, 0x37, 0x06, 0x03, 0x55, 0x04,
        0x0B, 0x13, 0x30, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x30, 0x38, 0x20,
        0x47, 0x65, 0x6F, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x49, 0x6E, 0x63,
        0x2E, 0x20, 0x2D, 0x20, 0x46, 0x6F, 0x72, 0x20, 0x61, 0x75, 0x74, 0x68,
        0x6F, 0x72, 0x69, 0x7A, 0x65, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6F,
        0x6E, 0x6C, 0x79, 0x31, 0x36, 0x30, 0x34, 0x06, 0x03, 0x55, 0x04, 0x03,
        0x13, 0x2D, 0x47, 0x65, 0x6F, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x50,
        0x72, 0x69, 0x6D, 0x61, 0x72, 0x79, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69,
        0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x41, 0x75, 0x74,
        0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x20, 0x2D, 0x20, 0x47, 0x33
};

static const unsigned char TA7_RSA_N[] = {
        0xDC, 0xE2, 0x5E, 0x62, 0x58, 0x1D, 0x33, 0x57, 0x39, 0x32, 0x33, 0xFA,
        0xEB, 0xCB, 0x87, 0x8C, 0xA7, 0xD4, 0x4A, 0xDD, 0x06, 0x88, 0xEA, 0x64,
        0x8E, 0x31, 0x98, 0xA5, 0x38, 0x90, 0x1E, 0x98, 0xCF, 0x2E, 0x63, 0x2B,
        0xF0, 0x46, 0xBC, 0x44, 0xB2, 0x89, 0xA1, 0xC0, 0x28, 0x0C, 0x49, 0x70,
        0x21, 0x95, 0x9F, 0x64, 0xC0, 0xA6, 0x93, 0x12, 0x02, 0x65, 0x26, 0x86,
        0xC6, 0xA5, 0x89, 0xF0, 0xFA, 0xD7, 0x84, 0xA0, 0x70, 0xAF, 0x4F, 0x1A,
        0x97, 0x3F, 0x06, 0x44, 0xD5, 0xC9, 0xEB, 0x72, 0x10, 0x7D, 0xE4, 0x31,
        0x28, 0xFB, 0x1C, 0x61, 0xE6, 0x28, 0x07, 0x44, 0x73, 0x92, 0x22, 0x69,
        0xA7, 0x03, 0x88, 0x6C, 0x9D, 0x63, 0xC8, 0x52, 0xDA, 0x98, 0x27, 0xE7,
        0x08, 0x4C, 0x70, 0x3E, 0xB4, 0xC9, 0x12, 0xC1, 0xC5, 0x67, 0x83, 0x5D,
        0x33, 0xF3, 0x03, 0x11, 0xEC, 0x6A, 0xD0, 0x53, 0xE2, 0xD1, 0xBA, 0x36,
        0x60, 0x94, 0x80, 0xBB, 0x61, 0x63, 0x6C, 0x5B, 0x17, 0x7E, 0xDF, 0x40,
        0x94, 0x1E, 0xAB, 0x0D, 0xC2, 0x21, 0x28, 0x70, 0x88, 0xFF, 0xD6, 0x26,
        0x6C, 0x6C, 0x60, 0x04, 0x25, 0x4E, 0x55, 0x7E, 0x7D, 0xEF, 0xBF, 0x94,
        0x48, 0xDE, 0xB7, 0x1D, 0xDD, 0x70, 0x8D, 0x05, 0x5F, 0x88, 0xA5, 0x9B,
        0xF2, 0xC2, 0xEE, 0xEA, 0xD1, 0x40, 0x41, 0x6D, 0x62, 0x38, 0x1D, 0x56,
        0x06, 0xC5, 0x03, 0x47, 0x51, 0x20, 0x19, 0xFC, 0x7B, 0x10, 0x0B, 0x0E,
        0x62, 0xAE, 0x76, 0x55, 0xBF, 0x5F, 0x77, 0xBE, 0x3E, 0x49, 0x01, 0x53,
        0x3D, 0x98, 0x25, 0x03, 0x76, 0x24, 0x5A, 0x1D, 0xB4, 0xDB, 0x89, 0xEA,
        0x79, 0xE5, 0xB6, 0xB3, 0x3B, 0x3F, 0xBA, 0x4C, 0x28, 0x41, 0x7F, 0x06,
        0xAC, 0x6A, 0x8E, 0xC1, 0xD0, 0xF6, 0x05, 0x1D, 0x7D, 0xE6, 0x42, 0x86,
        0xE3, 0xA5, 0xD5, 0x47
};

static const unsigned char TA7_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA8_DN[] = {
        0x30, 0x57, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x42, 0x45, 0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x10, 0x47, 0x6C, 0x6F, 0x62, 0x61, 0x6C, 0x53, 0x69, 0x67, 0x6E,
        0x20, 0x6E, 0x76, 0x2D, 0x73, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03,
        0x55, 0x04, 0x0B, 0x13, 0x07, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x41,
        0x31, 0x1B, 0x30, 0x19, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x12, 0x47,
        0x6C, 0x6F, 0x62, 0x61, 0x6C, 0x53, 0x69, 0x67, 0x6E, 0x20, 0x52, 0x6F,
        0x6F, 0x74, 0x20, 0x43, 0x41
};

static const unsigned char TA8_RSA_N[] = {
        0xDA, 0x0E, 0xE6, 0x99, 0x8D, 0xCE, 0xA3, 0xE3, 0x4F, 0x8A, 0x7E, 0xFB,
        0xF1, 0x8B, 0x83, 0x25, 0x6B, 0xEA, 0x48, 0x1F, 0xF1, 0x2A, 0xB0, 0xB9,
        0x95, 0x11, 0x04, 0xBD, 0xF0, 0x63, 0xD1, 0xE2, 0x67, 0x66, 0xCF, 0x1C,
        0xDD, 0xCF, 0x1B, 0x48, 0x2B, 0xEE, 0x8D, 0x89, 0x8E, 0x9A, 0xAF, 0x29,
        0x80, 0x65, 0xAB, 0xE9, 0xC7, 0x2D, 0x12, 0xCB, 0xAB, 0x1C, 0x4C, 0x70,
        0x07, 0xA1, 0x3D, 0x0A, 0x30, 0xCD, 0x15, 0x8D, 0x4F, 0xF8, 0xDD, 0xD4,
        0x8C, 0x50, 0x15, 0x1C, 0xEF, 0x50, 0xEE, 0xC4, 0x2E, 0xF7, 0xFC, 0xE9,
        0x52, 0xF2, 0x91, 0x7D, 0xE0, 0x6D, 0xD5, 0x35, 0x30, 0x8E, 0x5E, 0x43,
        0x73, 0xF2, 0x41, 0xE9, 0xD5, 0x6A, 0xE3, 0xB2, 0x89, 0x3A, 0x56, 0x39,
        0x38, 0x6F, 0x06, 0x3C, 0x88, 0x69, 0x5B, 0x2A, 0x4D, 0xC5, 0xA7, 0x54,
        0xB8, 0x6C, 0x89, 0xCC, 0x9B, 0xF9, 0x3C, 0xCA, 0xE5, 0xFD, 0x89, 0xF5,
        0x12, 0x3C, 0x92, 0x78, 0x96, 0xD6, 0xDC, 0x74, 0x6E, 0x93, 0x44, 0x61,
        0xD1, 0x8D, 0xC7, 0x46, 0xB2, 0x75, 0x0E, 0x86, 0xE8, 0x19, 0x8A, 0xD5,
        0x6D, 0x6C, 0xD5, 0x78, 0x16, 0x95, 0xA2, 0xE9, 0xC8, 0x0A, 0x38, 0xEB,
        0xF2, 0x24, 0x13, 0x4F, 0x73, 0x54, 0x93, 0x13, 0x85, 0x3A, 0x1B, 0xBC,
        0x1E, 0x34, 0xB5, 0x8B, 0x05, 0x8C, 0xB9, 0x77, 0x8B, 0xB1, 0xDB, 0x1F,
        0x20, 0x91, 0xAB, 0x09, 0x53, 0x6E, 0x90, 0xCE, 0x7B, 0x37, 0x74, 0xB9,
        0x70, 0x47, 0x91, 0x22, 0x51, 0x63, 0x16, 0x79, 0xAE, 0xB1, 0xAE, 0x41,
        0x26, 0x08, 0xC8, 0x19, 0x2B, 0xD1, 0x46, 0xAA, 0x48, 0xD6, 0x64, 0x2A,
        0xD7, 0x83, 0x34, 0xFF, 0x2C, 0x2A, 0xC1, 0x6C, 0x19, 0x43, 0x4A, 0x07,
        0x85, 0xE7, 0xD3, 0x7C, 0xF6, 0x21, 0x68, 0xEF, 0xEA, 0xF2, 0x52, 0x9F,
        0x7F, 0x93, 0x90, 0xCF
};

static const unsigned char TA8_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA9_DN[] = {
        0x30, 0x81, 0x83, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
        0x13, 0x02, 0x55, 0x53, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x03, 0x55, 0x04,
        0x08, 0x13, 0x07, 0x41, 0x72, 0x69, 0x7A, 0x6F, 0x6E, 0x61, 0x31, 0x13,
        0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13, 0x0A, 0x53, 0x63, 0x6F,
        0x74, 0x74, 0x73, 0x64, 0x61, 0x6C, 0x65, 0x31, 0x1A, 0x30, 0x18, 0x06,
        0x03, 0x55, 0x04, 0x0A, 0x13, 0x11, 0x47, 0x6F, 0x44, 0x61, 0x64, 0x64,
        0x79, 0x2E, 0x63, 0x6F, 0x6D, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E, 0x31,
        0x31, 0x30, 0x2F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x28, 0x47, 0x6F,
        0x20, 0x44, 0x61, 0x64, 0x64, 0x79, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20,
        0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x20,
        0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x20, 0x2D, 0x20,
        0x47, 0x32
};

static const unsigned char TA9_RSA_N[] = {
        0xBF, 0x71, 0x62, 0x08, 0xF1, 0xFA, 0x59, 0x34, 0xF7, 0x1B, 0xC9, 0x18,
        0xA3, 0xF7, 0x80, 0x49, 0x58, 0xE9, 0x22, 0x83, 0x13, 0xA6, 0xC5, 0x20,
        0x43, 0x01, 0x3B, 0x84, 0xF1, 0xE6, 0x85, 0x49, 0x9F, 0x27, 0xEA, 0xF6,
        0x84, 0x1B, 0x4E, 0xA0, 0xB4, 0xDB, 0x70, 0x98, 0xC7, 0x32, 0x01, 0xB1,
        0x05, 0x3E, 0x07, 0x4E, 0xEE, 0xF4, 0xFA, 0x4F, 0x2F, 0x59, 0x30, 0x22,
        0xE7, 0xAB, 0x19, 0x56, 0x6B, 0xE2, 0x80, 0x07, 0xFC, 0xF3, 0x16, 0x75,
        0x80, 0x39, 0x51, 0x7B, 0xE5, 0xF9, 0x35, 0xB6, 0x74, 0x4E, 0xA9, 0x8D,
        0x82, 0x13, 0xE4, 0xB6, 0x3F, 0xA9, 0x03, 0x83, 0xFA, 0xA2, 0xBE, 0x8A,
        0x15, 0x6A, 0x7F, 0xDE, 0x0B, 0xC3, 0xB6, 0x19, 0x14, 0x05, 0xCA, 0xEA,
        0xC3, 0xA8, 0x04, 0x94, 0x3B, 0x46, 0x7C, 0x32, 0x0D, 0xF3, 0x00, 0x66,
        0x22, 0xC8, 0x8D, 0x69, 0x6D, 0x36, 0x8C, 0x11, 0x18, 0xB7, 0xD3, 0xB2,
        0x1C, 0x60, 0xB4, 0x38, 0xFA, 0x02, 0x8C, 0xCE, 0xD3, 0xDD, 0x46, 0x07,
        0xDE, 0x0A, 0x3E, 0xEB, 0x5D, 0x7C, 0xC8, 0x7C, 0xFB, 0xB0, 0x2B, 0x53,
        0xA4, 0x92, 0x62, 0x69, 0x51, 0x25, 0x05, 0x61, 0x1A, 0x44, 0x81, 0x8C,
        0x2C, 0xA9, 0x43, 0x96, 0x23, 0xDF, 0xAC, 0x3A, 0x81, 0x9A, 0x0E, 0x29,
        0xC5, 0x1C, 0xA9, 0xE9, 0x5D, 0x1E, 0xB6, 0x9E, 0x9E, 0x30, 0x0A, 0x39,
        0xCE, 0xF1, 0x88, 0x80, 0xFB, 0x4B, 0x5D, 0xCC, 0x32, 0xEC, 0x85, 0x62,
        0x43, 0x25, 0x34, 0x02, 0x56, 0x27, 0x01, 0x91, 0xB4, 0x3B, 0x70, 0x2A,
        0x3F, 0x6E, 0xB1, 0xE8, 0x9C, 0x88, 0x01, 0x7D, 0x9F, 0xD4, 0xF9, 0xDB,
        0x53, 0x6D, 0x60, 0x9D, 0xBF, 0x2C, 0xE7, 0x58, 0xAB, 0xB8, 0x5F, 0x46,
        0xFC, 0xCE, 0xC4, 0x1B, 0x03, 0x3C, 0x09, 0xEB, 0x49, 0x31, 0x5C, 0x69,
        0x46, 0xB3, 0xE0, 0x47
};

static const unsigned char TA9_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA10_DN[] = {
        0x30, 0x4F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x29, 0x30, 0x27, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x20, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65, 0x74, 0x20, 0x53,
        0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x20, 0x52, 0x65, 0x73, 0x65,
        0x61, 0x72, 0x63, 0x68, 0x20, 0x47, 0x72, 0x6F, 0x75, 0x70, 0x31, 0x15,
        0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0C, 0x49, 0x53, 0x52,
        0x47, 0x20, 0x52, 0x6F, 0x6F, 0x74, 0x20, 0x58, 0x31
};

static const unsigned char TA10_RSA_N[] = {
        0xAD, 0xE8, 0x24, 0x73, 0xF4, 0x14, 0x37, 0xF3, 0x9B, 0x9E, 0x2B, 0x57,
        0x28, 0x1C, 0x87, 0xBE, 0xDC, 0xB7, 0xDF, 0x38, 0x90, 0x8C, 0x6E, 0x3C,
        0xE6, 0x57, 0xA0, 0x78, 0xF7, 0x75, 0xC2, 0xA2, 0xFE, 0xF5, 0x6A, 0x6E,
        0xF6, 0x00, 0x4F, 0x28, 0xDB, 0xDE, 0x68, 0x86, 0x6C, 0x44, 0x93, 0xB6,
        0xB1, 0x63, 0xFD, 0x14, 0x12, 0x6B, 0xBF, 0x1F, 0xD2, 0xEA, 0x31, 0x9B,
        0x21, 0x7E, 0xD1, 0x33, 0x3C, 0xBA, 0x48, 0xF5, 0xDD, 0x79, 0xDF, 0xB3,
        0xB8, 0xFF, 0x12, 0xF1, 0x21, 0x9A, 0x4B, 0xC1, 0x8A, 0x86, 0x71, 0x69,
        0x4A, 0x66, 0x66, 0x6C, 0x8F, 0x7E, 0x3C, 0x70, 0xBF, 0xAD, 0x29, 0x22,
        0x06, 0xF3, 0xE4, 0xC0, 0xE6, 0x80, 0xAE, 0xE2, 0x4B, 0x8F, 0xB7, 0x99,
        0x7E, 0x94, 0x03, 0x9F, 0xD3, 0x47, 0x97, 0x7C, 0x99, 0x48, 0x23, 0x53,
        0xE8, 0x38, 0xAE, 0x4F, 0x0A, 0x6F, 0x83, 0x2E, 0xD1, 0x49, 0x57, 0x8C,
        0x80, 0x74, 0xB6, 0xDA, 0x2F, 0xD0, 0x38, 0x8D, 0x7B, 0x03, 0x70, 0x21,
        0x1B, 0x75, 0xF2, 0x30, 0x3C, 0xFA, 0x8F, 0xAE, 0xDD, 0xDA, 0x63, 0xAB,
        0xEB, 0x16, 0x4F, 0xC2, 0x8E, 0x11, 0x4B, 0x7E, 0xCF, 0x0B, 0xE8, 0xFF,
        0xB5, 0x77, 0x2E, 0xF4, 0xB2, 0x7B, 0x4A, 0xE0, 0x4C, 0x12, 0x25, 0x0C,
        0x70, 0x8D, 0x03, 0x29, 0xA0, 0xE1, 0x53, 0x24, 0xEC, 0x13, 0xD9, 0xEE,
        0x19, 0xBF, 0x10, 0xB3, 0x4A, 0x8C, 0x3F, 0x89, 0xA3, 0x61, 0x51, 0xDE,
        0xAC, 0x87, 0x07, 0x94, 0xF4, 0x63, 0x71, 0xEC, 0x2E, 0xE2, 0x6F, 0x5B,
        0x98, 0x81, 0xE1, 0x89, 0x5C, 0x34, 0x79, 0x6C, 0x76, 0xEF, 0x3B, 0x90,
        0x62, 0x79, 0xE6, 0xDB, 0xA4, 0x9A, 0x2F, 0x26, 0xC5, 0xD0, 0x10, 0xE1,
        0x0E, 0xDE, 0xD9, 0x10, 0x8E, 0x16, 0xFB, 0xB7, 0xF7, 0xA8, 0xF7, 0xC7,
        0xE5, 0x02, 0x07, 0x98, 0x8F, 0x36, 0x08, 0x95, 0xE7, 0xE2, 0x37, 0x96,
        0x0D, 0x36, 0x75, 0x9E, 0xFB, 0x0E, 0x72, 0xB1, 0x1D, 0x9B, 0xBC, 0x03,
        0xF9, 0x49, 0x05, 0xD8, 0x81, 0xDD, 0x05, 0xB4, 0x2A, 0xD6, 0x41, 0xE9,
        0xAC, 0x01, 0x76, 0x95, 0x0A, 0x0F, 0xD8, 0xDF, 0xD5, 0xBD, 0x12, 0x1F,
        0x35, 0x2F, 0x28, 0x17, 0x6C, 0xD2, 0x98, 0xC1, 0xA8, 0x09, 0x64, 0x77,
        0x6E, 0x47, 0x37, 0xBA, 0xCE, 0xAC, 0x59, 0x5E, 0x68, 0x9D, 0x7F, 0x72,
        0xD6, 0x89, 0xC5, 0x06, 0x41, 0x29, 0x3E, 0x59, 0x3E, 0xDD, 0x26, 0xF5,
        0x24, 0xC9, 0x11, 0xA7, 0x5A, 0xA3, 0x4C, 0x40, 0x1F, 0x46, 0xA1, 0x99,
        0xB5, 0xA7, 0x3A, 0x51, 0x6E, 0x86, 0x3B, 0x9E, 0x7D, 0x72, 0xA7, 0x12,
        0x05, 0x78, 0x59, 0xED, 0x3E, 0x51, 0x78, 0x15, 0x0B, 0x03, 0x8F, 0x8D,
        0xD0, 0x2F, 0x05, 0xB2, 0x3E, 0x7B, 0x4A, 0x1C, 0x4B, 0x73, 0x05, 0x12,
        0xFC, 0xC6, 0xEA, 0xE0, 0x50, 0x13, 0x7C, 0x43, 0x93, 0x74, 0xB3, 0xCA,
        0x74, 0xE7, 0x8E, 0x1F, 0x01, 0x08, 0xD0, 0x30, 0xD4, 0x5B, 0x71, 0x36,
        0xB4, 0x07, 0xBA, 0xC1, 0x30, 0x30, 0x5C, 0x48, 0xB7, 0x82, 0x3B, 0x98,
        0xA6, 0x7D, 0x60, 0x8A, 0xA2, 0xA3, 0x29, 0x82, 0xCC, 0xBA, 0xBD, 0x83,
        0x04, 0x1B, 0xA2, 0x83, 0x03, 0x41, 0xA1, 0xD6, 0x05, 0xF1, 0x1B, 0xC2,
        0xB6, 0xF0, 0xA8, 0x7C, 0x86, 0x3B, 0x46, 0xA8, 0x48, 0x2A, 0x88, 0xDC,
        0x76, 0x9A, 0x76, 0xBF, 0x1F, 0x6A, 0xA5, 0x3D, 0x19, 0x8F, 0xEB, 0x38,
        0xF3, 0x64, 0xDE, 0xC8, 0x2B, 0x0D, 0x0A, 0x28, 0xFF, 0xF7, 0xDB, 0xE2,
        0x15, 0x42, 0xD4, 0x22, 0xD0, 0x27, 0x5D, 0xE1, 0x79, 0xFE, 0x18, 0xE7,
        0x70, 0x88, 0xAD, 0x4E, 0xE6, 0xD9, 0x8B, 0x3A, 0xC6, 0xDD, 0x27, 0x51,
        0x6E, 0xFF, 0xBC, 0x64, 0xF5, 0x33, 0x43, 0x4F
};

static const unsigned char TA10_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA11_DN[] = {
        0x30, 0x65, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
        0x02, 0x55, 0x53, 0x31, 0x1E, 0x30, 0x1C, 0x06, 0x03, 0x55, 0x04, 0x0A,
        0x13, 0x15, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20,
        0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x31,
        0x36, 0x30, 0x34, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x2D, 0x4D, 0x69,
        0x63, 0x72, 0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20,
        0x52, 0x6F, 0x6F, 0x74, 0x20, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69,
        0x63, 0x61, 0x74, 0x65, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6F, 0x72, 0x69,
        0x74, 0x79, 0x20, 0x32, 0x30, 0x31, 0x37
};

static const unsigned char TA11_RSA_N[] = {
        0xCA, 0x5B, 0xBE, 0x94, 0x33, 0x8C, 0x29, 0x95, 0x91, 0x16, 0x0A, 0x95,
        0xBD, 0x47, 0x62, 0xC1, 0x89, 0xF3, 0x99, 0x36, 0xDF, 0x46, 0x90, 0xC9,
        0xA5, 0xED, 0x78, 0x6A, 0x6F, 0x47, 0x91, 0x68, 0xF8, 0x27, 0x67, 0x50,
        0x33, 0x1D, 0xA1, 0xA6, 0xFB, 0xE0, 0xE5, 0x43, 0xA3, 0x84, 0x02, 0x57,
        0x01, 0x5D, 0x9C, 0x48, 0x40, 0x82, 0x53, 0x10, 0xBC, 0xBF, 0xC7, 0x3B,
        0x68, 0x90, 0xB6, 0x82, 0x2D, 0xE5, 0xF4, 0x65, 0xD0, 0xCC, 0x6D, 0x19,
        0xCC, 0x95, 0xF9, 0x7B, 0xAC, 0x4A, 0x94, 0xAD, 0x0E, 0xDE, 0x4B, 0x43,
        0x1D, 0x87, 0x07, 0x92, 0x13, 0x90, 0x80, 0x83, 0x64, 0x35, 0x39, 0x04,
        0xFC, 0xE5, 0xE9, 0x6C, 0xB3, 0xB6, 0x1F, 0x50, 0x94, 0x38, 0x65, 0x50,
        0x5C, 0x17, 0x46, 0xB9, 0xB6, 0x85, 0xB5, 0x1C, 0xB5, 0x17, 0xE8, 0xD6,
        0x45, 0x9D, 0xD8, 0xB2, 0x26, 0xB0, 0xCA, 0xC4, 0x70, 0x4A, 0xAE, 0x60,
        0xA4, 0xDD, 0xB3, 0xD9, 0xEC, 0xFC, 0x3B, 0xD5, 0x57, 0x72, 0xBC, 0x3F,
        0xC8, 0xC9, 0xB2, 0xDE, 0x4B, 0x6B, 0xF8, 0x23, 0x6C, 0x03, 0xC0, 0x05,
        0xBD, 0x95, 0xC7, 0xCD, 0x73, 0x3B, 0x66, 0x80, 0x64, 0xE3, 0x1A, 0xAC,
        0x2E, 0xF9, 0x47, 0x05, 0xF2, 0x06, 0xB6, 0x9B, 0x73, 0xF5, 0x78, 0x33,
        0x5B, 0xC7, 0xA1, 0xFB, 0x27, 0x2A, 0xA1, 0xB4, 0x9A, 0x91, 0x8C, 0x91,
        0xD3, 0x3A, 0x82, 0x3E, 0x76, 0x40, 0xB4, 0xCD, 0x52, 0x61, 0x51, 0x70,
        0x28, 0x3F, 0xC5, 0xC5, 0x5A, 0xF2, 0xC9, 0x8C, 0x49, 0xBB, 0x14, 0x5B,
        0x4D, 0xC8, 0xFF, 0x67, 0x4D, 0x4C, 0x12, 0x96, 0xAD, 0xF5, 0xFE, 0x78,
        0xA8, 0x97, 0x87, 0xD7, 0xFD, 0x5E, 0x20, 0x80, 0xDC, 0xA1, 0x4B, 0x22,
        0xFB, 0xD4, 0x89, 0xAD, 0xBA, 0xCE, 0x47, 0x97, 0x47, 0x55, 0x7B, 0x8F,
        0x45, 0xC8, 0x67, 0x28, 0x84, 0x95, 0x1C, 0x68, 0x30, 0xEF, 0xEF, 0x49,
        0xE0, 0x35, 0x7B, 0x64, 0xE7, 0x98, 0xB0, 0x94, 0xDA, 0x4D, 0x85, 0x3B,
        0x3E, 0x55, 0xC4, 0x28, 0xAF, 0x57, 0xF3, 0x9E, 0x13, 0xDB, 0x46, 0x27,
        0x9F, 0x1E, 0xA2, 0x5E, 0x44, 0x83, 0xA4, 0xA5, 0xCA, 0xD5, 0x13, 0xB3,
        0x4B, 0x3F, 0xC4, 0xE3, 0xC2, 0xE6, 0x86, 0x61, 0xA4, 0x52, 0x30, 0xB9,
        0x7A, 0x20, 0x4F, 0x6F, 0x0F, 0x38, 0x53, 0xCB, 0x33, 0x0C, 0x13, 0x2B,
        0x8F, 0xD6, 0x9A, 0xBD, 0x2A, 0xC8, 0x2D, 0xB1, 0x1C, 0x7D, 0x4B, 0x51,
        0xCA, 0x47, 0xD1, 0x48, 0x27, 0x72, 0x5D, 0x87, 0xEB, 0xD5, 0x45, 0xE6,
        0x48, 0x65, 0x9D, 0xAF, 0x52, 0x90, 0xBA, 0x5B, 0xA2, 0x18, 0x65, 0x57,
        0x12, 0x9F, 0x68, 0xB9, 0xD4, 0x15, 0x6B, 0x94, 0xC4, 0x69, 0x22, 0x98,
        0xF4, 0x33, 0xE0, 0xED, 0xF9, 0x51, 0x8E, 0x41, 0x50, 0xC9, 0x34, 0x4F,
        0x76, 0x90, 0xAC, 0xFC, 0x38, 0xC1, 0xD8, 0xE1, 0x7B, 0xB9, 0xE3, 0xE3,
        0x94, 0xE1, 0x46, 0x69, 0xCB, 0x0E, 0x0A, 0x50, 0x6B, 0x13, 0xBA, 0xAC,
        0x0F, 0x37, 0x5A, 0xB7, 0x12, 0xB5, 0x90, 0x81, 0x1E, 0x56, 0xAE, 0x57,
        0x22, 0x86, 0xD9, 0xC9, 0xD2, 0xD1, 0xD7, 0x51, 0xE3, 0xAB, 0x3B, 0xC6,
        0x55, 0xFD, 0x1E, 0x0E, 0xD3, 0x74, 0x0A, 0xD1, 0xDA, 0xAA, 0xEA, 0x69,
        0xB8, 0x97, 0x28, 0x8F, 0x48, 0xC4, 0x07, 0xF8, 0x52, 0x43, 0x3A, 0xF4,
        0xCA, 0x55, 0x35, 0x2C, 0xB0, 0xA6, 0x6A, 0xC0, 0x9C, 0xF9, 0xF2, 0x81,
        0xE1, 0x12, 0x6A, 0xC0, 0x45, 0xD9, 0x67, 0xB3, 0xCE, 0xFF, 0x23, 0xA2,
        0x89, 0x0A, 0x54, 0xD4, 0x14, 0xB9, 0x2A, 0xA8, 0xD7, 0xEC, 0xF9, 0xAB,
        0xCD, 0x25, 0x58, 0x32, 0x79, 0x8F, 0x90, 0x5B, 0x98, 0x39, 0xC4, 0x08,
        0x06, 0xC1, 0xAC, 0x7F, 0x0E, 0x3D, 0x00, 0xA5
};

static const unsigned char TA11_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const unsigned char TA12_DN[] = {
        0x30, 0x81, 0xCA, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
        0x13, 0x02, 0x55, 0x53, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04,
        0x0A, 0x13, 0x0E, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x2C,
        0x20, 0x49, 0x6E, 0x63, 0x2E, 0x31, 0x1F, 0x30, 0x1D, 0x06, 0x03, 0x55,
        0x04, 0x0B, 0x13, 0x16, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E,
        0x20, 0x54, 0x72, 0x75, 0x73, 0x74, 0x20, 0x4E, 0x65, 0x74, 0x77, 0x6F,
        0x72, 0x6B, 0x31, 0x3A, 0x30, 0x38, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13,
        0x31, 0x28, 0x63, 0x29, 0x20, 0x32, 0x30, 0x30, 0x36, 0x20, 0x56, 0x65,
        0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x2C, 0x20, 0x49, 0x6E, 0x63, 0x2E,
        0x20, 0x2D, 0x20, 0x46, 0x6F, 0x72, 0x20, 0x61, 0x75, 0x74, 0x68, 0x6F,
        0x72, 0x69, 0x7A, 0x65, 0x64, 0x20, 0x75, 0x73, 0x65, 0x20, 0x6F, 0x6E,
        0x6C, 0x79, 0x31, 0x45, 0x30, 0x43, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
        0x3C, 0x56, 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6E, 0x20, 0x43, 0x6C,
        0x61, 0x73, 0x73, 0x20, 0x33, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63,
        0x20, 0x50, 0x72, 0x69, 0x6D, 0x61, 0x72, 0x79, 0x20, 0x43, 0x65, 0x72,
        0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x41,
        0x75, 0x74, 0x68, 0x6F, 0x72, 0x69, 0x74, 0x79, 0x20, 0x2D, 0x20, 0x47,
        0x35
};

static const unsigned char TA12_RSA_N[] = {
        0xAF, 0x24, 0x08, 0x08, 0x29, 0x7A, 0x35, 0x9E, 0x60, 0x0C, 0xAA, 0xE7,
        0x4B, 0x3B, 0x4E, 0xDC, 0x7C, 0xBC, 0x3C, 0x45, 0x1C, 0xBB, 0x2B, 0xE0,
        0xFE, 0x29, 0x02, 0xF9, 0x57, 0x08, 0xA3, 0x64, 0x85, 0x15, 0x27, 0xF5,
        0xF1, 0xAD, 0xC8, 0x31, 0x89, 0x5D, 0x22, 0xE8, 0x2A, 0xAA, 0xA6, 0x42,
        0xB3, 0x8F, 0xF8, 0xB9, 0x55, 0xB7, 0xB1, 0xB7, 0x4B, 0xB3, 0xFE, 0x8F,
        0x7E, 0x07, 0x57, 0xEC, 0xEF, 0x43, 0xDB, 0x66, 0x62, 0x15, 0x61, 0xCF,
        0x60, 0x0D, 0xA4, 0xD8, 0xDE, 0xF8, 0xE0, 0xC3, 0x62, 0x08, 0x3D, 0x54,
        0x13, 0xEB, 0x49, 0xCA, 0x59, 0x54, 0x85, 0x26, 0xE5, 0x2B, 0x8F, 0x1B,
        0x9F, 0xEB, 0xF5, 0xA1, 0x91, 0xC2, 0x33, 0x49, 0xD8, 0x43, 0x63, 0x6A,
        0x52, 0x4B, 0xD2, 0x8F, 0xE8, 0x70, 0x51, 0x4D, 0xD1, 0x89, 0x69, 0x7B,
        0xC7, 0x70, 0xF6, 0xB3, 0xDC, 0x12, 0x74, 0xDB, 0x7B, 0x5D, 0x4B, 0x56,
        0xD3, 0x96, 0xBF, 0x15, 0x77, 0xA1, 0xB0, 0xF4, 0xA2, 0x25, 0xF2, 0xAF,
        0x1C, 0x92, 0x67, 0x18, 0xE5, 0xF4, 0x06, 0x04, 0xEF, 0x90, 0xB9, 0xE4,
        0x00, 0xE4, 0xDD, 0x3A, 0xB5, 0x19, 0xFF, 0x02, 0xBA, 0xF4, 0x3C, 0xEE,
        0xE0, 0x8B, 0xEB, 0x37, 0x8B, 0xEC, 0xF4, 0xD7, 0xAC, 0xF2, 0xF6, 0xF0,
        0x3D, 0xAF, 0xDD, 0x75, 0x91, 0x33, 0x19, 0x1D, 0x1C, 0x40, 0xCB, 0x74,
        0x24, 0x19, 0x21, 0x93, 0xD9, 0x14, 0xFE, 0xAC, 0x2A, 0x52, 0xC7, 0x8F,
        0xD5, 0x04, 0x49, 0xE4, 0x8D, 0x63, 0x47, 0x88, 0x3C, 0x69, 0x83, 0xCB,
        0xFE, 0x47, 0xBD, 0x2B, 0x7E, 0x4F, 0xC5, 0x95, 0xAE, 0x0E, 0x9D, 0xD4,
        0xD1, 0x43, 0xC0, 0x67, 0x73, 0xE3, 0x14, 0x08, 0x7E, 0xE5, 0x3F, 0x9F,
        0x73, 0xB8, 0x33, 0x0A, 0xCF, 0x5D, 0x3F, 0x34, 0x87, 0x96, 0x8A, 0xEE,
        0x53, 0xE8, 0x25, 0x15
};

static const unsigned char TA12_RSA_E[] = {
        0x01, 0x00, 0x01
};

static const br_x509_trust_anchor TAs[13] = {
        {
                { (unsigned char *)TA0_DN, sizeof TA0_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA0_RSA_N, sizeof TA0_RSA_N,
                                (unsigned char *)TA0_RSA_E, sizeof TA0_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA1_DN, sizeof TA1_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA1_RSA_N, sizeof TA1_RSA_N,
                                (unsigned char *)TA1_RSA_E, sizeof TA1_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA2_DN, sizeof TA2_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA2_RSA_N, sizeof TA2_RSA_N,
                                (unsigned char *)TA2_RSA_E, sizeof TA2_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA3_DN, sizeof TA3_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA3_RSA_N, sizeof TA3_RSA_N,
                                (unsigned char *)TA3_RSA_E, sizeof TA3_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA4_DN, sizeof TA4_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA4_RSA_N, sizeof TA4_RSA_N,
                                (unsigned char *)TA4_RSA_E, sizeof TA4_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA5_DN, sizeof TA5_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA5_RSA_N, sizeof TA5_RSA_N,
                                (unsigned char *)TA5_RSA_E, sizeof TA5_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA6_DN, sizeof TA6_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA6_RSA_N, sizeof TA6_RSA_N,
                                (unsigned char *)TA6_RSA_E, sizeof TA6_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA7_DN, sizeof TA7_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA7_RSA_N, sizeof TA7_RSA_N,
                                (unsigned char *)TA7_RSA_E, sizeof TA7_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA8_DN, sizeof TA8_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA8_RSA_N, sizeof TA8_RSA_N,
                                (unsigned char *)TA8_RSA_E, sizeof TA8_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA9_DN, sizeof TA9_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA9_RSA_N, sizeof TA9_RSA_N,
                                (unsigned char *)TA9_RSA_E, sizeof TA9_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA10_DN, sizeof TA10_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA10_RSA_N, sizeof TA10_RSA_N,
                                (unsigned char *)TA10_RSA_E, sizeof TA10_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA11_DN, sizeof TA11_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA11_RSA_N, sizeof TA11_RSA_N,
                                (unsigned char *)TA11_RSA_E, sizeof TA11_RSA_E,
                        } }
                }
        },
        {
                { (unsigned char *)TA12_DN, sizeof TA12_DN },
                BR_X509_TA_CA,
                {
                        BR_KEYTYPE_RSA,
                        { .rsa = {
                                (unsigned char *)TA12_RSA_N, sizeof TA12_RSA_N,
                                (unsigned char *)TA12_RSA_E, sizeof TA12_RSA_E,
                        } }
                }
        }
};

#define TAs_NUM   13

#endif 
