/*!
 * @file ws_uart_drv.h
 *
 * Base implementation for UART device drivers
 *
 * Adafruit invests time and resources providing this open source code,
 * please support Adafruit and open-source hardware by purchasing
 * products from Adafruit!
 *
 * Copyright (c) Brent Rubell 2023 for Adafruit Industries.
 *
 * MIT license, all text here must be included in any redistribution.
 *
 */

#ifndef WS_UART_DRV_H
#define WS_UART_DRV_H
#include "Wippersnapper.h"
#include <Adafruit_Sensor.h>

// ESP8266 platform uses SoftwareSerial
// so does RP2040 (note that this has differences from the pure softwareserial
// library, see: https://arduino-pico.readthedocs.io/en/latest/piouart.html)
#if defined(ARDUINO_ARCH_ESP8266) || defined(ARDUINO_ARCH_RP2040)
#define USE_SW_UART
#include <SoftwareSerial.h>
#else
#include <HardwareSerial.h>
#endif

/**************************************************************************/
/*!
    @brief  Base class for UART Device Drivers.
*/
/**************************************************************************/
class ws_uart_drv {
public:
#ifdef USE_SW_UART
  /*******************************************************************************/
  /*!
      @brief    Initializes a UART device driver.
      @param    swSerial
                Pointer to an instance of a SoftwareSerial object.
      @param    interval
                How often the UART device will be polled, in milliseconds.
  */
  /*******************************************************************************/
  ws_uart_drv(SoftwareSerial *swSerial, int32_t interval){};
#else
  /*******************************************************************************/
  /*!
      @brief    Initializes a UART device driver.
      @param    hwSerial
                Pointer to an instance of a HardwareSerial object.
      @param    interval
                How often the UART device will be polled, in milliseconds.
  */
  /*******************************************************************************/
  ws_uart_drv(HardwareSerial *hwSerial, int32_t interval){};
#endif
  ~ws_uart_drv(void) {}

  /*******************************************************************************/
  /*!
      @brief   Checks if the UART device is ready to be polled at its time
     interval.
      @returns True if the UART device is ready to be polled, False otherwise.
  */
  /*******************************************************************************/
  bool isReady() {
    if (millis() - _prvPoll > pollingInterval) {
      return true;
    }
    return false;
  }

  /*******************************************************************************/
  /*!
      @brief   Sets the last time a UART device driver was polled
      @param   curTime
               The current time, in milliseconds.
  */
  /*******************************************************************************/
  void setPrvPollTime(unsigned long curTime) { _prvPoll = curTime; }

  /*******************************************************************************/
  /*!
      @brief   Gets the UART device's unique identifier.
      @returns The UART device's unique identifier.
  */
  /*******************************************************************************/
  const char *getDriverID() { return _driverID; }

  /*******************************************************************************/
  /*!
      @brief   Sets the UART driver's identifer.
      @param   id
               The UART device's unique identifier.
  */
  /*******************************************************************************/
  void setDriverID(const char *id) { _driverID = strdup(id); }

  /*******************************************************************************/
  /*!
      @brief  Provides the UART device driver with an instance of the
     application's MQTT configuration.
      @param  AMQTT
              Pointer to an Adafruit_MQTT object.
      @param  mqtt_topic
              UART topic string, generated by adafruit MQTT
  */
  /*******************************************************************************/
  virtual void set_mqtt_client(Adafruit_MQTT *AMQTT, const char *mqtt_topic) {
    mqttClient = AMQTT;
    uartTopic = mqtt_topic;
  }

  /*******************************************************************************/
  /*!
      @brief   Initializes the UART device driver.
      @returns True if UART device driver initialized successfully, False
               otherwise.
  */
  /*******************************************************************************/
  virtual bool begin() { return false; }

  /*******************************************************************************/
  /*!
      @brief   Checks if the UART device's data is ready.
      @returns True if data is available, False otherwise.
  */
  /*******************************************************************************/
  virtual bool read_data() { return false; }

  /*******************************************************************************/
  /*!
      @brief   Packs the UART device's data into a UARTResponse message.
      @param   msgUARTResponse
               Pointer to a UARTResponse message.
      @param   event_index
               Index of the UART device's event.
      @param   sensor_type
               Type of sensor data.
      @param   sensor_value
               Sensor data value.
  */
  /*******************************************************************************/
  void packUARTResponse(wippersnapper_signal_v1_UARTResponse *msgUARTResponse,
                        int event_index,
                        wippersnapper_i2c_v1_SensorType sensor_type,
                        float sensor_value) {
    msgUARTResponse->payload.resp_uart_device_event.sensor_event[event_index]
        .type = sensor_type;
    msgUARTResponse->payload.resp_uart_device_event.sensor_event[event_index]
        .value = sensor_value;
  }

  /*******************************************************************************/
  /*!
      @brief   Reads the UART device's data then packs and sends it to IO.
  */
  /*******************************************************************************/
  virtual void send_data() {};

  const char *uartTopic = nullptr;     ///< UART device's MQTT topic
  Adafruit_MQTT *mqttClient = nullptr; ///< Pointer to MQTT client object
  unsigned long
      pollingInterval; ///< UART device's polling interval, in milliseconds
private:
  long _prvPoll = millis() - (24 * 60 * 60 * 1000);
  ///< Last time (ms) the UART device was polled, set to 24 hours (max period)
  const char *_driverID = nullptr; ///< UART device's ID
};

#endif // WS_UART_DRV_H
