# SerialConsole
A small and simple command line interface designed for Arduino projects. It provides argument parsing and is easily customized to fit any hardware or RAM constraint.

You can bind your own commands to the console like this:

```cpp
console.AddCommand("LED", cmd_LED, "Toggle the builtin LED.");
```

Then you open up the Arduino Serial Monitor, type in LED, and watch it turn on!

In the serial monitor, the console will tell you what the available commands are when you type anything in. If you type an unrecognized command, it will respond by listing out all the commands that are available. There is a builtin "help" function to display the optional helper messages that you put in when you add commands.

SerialConsole is very lightweight on ROM and RAM, can be configured to work with any Arduino Stream object, and should run on any Arduino board that supports millis(). The whole library is barely 200 lines long.

## Features
There are already tools like this out there. Why did I make a new one? Well, there are some features and values that I feel are core to a good terminal, that seemed to be missing in existing libraries.
* **Precise RAM Control (Fixed Memory Footprint):** Uses the SerialConsoleConfig structure to allow you to precisely define the maximum number of commands, maximum command length, and maximum arguments. This lets you tune the memory footprint to your exact RAM budget on small microcontrollers.
* **Efficient Argument Parsing:** Arguments are parsed in-place within a fixed buffer. Command functions receive direct pointers to the indexed arguments (console.Arguments[i]) and the argument count (console.ArgCount).
* **Built-in Help System:** Includes automatic command listing for unrecognized inputs and a structured help \<command> function to display optional command-specific helper messages. Error handling is clean and error messsages make sense.
* **Integrated Polling Timer:** Uses millis() internally to allow you to set a configurable poll rate (scanPeriod_ms) without adding any timing complexity to your main loop().
* **Bite Sized:** Good clean readability in the .h and .cpp files, and a very small code base so it's easy to read, understand, and fork.
* **Stream Agnostic:** The IO_Stream can be changed from the default Serial class to any other Arduino Stream object (e.g., Serial1, SoftwareSerial).
* **Custom Interface:** Delimiters, line terminators, and the command prompter (>>) are all fully adjustable via the config struct.
* **PuttyMode Support:** Built-in configuration for character-by-character terminals like PuTTY, VSCode, and other real terminal emulators with live character echo and backspace support.

## Terminal Modes

SerialConsole works in two modes:

**Line Mode (default)** - For Arduino IDE Serial Monitor
- Type commands in the text box at the top, press Enter to send
- Full command is echoed back with `>>` prefix after you hit Enter
- Best for: Arduino IDE Serial Monitor, basic serial terminals

**PuttyMode** - For real terminal emulators
- Characters appear as you type them (live echo)
- Backspace works to delete characters
- `>>` prompt shows when ready for input
- Best for: PuTTY, VSCode Serial Monitor, Terminal/iTerm2, screen/minicom

Enable PuttyMode with one line:
```cpp
SerialConsole console(PuttyMode());
```

See `examples/BasicExample` for line mode, and `examples/PuttyMode` for PuttyMode.

## Usage
Here is a simplified snippet from the BasicExample.ino to show how it works...
```cpp
#include <SerialConsole.h>

SerialConsole console; // Create a SerialConsole object

void setup(){
	Serial.begin(9600);

	/* ...some other setup stuff... */

	// Add commands to the SerialConsole like this...
	console.AddCommand("setpin", cmd_setpin, "Turn a digital output pin on or off"); 
	// This lets the user type in "setpin 13 1" in the Serial Monitor, and then the builtin LED will turn on!
	// The description at the end shows up if you type "help setpin".
}

void loop(){
	console.Listen(); // This goes in your main loop to process your commands
}

// Your commands call functions, which are set up like this...
void cmd_setpin(){
	// Check if you have the right number of arguments
	if(console.ArgCount != 3) Serial.println("ERROR: Must specify a pin and a state to set it to.");

	// Arguments[0] is the command name ("setpin")	
	int arg1 = String(console.Arguments[1]).toInt(); // Arguments[1] is the first argument (e.g., "13")
	int arg2 = String(console.Arguments[2]).toInt(); // Arguments[2] is the second argument (e.g., "1")

	if(arg1 > 0) pinMode(arg1, OUTPUT);
	else Serial.println("ERROR: The pin can not be negative.");

	if(arg2 == 0 || arg2 == 1) digitalWrite(arg1, arg2);
	else Serial.println("ERROR: Must set the pin to either 0 or 1.");
}
```

There are more features shown in the examples, for working with arguments, and for customizing the console's behavior, appearance, and memory footprint.

## Installation
1) Click the green "Code" button at the top right of the page and download everything as a .zip.
2) Extract the .zip to get a SerialConsole folder with all the contents from GitHub.
3) Put that folder in your Arduino libraries folder (Document\Arduino\Libraries on windows).
4) #include <SerialConsole.h> in your sketch. Check the BasicExample.ino for a quick start guide.