/**
 **************************************************
 *
 * @file        Soldered-OpenAI-Library.cpp
 * @brief       Core file that implements the functionalities
 *              of the LLM class
 *
 * @copyright   GNU General Public License v3.0
 * @authors     Josip Šimun Kuči @ soldered.com
 ***************************************************/
#include "Soldered-OpenAI-Library.h"

// helper to JSON-escape the prompt
String jsonEscape(const String &in)
{
    String out;
    out.reserve(in.length() + 16);
    for (size_t i = 0; i < in.length(); ++i)
    {
        char c = in[i];
        switch (c)
        {
        case '\"':
            out += F("\\\"");
            break;
        case '\\':
            out += F("\\\\");
            break;
        case '\b':
            out += F("\\b");
            break;
        case '\f':
            out += F("\\f");
            break;
        case '\n':
            out += F("\\n");
            break;
        case '\r':
            out += F("\\r");
            break;
        case '\t':
            out += F("\\t");
            break;
        default:
            if ((uint8_t)c < 0x20)
            {
                char buf[7];
                snprintf(buf, sizeof(buf), "\\u%04X", (unsigned)c);
                out += buf;
            }
            else
            {
                out += c;
            }
        }
    }
    return out;
}

/**
 * @brief         LLM constructor
 *
 * @param api     AI service provider (currently only "openai" supported)
 * @param key     API key for authentication
 * @param model   AI model to use (e.g., "gpt-3.5-turbo", "gpt-4")
 * @param maxTokens Maximum number of tokens in response (default: 300)
 */
LLM::LLM(const char *key, const char *model, unsigned int maxTokens)
{
    _api = "openai";
    _key = key;
    _model = model;
    _maxTokens = maxTokens;
}

void LLM::changeModel(const char *model)
{
    _model = model;
}

/**
 * @brief         Basic text response from AI
 *
 * @param prompt  Question or prompt to send to AI
 * @returns String AI response text
 */
String LLM::ask(const String &prompt)
{
    return sendToOpenAI(prompt);
}

/**
 * @brief         Yes/No classification response from AI
 *
 * @param prompt  Question or prompt to send to AI
 * @returns bool  True if AI response contains "yes", false otherwise
 */
bool LLM::askYesNo(const String &prompt)
{
    String response = sendToOpenAI(prompt + String(PROMPT_YESNO));
    response.toLowerCase();
    if (response.indexOf("yes") != -1)
        return true;
    return false;
}

/**
 * @brief               Multiple choice classification
 *
 * @param prompt        Question or prompt to send to AI
 * @param labels        Array of possible classification labels
 * @param labelCount    Number of labels in the array
 * @returns String      AI-selected label from the provided options
 */
String LLM::classify(const String &prompt, const String labels[], size_t labelCount)
{
    return sendToOpenAI(prompt, true, labels, labelCount);
}

/**
 * @brief                       Internal function: send request to OpenAI API
 *
 * @param prompt                Question or prompt to send
 * @param echoChoices           Whether to include choice labels in prompt
 * @param labels                Array of labels for classification
 * @param labelCount            Number of labels in array
 * @returns String              AI response text
 */
String LLM::sendToOpenAI(const String &prompt, bool echoChoices, const String labels[], size_t labelCount)
{
    // Build full prompt
    String fullPrompt = prompt;
    if (echoChoices && labels && labelCount > 0)
    {
        fullPrompt += "\nChoices: ";
        for (size_t i = 0; i < labelCount; i++)
        {
            fullPrompt += labels[i];
            if (i < labelCount - 1)
                fullPrompt += ", ";
        }
        fullPrompt += PROMPT_LABEL;
    }

    // Build request JSON
    DynamicJsonDocument doc(1024);
    doc["model"] = _model;
    doc["max_completion_tokens"] = _maxTokens;

    JsonArray messages = doc.createNestedArray("messages");
    JsonObject msg = messages.createNestedObject();
    msg["role"] = "user";
    msg["content"] = fullPrompt;

    String body;
    serializeJson(doc, body);

    // Send HTTPS request using HTTPClient
    HTTPClient http;
    http.begin("https://api.openai.com/v1/chat/completions");
    http.addHeader("Content-Type", "application/json");
    http.addHeader("Authorization", "Bearer " + String(_key));
    http.setTimeout(30000);

    int httpCode = http.POST(body);

    String reply = "";
    if (httpCode > 0)
    {
        String res = http.getString();

        DynamicJsonDocument resDoc(8192);
        DeserializationError error = deserializeJson(resDoc, res);

        if (error)
        {
            Serial.printf("JSON parse error: %s\n", error.c_str());
            http.end();
            return "JSON parse error";
        }

        if (resDoc.containsKey("error"))
        {
            Serial.println("An error occurred while trying to create a prompt:");
            Serial.println(resDoc["error"]["message"].as<String>());
            http.end();
            return "";
        }

        if (resDoc.containsKey("choices") && resDoc["choices"].is<JsonArray>() && resDoc["choices"].size() > 0 &&
            resDoc["choices"][0]["message"]["content"])
        {
            reply = resDoc["choices"][0]["message"]["content"].as<String>();
            reply.trim();
        }
        else
        {
            Serial.println("Unexpected response structure:");
            Serial.println(res);
            reply = "";
        }
    }
    else
    {
        Serial.printf("HTTP POST failed, error: %s\n", http.errorToString(httpCode).c_str());
        reply = "Connection failed!";
    }

    http.end();
    return reply;
}


/**
 * @brief                   Ask GPT Vision with base64 encoded image
 *
 * @param buffer            Buffer pointer to image data
 * @param bufferLength      Length of the buffer in bytes
 * @param prompt            Question or prompt about the image
 * @returns String          AI response describing the image
 */
String LLM::askImage(uint8_t *buffer, int bufferLength, const String &prompt)
{
    // Convert image to base64 for transmission
    String base64Image = base64::encode(buffer, bufferLength);

    // Use larger JSON document for response
    DynamicJsonDocument doc(8192);

    // Build the JSON packet which will be sent via HTTP
    doc["model"] = _model;

    JsonArray messages = doc.createNestedArray("messages");
    JsonObject message = messages.createNestedObject();
    message["role"] = "user";

    JsonArray content = message.createNestedArray("content");

    JsonObject textContent = content.createNestedObject();
    textContent["type"] = "text";
    textContent["text"] = prompt;

    JsonObject imageContent = content.createNestedObject();
    imageContent["type"] = "image_url";
    JsonObject imageUrl = imageContent.createNestedObject("image_url");

    String imageData = base64Image;

    imageUrl["url"] = "data:image/jpeg;base64," + imageData;
    imageContent["image_url"]["detail"] = "auto";
    doc["max_completion_tokens"] = _maxTokens;

    String payload;
    serializeJson(doc, payload);

    // Check if we have enough memory
    if (ESP.getFreeHeap() < 50000)
    {
        Serial.println("ERROR: Low memory, aborting HTTP request");
        return "Error: Low memory";
    }

    HTTPClient http;
    http.begin("https://api.openai.com/v1/chat/completions");
    http.addHeader("Content-Type", "application/json");
    http.addHeader("Authorization", "Bearer " + String(_key));
    http.setTimeout(30000); // Increase timeout to 30 seconds

    int httpCode = http.POST(payload);
    Serial.printf("HTTP code: %d\n", httpCode);

    String reply = "";
    if (httpCode > 0)
    {
        String res = http.getString();

        // Use larger document for response parsing
        DynamicJsonDocument resDoc(16384);
        DeserializationError error = deserializeJson(resDoc, res);

        if (error)
        {
            Serial.printf("JSON parse error: %s\n", error.c_str());
        }
        else if (resDoc.containsKey("choices"))
        {
            if (resDoc["choices"][0]["message"]["content"])
            {
                reply = resDoc["choices"][0]["message"]["content"].as<String>();
                reply.trim();
            }
        }
        else if (resDoc.containsKey("error"))
        {
            Serial.printf("API Error: %s\n", resDoc["error"]["message"].as<String>().c_str());
        }
    }
    else
    {
        Serial.printf("HTTP POST failed, error: %s\n", http.errorToString(httpCode).c_str());
    }

    http.end();
    return reply;
}

/**
 * @brief                   Changes max token count to be used by LLM
 *                          in a single font
 *
 * @param maxTokens         Number of maxTokens to be set
 */
void LLM::changeMaxTokenCount(unsigned int maxTokens)
{
    _maxTokens = maxTokens;
}


/**
 * @brief                   Ask GPT-4o Audio Preview with base64 encoded WAV/MP3 audio
 *
 * @param audioBuffer       Pointer to buffer containing audio data
 * @param audioLength       Length of the audio data in bytes
 * @param format            Audio format string ("wav" or "mp3")
 * @param prompt            Question or prompt about the audio
 * @returns String          AI response text describing or analyzing the audio
 */
String LLM::askAudio(const uint8_t *audioBuffer, size_t audioLength, const String &format, const String &prompt)
{
    if (!audioBuffer || audioLength == 0)
        return "Invalid audio buffer!";

    // Base64 encode audio
    String base64Audio = base64::encode(audioBuffer, audioLength);
    if (base64Audio.length() == 0)
    {
        Serial.println("Error: Failed to Base64 encode audio!");
        return "Error: Failed to encode audio";
    }
    // Build request JSON manually as a String due to the size of the base64 audio
    String payload;
    payload.reserve(128 + strlen(_model) + format.length() + prompt.length() + base64Audio.length());

    payload += F("{\"model\":\"");
    payload += _model;
    payload += F("\",\"max_completion_tokens\":");
    payload += String(_maxTokens);
    payload += F(",\"messages\":[{\"role\":\"user\",\"content\":[");
    payload += F("{\"type\":\"text\",\"text\":\"");
    payload += jsonEscape(prompt);
    payload += F("\"},");
    payload += F("{\"type\":\"input_audio\",\"input_audio\":{");
    payload += F("\"data\":\"");
    payload += base64Audio;
    payload += F("\",\"format\":\"");
    payload += format; // "wav" or "mp3"
    payload += F("\"}}]}]}");

    // Check available heap before sending
    if (ESP.getFreeHeap() < 50000)
    {
        Serial.println("ERROR: Low memory, aborting HTTP request");
        return "Error: Low memory";
    }

    // Send HTTPS request
    HTTPClient http;
    http.begin("https://api.openai.com/v1/chat/completions");
    http.addHeader("Content-Type", "application/json");
    http.addHeader("Authorization", "Bearer " + String(_key));
    http.setTimeout(60000); // 60 sec timeout

    int httpCode = http.POST(payload);
    Serial.printf("HTTP code: %d\n", httpCode);

    String reply = "";
    if (httpCode > 0)
    {
        String res = http.getString();

        DynamicJsonDocument resDoc(16384);
        DeserializationError error = deserializeJson(resDoc, res);

        if (error)
        {
            Serial.printf("JSON parse error: %s\n", error.c_str());
        }
        else if (resDoc.containsKey("choices"))
        {
            if (resDoc["choices"][0]["message"]["content"])
            {
                reply = resDoc["choices"][0]["message"]["content"].as<String>();
                reply.trim();
            }
        }
        else if (resDoc.containsKey("error"))
        {
            Serial.printf("API Error: %s\n", resDoc["error"]["message"].as<String>().c_str());
        }
    }
    else
    {
        Serial.printf("HTTP POST failed, error: %s\n", http.errorToString(httpCode).c_str());
    }

    http.end();
    return reply;
}


/**
 * @brief                   Generate an image using OpenAI's image generation models (DALL·E)
 *
 *                          Sends a text prompt to the OpenAI Images API and requests
 *                          generation of a single image. The API returns a temporary
 *                          URL pointing to the generated image.
 *
 * @param prompt            Text prompt describing the image to generate
 * @param resolution        Image resolution string (e.g. "1024x1024", "1792x1024", "1024x1792")
 * @param imageFormat       Desired image format (currently unused, reserved for future use)
 * @param quality           Image quality ("standard" or "hd", DALL·E 3 only)
 * @param style             Image style ("vivid" or "natural", DALL·E 3 only)
 * @returns String          URL of the generated image, or an error message
 */
String LLM::generateImage(const String &prompt, const String &resolution, const String &imageFormat,
                          const String &quality, const String &style)
{
    // Reject empty prompts early
    if (prompt.length() == 0)
        return "Error: Empty prompt";

    // Cache model name as String for comparisons
    String modelStr = String(_model);

    // Create JSON document that will be sent to OpenAI Images API
    DynamicJsonDocument doc(1024);

    // Set basic image generation parameters
    doc["model"] = _model;          // e.g. "dall-e-3" or "dall-e-2"
    doc["prompt"] = prompt;         // Text prompt describing the image
    doc["n"] = 1;                   // Generate only one image
    doc["size"] = resolution;       // Image resolution (e.g. "1024x1024")
    doc["response_format"] = "url"; // Request URL response instead of base64

    // DALL·E 3 supports additional quality and style parameters
    if (_model == "dall-e-3")
    {
        doc["quality"] = quality; // "standard" or "hd"
        doc["style"] = style;     // "vivid" or "natural"
    }

    // Serialize JSON document into a string payload
    String payload;
    serializeJson(doc, payload);

    // Create HTTP client and configure request
    HTTPClient http;
    http.begin("https://api.openai.com/v1/images/generations");
    http.addHeader("Content-Type", "application/json");
    http.addHeader("Authorization", "Bearer " + String(_key));
    http.setTimeout(60000); // Allow up to 60 seconds (image generation can be slow)

    // Send POST request to OpenAI
    int httpCode = http.POST(payload);
    Serial.printf("HTTP code: %d\n", httpCode);

    String result = "";

    // Check if HTTP request succeeded
    if (httpCode > 0)
    {
        // Read full response body
        String res = http.getString();

        // Parse JSON response
        DynamicJsonDocument resDoc(8192);
        DeserializationError error = deserializeJson(resDoc, res);

        if (error)
        {
            // JSON parsing failed
            Serial.printf("JSON parse error: %s\n", error.c_str());
            result = "Error: JSON parse error";
        }
        else if (resDoc.containsKey("error"))
        {
            // OpenAI API returned an error object
            const char *msg = resDoc["error"]["message"] | "Unknown API error";
            Serial.printf("API Error: %s\n", msg);
            result = String("Error: ") + msg;
        }
        else if (resDoc.containsKey("data") && resDoc["data"].is<JsonArray>() && resDoc["data"].size() > 0)
        {
            // The generated image data is in data[0]
            // When response_format is "url", the image URL is in data[0].url
            if (resDoc["data"][0].containsKey("url"))
            {
                result = resDoc["data"][0]["url"].as<String>();
                result.trim();
            }
            else if (resDoc["data"][0].containsKey("b64_json"))
            {
                // This happens if the model returns base64 instead of URL
                result = "Error: API returned base64 (b64_json) instead of URL. Use dall-e-2/dall-e-3 with "
                         "response_format:url.";
            }
            else
            {
                // Unexpected response structure
                result = "Error: No url field in response";
            }
        }
        else
        {
            // Unexpected JSON layout
            result = "Error: Unexpected response structure";
        }
    }
    else
    {
        // HTTP request itself failed
        Serial.printf("HTTP POST failed, error: %s\n", http.errorToString(httpCode).c_str());
        result = String("Error: HTTP POST failed: ") + http.errorToString(httpCode);
    }

    // Always close HTTP connection
    http.end();

    return result;
}
