/**
 **************************************************
 *
 * @file        imageGeneration.ino
 * 
 * @brief       An example showing how to generate an image via a prompt and setting its size, format, quality and style. 
 *              The OpenAI API returns the image as a URL, which is then rendered and displayed on an Inkplate device.
 *         
 *              Enter your WiFi credentials and your OpenAI API key, which you can get by going to https://platform.openai.com/api-keys
 *              You can change the prompt, size etc. below in the code.
 *
 * @link        Devices used in this example: Soldered Inkplate 10: https://soldered.com/product/inkplate-10-9-7-e-paper-board-copy/
 * 
 * @link        Inkplate Arduino Library: https://github.com/SolderedElectronics/Inkplate-Arduino-library
 *
 * @authors     Josip Šimun Kuči @ soldered.com
 ***************************************************/
#include "Soldered-OpenAI-Library.h"

#include "Inkplate.h"

// WiFi credentials
const char* WIFI_SSID = "YOUR_SSID_HERE";
const char* WIFI_PASS = "YOUR_PASSWORD_HERE";

// Specifying what format the image will be (Used by the Inkplate library)
Image::Format imageFormat = Image::PNG;

// Create an instance of the Inkplate display in 3-bit mode
Inkplate display(INKPLATE_3BIT);

// OpenAI API key
const char* OPENAI_API_KEY = "YOUR_API_KEY_HERE";

// Create LLM instance, which uses the dall-e-3 model and a max_token size of 300
// Use dall-e-3 or dall-e-2 to get URL output
LLM ai(OPENAI_API_KEY, "dall-e-3", 300);

// Helper function which connects to WiFi
void connectWiFi()
{
    Serial.print("Connecting to WiFi");
    WiFi.begin(WIFI_SSID, WIFI_PASS);

    while (WiFi.status() != WL_CONNECTED)
    {
        delay(500);
        Serial.print(".");
    }

    Serial.println();
    Serial.print("Connected! IP: ");
    Serial.println(WiFi.localIP());
}

void setup()
{
    Serial.begin(115200);
    delay(1000);
    // Initialize the display
    display.begin();
    // Connect to WiFi
    connectWiFi();

    Serial.println();
    Serial.println("Generating image...");

    // Prompt which defines what image you want to generate
    String prompt = "Generate a quite dashing looking french cat with a baguette and a moustache in a hot air baloon above green hills";

    // Function that generates the image itself, for example, this generates a french cat in 1024x1024 resolution as a png in hd quality and a vivid style.
    String imageUrl = ai.generateImage(prompt, "1024x1024", "png", "hd", "vivid");
    
    // If the image generation is successfull, the imageURL will start with http/https
    if (imageUrl.startsWith("http"))
    {
        Serial.println("Image generated!");
        Serial.println("Image URL:");
        Serial.println(imageUrl);
        // Draw the image onto the display by downloading it to the Inkplate device
        display.image.draw(imageUrl, imageFormat,0,0);
        display.display();
        Serial.println("Display finished!");
    }
    else
    {
        Serial.println("Failed to generate image:");
        Serial.println(imageUrl);
    }
}

void loop()
{
    // Nothing to do
}