/**
 **************************************************
 *
 * @file        cameraVision.ino
 * 
 * @brief       Example showing how to capture images with a camera
 *              and send them to AI for analysis when a button is pressed.
 *              The answer comes in the form of a string.
 *              The camera used in this example was a OV5640 camera.
 * 
 *              Enter your WiFi credentials and your OpenAI API key, which you can get by going to https://platform.openai.com/api-keys
 *              You can change the prompt, image format etc. in the code below.
 * 
 *              NOTE: for simplicity, this example uses the esp_camera library to communicate with the camera, so this example is designed
 *                    for ESP32 microcontrollers.
 *
 * @authors     Josip Šimun Kuči @ soldered.com
 ***************************************************/
#include <Arduino.h>
#include <base64.h>
#include "esp_camera.h"
#include "Soldered-OpenAI-Library.h"
#include <WiFi.h>

// WiFi network credentials
const char* ssid = "YOUR_SSID_HERE";        // Your WiFi network name
const char* password = "YOUR_PASSWORD_HERE"; // Your WiFi password

// OpenAI API configuration
const char* api_key = "YOUR_API_KEY_HERE";   // Get this from https://platform.openai.com/

// Create AI instance using GPT-4 Vision model for image analysis
LLM llm(api_key, "gpt-4.1");

// Camera pin configuration for OV5640 camera module
#define CAM_PIN_PWDN -1      // Power down pin (not used)
#define CAM_PIN_RESET 20     // Software reset pin
#define CAM_PIN_VSYNC 6      // Vertical sync
#define CAM_PIN_HREF 7       // Horizontal reference
#define CAM_PIN_PCLK 13      // Pixel clock
#define CAM_PIN_XCLK 15      // System clock
#define CAM_PIN_SIOD 4       // I2C data
#define CAM_PIN_SIOC 5       // I2C clock
#define CAM_PIN_D0 11        // Data bit 0
#define CAM_PIN_D1 9         // Data bit 1
#define CAM_PIN_D2 8         // Data bit 2
#define CAM_PIN_D3 10        // Data bit 3
#define CAM_PIN_D4 12        // Data bit 4
#define CAM_PIN_D5 18        // Data bit 5
#define CAM_PIN_D6 17        // Data bit 6
#define CAM_PIN_D7 16        // Data bit 7

#define BUTTON_PIN 48        // Pin connected to the push button

// Camera configuration structure
static camera_config_t camera_config = {
    .pin_pwdn = CAM_PIN_PWDN,
    .pin_reset = CAM_PIN_RESET,
    .pin_xclk = CAM_PIN_XCLK,
    .pin_sccb_sda = CAM_PIN_SIOD,
    .pin_sccb_scl = CAM_PIN_SIOC,

    .pin_d7 = CAM_PIN_D7,
    .pin_d6 = CAM_PIN_D6,
    .pin_d5 = CAM_PIN_D5,
    .pin_d4 = CAM_PIN_D4,
    .pin_d3 = CAM_PIN_D3,
    .pin_d2 = CAM_PIN_D2,
    .pin_d1 = CAM_PIN_D1,
    .pin_d0 = CAM_PIN_D0,
    .pin_vsync = CAM_PIN_VSYNC,
    .pin_href = CAM_PIN_HREF,
    .pin_pclk = CAM_PIN_PCLK,

    .xclk_freq_hz = 10000000,        // 10MHz clock
    .ledc_timer = LEDC_TIMER_0,      // LED controller timer
    .ledc_channel = LEDC_CHANNEL_0,  // LED controller channel

    .pixel_format = PIXFORMAT_JPEG,  // JPEG format for smaller file size
    .frame_size = FRAMESIZE_QVGA,    // 320x240 resolution (good balance of quality/size)

    .jpeg_quality = 13,              // Quality level (0-63, lower = higher quality)
    .fb_count = 1,                   // Number of frame buffers
    .fb_location = CAMERA_FB_IN_PSRAM, // Store frames in PSRAM if available
    .grab_mode = CAMERA_GRAB_WHEN_EMPTY,
};

/**
 * @brief Initialize the camera module
 * 
 * @returns esp_err_t ESP_OK if successful, error code if failed
 */
esp_err_t camera_init() {
    // Initialize the camera with the configuration
    esp_err_t err = esp_camera_init(&camera_config);
    if (err != ESP_OK) {
        return err;
    }
    return ESP_OK;
}


// Capture an image and send it to AI for analysis
esp_err_t camera_capture() {
    // Step 1: Capture image from camera
    camera_fb_t* fb = esp_camera_fb_get();  // First capture (discarded)
    esp_camera_fb_return(fb);               // Free the buffer
    fb = esp_camera_fb_get();               // New capture (better quality)
    
    // Check if capture was successful
    if (!fb) {
        Serial.println("Error: image capture failed");
        return ESP_FAIL;
    }

    // Display image information
    Serial.print("Image captured successfully! Size: ");
    Serial.print(fb->len);
    Serial.println(" bytes");

    // Free the camera buffer
    esp_camera_fb_return(fb);

    // Send image to AI and ask for analysis
    String response = llm.askImage(fb->buf, fb->len, "Explain what is on the image provided in detail (up to 100 words)");
    Serial.println("RESPONSE: " + String(response));
    
    return ESP_OK;
}


void setup() {
    // Initialize camera reset pin
    pinMode(CAM_PIN_RESET, OUTPUT);
    digitalWrite(CAM_PIN_RESET, HIGH);

    // Initialize button pin with internal pull-up resistor
    pinMode(BUTTON_PIN, INPUT_PULLUP);
    
    // Start serial communication for debugging
    Serial.begin(115200);
    delay(1000);
    Serial.println("Starting Camera AI Example...");

    // Connect to WiFi
    Serial.print("Connecting to WiFi");
    WiFi.begin(ssid, password);
    while (WiFi.status() != WL_CONNECTED) {
        delay(1000);
        Serial.print(".");
    }
    Serial.println("\nWiFi Connected!");
    
    // Enable debug logging
    esp_log_level_set("*", ESP_LOG_DEBUG);
    
    // Initialize camera
    if (camera_init() != ESP_OK) {
        Serial.println("Camera initialization failed!");
        while(1); // Stop program if camera fails
    }

    Serial.println("Camera initialized successfully!");
    Serial.println("Press the button to capture and analyze an image...");
}

void loop() {
    // Check if button is pressed (LOW because of pull-up resistor)
    if (!digitalRead(BUTTON_PIN)) {
        Serial.println("Button pressed! Capturing image...");
        
        // Capture image and send to AI
        if (camera_capture() == ESP_OK) {
            Serial.println("Capture and analysis successful!");
        } else {
            Serial.println("Failed to capture image");
        }
        
        // Small delay to prevent multiple captures from single press
        delay(1000);
    }
}