/**
 **************************************************
 *
 * @file        basicExample.ino
 * 
 * @brief       Example showing how to ask text prompts,
 *              yes/no and classification questions using the
 *              ChatGPT API
 * 
 *              Enter your WiFi credentials and your OpenAI API key, which you can get by going to https://platform.openai.com/api-keys
 *              You can change the prompts used in the code below.
 *
 * @authors     Josip Šimun Kuči @ soldered.com
 ***************************************************/
#include <WiFi.h>
#include "Soldered-OpenAI-Library.h"

// WiFi network credentials - REPLACE WITH YOUR NETWORK INFO
const char* ssid = "YOUR_SSID_HERE";        // Your WiFi network name
const char* password = "YOUR_PASSWORD_HERE"; // Your WiFi password

// OpenAI API configuration - REPLACE WITH YOUR API KEY
const char* api_key = "YOUR_API_KEY_HERE";   // Get this from https://platform.openai.com/

// Create an AI instance with OpenAI service
// Parameters: service name, API key, model name ("o3-mini" is a small, efficient model)
LLM llm(api_key, "o3-mini");


void setup() {
  // Start serial communication for debugging (115200 baud rate)
  Serial.begin(115200);
  
  // Connect to WiFi network
  Serial.print("Connecting to WiFi");
  WiFi.begin(ssid, password);
  
  // Wait until connected to WiFi
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("\nConnected to WiFi!");
  
  // Ask a simple question and get a text response
  Serial.println("\n=== BASIC TEXT QUESTION ===");
  const String basicQuestion = "Got a fun fact for me today? (20 words or less)";
  Serial.println("Q: " + basicQuestion);
  String reply = llm.ask(basicQuestion);
  Serial.println("Reply: " + reply);
  
  // Ask a safety question that requires a yes/no answer
  Serial.println("\n=== YES/NO QUESTION ===");
  const String yesNoQuestion = "Can I feed my dog chocolate?";
  Serial.print("Q: " + yesNoQuestion);
  if (llm.askYesNo(yesNoQuestion)) {
      Serial.println("YES");      // AI said "yes"
  } else {
      Serial.println("NO");  // AI said "no" or didn't confirm
  }
  
  // Example 3: Multiple choice classification
  // Ask AI to choose the correct option from given choices
  Serial.println("\n=== MULTIPLE CHOICE ===");
  const String multipleChoiceQuestion = "Which of these lives in a jungle";
  String choices[] = {"cat", "dog", "monkey"};  // Available options
  int choiceCount = 3;  // Number of choices in the array
  Serial.println(multipleChoiceQuestion);
  String result = llm.classify(multipleChoiceQuestion, choices, choiceCount);
  Serial.println("Result: " + result);  // Should print "monkey"
}


void loop() {
  // Nothing to do here - examples run once in setup()
}