/*!
 * @file SM_8CRT.h
 * 
 * Designed specifically to work with the Sequent Microsysatems 
 * HALL Current Sensor DAQ 8-Layer Stackable HAT for Raspberry Pi
 * 
 * ----> https://sequentmicrosystems.com/products/hall-current-sensor-daq-stackable-hat-for-raspberry-pi
 * 
 *   This card use CRT8 to communicate.
 *   
 *   Written by Alexandru Burcea for Sequent Microsystems
 * 
 *   Software License Agreement (BSD License)
 *
 *  Copyright (c) 2025, Sequent Microsystems 
 *  
 */
#ifndef __SM_8CRT__
#define __SM_8CRT__

#define SLAVE_OWN_ADDRESS_8CRT 0x19

#define CRT_IN_CH_NR 8
#define COUNTER_SIZE 4
#define ANALOG_VAL_SIZE 2
#define MODBUS_SETTINGS_SIZE_B 5
#define SM_8CRT_I2C_MAX_BUFF  32
#define SM_CRT_SCALE_FACTOR ((float)100)
#define SM_CRT_RTC_CMD 0xaa

enum
{

	CRT8_LEDS,
	CRT8_LED_SET,
	CRT8_LED_CLR,

	CRT8_CRT_IN_VAL1_ADD , // current vales scaled as A/100 (1 = 0.01A) 16-bit signed integer
	CRT8_CRT_IN_RMS_VAL1_ADD  = CRT8_CRT_IN_VAL1_ADD + ANALOG_VAL_SIZE * CRT_IN_CH_NR, //current RMS values scaled as A/100 16bit unsigned integer
	CRT8_CRT_SNS_RANGE1_ADD = CRT8_CRT_IN_RMS_VAL1_ADD + ANALOG_VAL_SIZE * CRT_IN_CH_NR,// full scale in A for each sensor 16-bit unsigned integer
	CRT8_CRT_SNS_TYPE = CRT8_CRT_SNS_RANGE1_ADD + ANALOG_VAL_SIZE * CRT_IN_CH_NR,// a bitmap for all 8 channels 0 = 2.5V +/- 0.625V , and  1 = 2.5V +/- 1V type of sensor

	CRT8_CALIB_VALUE, // floating point value expressing the current in A
	CRT8_CALIB_CHANNEL = CRT8_CALIB_VALUE + 4,
	CRT8_CALIB_KEY, //set calib point -> 0xaa; reset calibration on the channel -> 0x55; save zero current offset -> 0x11
	CRT8_CALIB_STATUS,

	CRT8_RTC_YEAR_ADD,
	CRT8_RTC_MONTH_ADD,
	CRT8_RTC_DAY_ADD,
	CRT8_RTC_HOUR_ADD,
	CRT8_RTC_MINUTE_ADD,
	CRT8_RTC_SECOND_ADD,
	CRT8_RTC_SET_YEAR_ADD,
	CRT8_RTC_SET_MONTH_ADD,
	CRT8_RTC_SET_DAY_ADD,
	CRT8_RTC_SET_HOUR_ADD,
	CRT8_RTC_SET_MINUTE_ADD,
	CRT8_RTC_SET_SECOND_ADD,
	CRT8_RTC_CMD_ADD,

	CRT8_WDT_RESET_ADD,
	CRT8_WDT_INTERVAL_SET_ADD,
	CRT8_WDT_INTERVAL_GET_ADD = CRT8_WDT_INTERVAL_SET_ADD + 2,
	CRT8_WDT_INIT_INTERVAL_SET_ADD = CRT8_WDT_INTERVAL_GET_ADD + 2,
	CRT8_WDT_INIT_INTERVAL_GET_ADD = CRT8_WDT_INIT_INTERVAL_SET_ADD + 2,
	CRT8_WDT_RESET_COUNT_ADD = CRT8_WDT_INIT_INTERVAL_GET_ADD + 2,
	CRT8_WDT_CLEAR_RESET_COUNT_ADD = CRT8_WDT_RESET_COUNT_ADD + 2,
	CRT8_WDT_POWER_OFF_INTERVAL_SET_ADD,
	CRT8_WDT_POWER_OFF_INTERVAL_GET_ADD = CRT8_WDT_POWER_OFF_INTERVAL_SET_ADD + 4,
	CRT8_BUTTON = CRT8_WDT_POWER_OFF_INTERVAL_GET_ADD + 4,
	CRT8_DIAG_RASP_V,
	CRT8_DIAG_RASP_V1,

    CRT8_MODBUS_SETINGS_ADD,
	CRT8_MODBUS_SETINGS_END_ADD = CRT8_MODBUS_SETINGS_ADD + MODBUS_SETTINGS_SIZE_B,

	CRT8_UPDATE_ADD = 0xaa,

	CRT8_REVISION_HW_MAJOR_ADD = 250,
	CRT8_REVISION_HW_MINOR_ADD,
	CRT8_REVISION_MAJOR_ADD,
	CRT8_REVISION_MINOR_ADD,



	SLAVE_BUFF_SIZE = 255,
};

/*!
 * @brief SM_8CRT class
 */
class SM_8CRT
{
public:
	/*!
	 * @brief Class constructor.
	 */
	SM_8CRT(uint8_t stack = 0);

	/*!
	 * @brief Check card presence
	 * @return Returns true is successful
	 */
	bool begin();

	/*!
	 * @brief Return card existance status
	 * @return Returns true if card is present
	 */
	bool isAlive();
	/*!
	 * @brief Read current for one input channel.
	 * @param channel [1..8]
	 * @return momentary current in A
	 */
	float read(uint8_t channel);

    /*!
	 * @brief Read RMS current for one input channel.
	 * @param channel [1..8]
	 * @return RMS current in A
	 */
	float readRMS(uint8_t channel);
	
	
	/*!
	 * @brief Turn on or Off one led
	 * @param led The led number 1..8
	 * @param val The new state true = on, false = off
	 * @return true if succeed, false else
	 */
	bool writeLED(uint8_t led, bool val);

	/*!
	 * @brief Turn on or Off all leds
	 * @param val All 8 led's state as a bitmap
	 * @return true if succeed, false else
	 */
	bool writeLED(uint8_t val);
	
	
	/*!
	 * @brief Set the sensor current range or full scale.
	 * @param channel [1..8]
	 * @param range [2..300]
	 * @return true if success false elese
	 */
	bool writeRange(uint8_t channel, int range);
	
	/*!
	 * @brief Get the sensor current range or full scale.
	 * @param channel [1..8]
	 * @return senzor range
	 */
	int readRange(uint8_t channel);
	
		
	/*!
	 * @brief Set the RTC.
	 * @param m month
	 * @param d day
	 * @param y year
	 * @param h hour
	 * @param mi minutes
	 * @param s seconds
	 * @return true if success false elese
	 */
	bool writeRTC(int m, int d, int y, int h, int mi, int s);
	
	
	/*!
	 * @brief Get the RTC time and date.
	 * @param m month
	 * @param d day
	 * @param y year
	 * @param h hour
	 * @param mi minutes
	 * @param s seconds
	 * @return true if success false elese
	 */
	bool readRTC(int* m, int* d, int* y, int* h, int* mi, int* s);
	
	/*!
	 * @brief Set offset on specified channel.
	 * @param channel [1..8]
	 * @return true if success false elese
	 */
	bool zero(uint8_t channel);
		
private:
	uint8_t _hwAdd;
	bool _detected;
	int writeBuff(uint8_t add, uint8_t* buff, uint8_t size);
	int readBuff(uint8_t add, uint8_t* buff, uint8_t size);
	int readWord(uint8_t add, uint16_t* value);
	int readSWord(uint8_t add, int16_t* value);
	int writeByte(uint8_t add, uint8_t value);
	int writeWord(uint8_t add, uint16_t value);
	int writeSWord(uint8_t add, int16_t value);
};
#endif // __SM_16DIGIN__
