extern "C" {
#include <inttypes.h>
}

#include "Arduino.h"
#include "Wire.h"
#include "SM_8CRT.h"



SM_8CRT::SM_8CRT(uint8_t stack)
{
	if (stack > 7)
		stack = 7;
	_hwAdd = SLAVE_OWN_ADDRESS_8CRT + stack;
	_detected = false;
}

bool SM_8CRT::begin()
{
	uint16_t value = 0;
	Wire.begin();
	if (0 == readWord(CRT8_REVISION_MAJOR_ADD, &value))
	{
		_detected = true;
	}
	return _detected;
}

bool SM_8CRT::isAlive()
{
	return _detected;
}


float SM_8CRT::read(uint8_t channel)
{
	int ret = 0;
	int16_t val = 0;
	
	
	if ((channel > CRT_IN_CH_NR) || (channel < 1))
		return -1000;

	ret = readSWord(CRT8_CRT_IN_VAL1_ADD, &val);
	if(ret < 0)
	{
		return -2000;
	}
	
	return((float)val / SM_CRT_SCALE_FACTOR);
}

float SM_8CRT::readRMS(uint8_t channel)
{
	int ret = 0;
	uint16_t val = 0;
	
	
	if ((channel > CRT_IN_CH_NR) || (channel < 1))
		return -1000;

	ret = readWord(CRT8_CRT_IN_RMS_VAL1_ADD, &val);
	if(ret < 0)
	{
		return -2000;
	}
	
	return((float)val / SM_CRT_SCALE_FACTOR);
}

bool SM_8CRT::writeLED(uint8_t led, bool val)
{
	if (led > CRT_IN_CH_NR || led == 0)
  {
    return false;
  }
  if (val)
  {
    if (OK == writeByte(CRT8_LED_SET, led))
      return true;
  }
  else
  {
    if (OK == writeByte(CRT8_LED_CLR, led))
      return true;
  }
  return false;
}

bool SM_8CRT::writeLED(uint8_t val)
{
	if (OK == writeByte(CRT8_LEDS, val))
      return true;
  return false;
}

bool SM_8CRT::writeRange(uint8_t channel, int range)
{
	if ((channel > CRT_IN_CH_NR) || (channel < 1))
		return false;
	if((range > 300) || (range < 2))
		return false;
	if(OK == writeWord(CRT8_CRT_SNS_RANGE1_ADD + 2*channel, (uint16_t)range))
		return true;
	return false;
}

int SM_8CRT::readRange(uint8_t channel)
{
	uint16_t val = 0;
	
	if ((channel > CRT_IN_CH_NR) || (channel < 1))
		return 0;
	readWord(CRT8_CRT_SNS_RANGE1_ADD + 2*channel, &val);
	return (int)val;
}

bool SM_8CRT::writeRTC(int m, int d, int y, int h, int mi, int s)
{
	uint8_t buff[7];
	
	if((m < 1) || (m > 12))
		return false;
	buff[1] = (uint8_t)m;
	
	if((d < 1) || (d > 31))
		return false;
	buff[2] = (uint8_t)d;
	
	if(y >= 100) 
		y = y % 100;
	if((y < 0) || (y > 99))
		return false;
	buff[0] = (uint8_t)y;
	
	if((h < 0) || (h > 23))
		return false;
	buff[3] = (uint8_t)h;
	
	if((mi < 0)|| (mi > 59))
		return false;
	buff[4] = (uint8_t)mi;
	
	if((s < 0) || (s > 59))
		return false;
	buff[5] = (uint8_t)s;
	
	buff[6] = SM_CRT_RTC_CMD;
	
	if(OK == writeBuff(CRT8_RTC_SET_YEAR_ADD, buff, 7))
		return true;
	return false;
}

bool SM_8CRT::readRTC(int* m, int* d, int* y, int* h, int* mi, int* s)
{
	uint8_t buff[6];
	
	if(OK != readBuff(CRT8_RTC_YEAR_ADD,buff, 6))
		return false;
	*m = buff[1];
	*d = buff[2];
	*y = buff[0] + 2000;
	*h = buff[3];
	*m = buff[4];
	*s = buff[5];
	return true;
}

bool SM_8CRT::zero(uint8_t channel)
{
	if ((channel > CRT_IN_CH_NR) || (channel < 1))
		return false;
	
	if(OK != writeByte(CRT8_CALIB_CHANNEL, channel))
		return false;
	
	if(OK != writeByte(CRT8_CALIB_KEY, 0x11))
		return false;
	return true;
}

/*
 ***************** 8crt_I2C access functions ****************************
 **********************************************************************
 */
 int SM_8CRT::writeBuff(uint8_t add, uint8_t* buff, uint8_t size)
 {
	int i = 0;

	if(size > SM_8CRT_I2C_MAX_BUFF)
	{
		return -1;
	}
	
	Wire.beginTransmission(_hwAdd);
	Wire.write(add);
	for (i = 0; i < size; i++)
	{
		Wire.write(buff[i]);
	}
	return Wire.endTransmission();
 }
 
int SM_8CRT::readBuff(uint8_t add, uint8_t* buff, uint8_t size)
{
	uint8_t i = 0;

	if (0 == buff)
	{
		return -1;
	}
	if(size > SM_8CRT_I2C_MAX_BUFF)
	{
		return -1;
	}
	
	Wire.beginTransmission(_hwAdd);
	Wire.write(add);
	if (Wire.endTransmission() != 0)
	{
		return -1;
	}
	Wire.requestFrom(_hwAdd, size);
	if (size <= Wire.available())
	{
		for(i = 0; i < size; i++)
		{
			buff[i] = Wire.read();
		}
	}
	else
	{
		return -1;
	}
	return 0;
}

int SM_8CRT::writeByte(uint8_t add, uint8_t value)
{
  Wire.begin();
  Wire.beginTransmission(_hwAdd);
  Wire.write(add);
  Wire.write(value);
  return Wire.endTransmission();
}
	
int SM_8CRT::readWord(uint8_t add, uint16_t* value)
{
	uint8_t buff[2];

	if (0 == value)
	{
		return -1;
	}
	//Wire.begin();
	Wire.beginTransmission(_hwAdd);
	Wire.write(add);
	if (Wire.endTransmission() != 0)
	{
		return -1;
	}
	Wire.requestFrom(_hwAdd, (uint8_t)2);
	if (2 <= Wire.available())
	{
		buff[0] = Wire.read();
		buff[1] = Wire.read();
	}
	else
	{
		return -1;
	}
	memcpy(value, buff, 2);
	return 0;
}

int SM_8CRT::readSWord(uint8_t add, int16_t* value)
{
	uint8_t buff[2];

	if (0 == value)
	{
		return -1;
	}
	//Wire.begin();
	Wire.beginTransmission(_hwAdd);
	Wire.write(add);
	if (Wire.endTransmission() != 0)
	{
		return -1;
	}
	Wire.requestFrom(_hwAdd, (uint8_t)2);
	if (2 <= Wire.available())
	{
		buff[0] = Wire.read();
		buff[1] = Wire.read();
	}
	else
	{
		return -1;
	}
	memcpy(value, buff, 2);
	return 0;
}

int SM_8CRT::writeWord(uint8_t add, uint16_t value)
{
	uint8_t buff[2];

	memcpy(buff, &value, 2);
	//Wire.begin();
	Wire.beginTransmission(_hwAdd);
	Wire.write(add);
	Wire.write(buff[0]);
	Wire.write(buff[1]);
	return Wire.endTransmission();

}

int SM_8CRT::writeSWord(uint8_t add, int16_t value)
{
	uint8_t buff[2];

	memcpy(buff, &value, 2);
	//Wire.begin();
	Wire.beginTransmission(_hwAdd);
	Wire.write(add);
	Wire.write(buff[0]);
	Wire.write(buff[1]);
	return Wire.endTransmission();

}

 
 

