/*
 * Copyright (c) 2025, Sensirion AG
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * * Neither the name of Sensirion AG nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef I_BLE_SERVICE_LIBRARY_H
#define I_BLE_SERVICE_LIBRARY_H
#include <functional>
#include <string>

namespace sensirion::upt::ble_server {

enum class Permission : uint8_t {
  READ_PERMISSION = 1 << 0,
  WRITE_PERMISSION = 1 << 1,
  NOTIFY_PERMISSION = 1 << 2
};

inline Permission operator|(Permission a, Permission b) {
  return static_cast<Permission>(static_cast<uint8_t>(a) |
                                 static_cast<uint8_t>(b));
}

inline Permission &operator|=(Permission &a, const Permission b) {
  a = a | b;
  return a;
}

inline bool operator==(Permission a, Permission b) {
  return (static_cast<uint8_t>(a) & static_cast<uint8_t>(b)) != 0;
}

using ble_service_callback_t = std::function<void(std::string)>;

class IBleServiceLibrary {
public:
  virtual ~IBleServiceLibrary() = default;

  virtual bool createService(const char *uuid) = 0;

  virtual bool startService(const char *uuid) = 0;

  virtual bool createCharacteristic(const char *serviceUuid,
                                    const char *characteristicUuid,
                                    Permission permission) = 0;

  virtual bool characteristicSetValue(const char *uuid, const uint8_t *data,
                                      size_t size) = 0;

  virtual bool characteristicSetValue(const char *uuid, int value) = 0;

  virtual bool characteristicSetValue(const char *uuid, uint32_t value) = 0;

  virtual bool characteristicSetValue(const char *uuid, uint64_t value) = 0;

  virtual std::string characteristicGetValue(const char *uuid) = 0;

  virtual bool characteristicNotify(const char *uuid) = 0;

  virtual void
  registerCharacteristicCallback(const char *uuid,
                                 const ble_service_callback_t &callback) = 0;
};

} // namespace sensirion::upt::ble_server

#endif // I_BLE_SERVICE_LIBRARY_H
