/*!\file arm_inlines_angle.h
** \author SMFSW
** \copyright MIT (c) 2017-2026, SMFSW
** \brief Angle conversion inlines
*/
/****************************************************************/
#ifndef ARM_INLINES_ANGLE_H_
	#define ARM_INLINES_ANGLE_H_

#ifdef __cplusplus
	extern "C" {
#endif

#include <math.h>
/****************************************************************/


#if !defined(M_SQRT2)
#define M_SQRT2		1.41421356237309504880		//!< sqrt(2) approximation constant
#endif

#if !defined(M_PI)
#define M_PI		3.14159265358979323846		//!< Pi approximation constant
#endif

#if !defined(M_TWOPI)
#define M_TWOPI		(M_PI * 2.0)				//!< 2*Pi approximation constant
#endif

#define M_PI_180	(M_PI / 180.0)				//!< Pi/180 approximation constant
#define M_180_PI	(180.0 / M_PI)				//!< 180/Pi approximation constant


/*!\brief Radians normalization (normalize angle to single rotation)
** \param[in] rad - Radians to convert
** \return Converted angle
**/
__INLINE float RAD_NORM(const float rad)
{
	const float val = (float) ((rad < 0.0F) ? M_TWOPI : -M_TWOPI);
	float r = rad;

	while (fabs(r) > (float) M_TWOPI)	{ r += val; }

	return r;
}


/*!\brief Degrees normalization (normalize angle to single rotation)
** \param[in] deg - Degrees to convert
** \return Converted angle
**/
__INLINE float DEG_NORM(const float deg)
{
	const float val = (deg < 0.0F) ? 360.0F : -360.0F;
	float d = deg;

	while (fabs(d) > 360.0F)	{ d += val; }

	return d;
}


#if !defined(ARDUINO)
/*!\brief Radians to degrees conversion
** \note On Arduino platform, RAD_TO_DEG is already defined as conversion value
** \param[in] rad - Radians to convert
** \return Converted angle
**/
__INLINE float RAD_TO_DEG(const float rad)
{
	return RAD_NORM(rad) * (float) M_180_PI;
}


/*!\brief Degrees to radians conversion
** \note On Arduino platform, DEG_TO_RAD is already defined as conversion value
** \param[in] deg - Degrees to convert
** \return Converted angle
**/
__INLINE float DEG_TO_RAD(const float deg)
{
	return DEG_NORM(deg) * (float) M_PI_180;
}
#endif


/*!\brief Radians to 0-1 scaled float conversion
** \param[in] rad - Radians to convert
** \return Converted angle
**/
__INLINE float RAD_TO_FLOAT(const float rad)
{
	return RAD_NORM(rad) / (float) M_TWOPI;
}

/*!\brief 0-1 scaled float to radians conversion
** \param[in] val - 0-1 scaled float to convert
** \return Converted angle
**/
__INLINE float FLOAT_TO_RAD(const float val)
{
	return (val - (float) ((intCPU_t) val)) * (float) M_TWOPI;
}


/*!\brief Degrees to 0-1 scaled float conversion
** \param[in] deg - Degrees to convert
** \return Converted angle
**/
__INLINE float DEG_TO_FLOAT(const float deg)
{
	return DEG_NORM(deg) / 360.0F;
}

/*!\brief 0-1 scaled float to degrees conversion
** \param[in] val - 0-1 scaled float to convert
** \return Converted angle
**/
__INLINE float FLOAT_TO_DEG(const float val)
{
	return (val - (float) ((intCPU_t) val)) * 360.0F;
}


/****************************************************************/
#ifdef __cplusplus
	}
#endif

#endif /* ARM_INLINES_ANGLE_H_ */
/****************************************************************/
