//
//    FILE: unit_test_001.cpp
//  AUTHOR: Rob Tillaart
//    DATE: 2023-05-02
// PURPOSE: unit tests for the LUHN checksum
//          https://github.com/RobTillaart/LUHN
//          https://github.com/Arduino-CI/arduino_ci/blob/master/REFERENCE.md
//

// supported assertions
// ----------------------------
// assertEqual(expected, actual)
// assertNotEqual(expected, actual)
// assertLess(expected, actual)
// assertMore(expected, actual)
// assertLessOrEqual(expected, actual)
// assertMoreOrEqual(expected, actual)
// assertTrue(actual)
// assertFalse(actual)
// assertNull(actual)


#include <ArduinoUnitTests.h>

#include "Arduino.h"
#include "LUHN.h"


unittest_setup()
{
    fprintf(stderr, "LUHN_LIB_VERSION: %s\n", (char *) LUHN_LIB_VERSION);
}

unittest_teardown()
{
}


unittest(test_constructor)
{
  LUHN luhn;

  assertEqual('0', luhn.reset());
  assertEqual('0', luhn.reset());
}


unittest(test_isValid)
{
  LUHN luhn;

  assertTrue(luhn.isValid("0"));
  assertTrue(luhn.isValid("79927398713"));
  assertTrue(luhn.isValid("1111111"));
  assertTrue(luhn.isValid("11111111111111111111"));

  //  generated by https://www.dcode.fr/luhn-algorithm
  assertTrue(luhn.isValid("08192186963920766401"));


  // https://www.mobilefish.com/services/credit_card_number_generator/credit_card_number_generator.php

  fprintf(stderr, "Example American Express credit card number\n");
  //  An American Express credit card number starts with number 34 or 37 and
  //  the credit card number has total 15 digits:
  assertTrue(luhn.isValid("377261620324999"));
  assertTrue(luhn.isValid("349854194206314"));

  fprintf(stderr, "Example Mastercard credit card number\n");
  //  A Mastercard credit card number starts with number 51, 52, 53, 54 or 55 and
  //  the credit card number has total 16 digits:
  assertTrue(luhn.isValid("5181975718047403"));
  assertTrue(luhn.isValid("5204571199083364"));
  assertTrue(luhn.isValid("5322683667269933"));
  assertTrue(luhn.isValid("5477754834149242"));
  assertTrue(luhn.isValid("5539624233693270"));

  fprintf(stderr, "Example Laser credit card number\n");
  //  A Laser credit card number starts with number 6304, 6706, 6771 or 6709 and
  //  the credit card number has total 16, 17, 18 or 19 digits:
  assertTrue(luhn.isValid("670676038979126821"));
  assertTrue(luhn.isValid("6771363087405086"));
  assertTrue(luhn.isValid("6304096514549839"));
  assertTrue(luhn.isValid("6304219447607087665"));
}


unittest(test_generateChecksum)
{
  LUHN luhn;

  assertEqual('3', luhn.generateChecksum("7992739871"));
}


unittest(test_generate)
{
  LUHN luhn;

  for (int i = 0; i < 10; i++)
  {
    char buffer[24];
    char prefix[10] = "007";
    assertTrue(luhn.generate(buffer, 20, prefix));
    assertTrue(luhn.isValid(buffer));
    fprintf(stderr, "%d %s\n", i, buffer);
  }
}


unittest(test_stream)
{
  LUHN luhn;

  luhn.reset();
  char buffer[24] = "7992739871";
  for (int i = 0; i < strlen(buffer); i++)
  {
    char c = luhn.add(buffer[i]);
    fprintf(stderr, "%d %c\n", i, c);
  }
  assertEqual('3', luhn.reset());
}




unittest_main()


//  -- END OF FILE --
