#ifndef RemoteXYAes_h
#define RemoteXYAes_h

#include "RemoteXYStream.h"


/*

https://github.com/kokke/tiny-AES-c/blob/master/aes.c
https://github.com/bigfighter/arduino-AES/blob/master/aes.h
https://rweather.github.io/arduinolibs/AES128_8cpp_source.html

*/

#define REMOTEXY_AES_BLOCK_SIZE 16
#define REMOTEXY_AES_KEY_SIZE 32

const uint8_t REMOTEXY_AES_SBOX[] PROGMEM = {
    0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76, 
    0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0, 
    0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15, 
    0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75, 
    0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84, 
    0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf, 
    0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8, 
    0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2, 
    0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73, 
    0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb, 
    0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79, 
    0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08, 
    0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a, 
    0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e, 
    0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf, 
    0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16
};
   
class CRemoteXYAes {
  private:
  
  uint8_t * pkey;

  public:
  uint8_t rj_xtime (uint8_t x) {
    return (x & 0x80) ? ((x << 1) ^ 0x1b) : (x << 1);
  }
  
  void subBytes (uint8_t * buf) {
    for (uint8_t i = 0; i < REMOTEXY_AES_BLOCK_SIZE; i++) {
      buf[i] = rxy_pgm_read_byte(REMOTEXY_AES_SBOX + buf[i]);
    }
  }
  
  void addRoundKey (uint8_t * buf, uint8_t * key) {
	for (uint8_t i = 0; i < REMOTEXY_AES_BLOCK_SIZE; i++) {
      buf[i] ^= key[i];
	}
  }
  

  void shiftRows (uint8_t * buf) {
    uint8_t i;
  
    i = buf[1];
    buf[1] = buf[5];
    buf[5] = buf[9];
    buf[9] = buf[13];
    buf[13] = i;
  
    i = buf[10];
    buf[10] = buf[2];
    buf[2] = i;
  
    i = buf[3];
    buf[3] = buf[15];
    buf[15] = buf[11];
    buf[11] = buf[7];
    buf[7] = i;
  
    i = buf[14];
    buf[14] = buf[6];
    buf[6]  = i;
  }

  void mixColumns (uint8_t* buf) {
    uint8_t i, a, b, c, d, e;
  
    for (i = 0; i < REMOTEXY_AES_BLOCK_SIZE; i += 4) {
      a = buf[i+0];
      b = buf[i+1];
      c = buf[i+2];
      d = buf[i+3];
      e = a ^ b ^ c ^ d;
      buf[i+0] ^= e ^ rj_xtime(a^b);
      buf[i+1] ^= e ^ rj_xtime(b^c);
      buf[i+2] ^= e ^ rj_xtime(c^d);
      buf[i+3] ^= e ^ rj_xtime(d^a);
    }
  }
  
  void keyExpansion (uint8_t * key, uint8_t * rc) {
    uint8_t i;
  
    key[0] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[29]) ^ (*rc);
    key[1] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[30]);
    key[2] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[31]);
    key[3] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[28]);
  
    *rc = ((*rc<<1) ^ (((*rc>>7) & 1) * 0x1b));
  
    for (i = 4; i < 16; i += 4) {
      key[i+0] ^= key[i-4];
      key[i+1] ^= key[i-3];
      key[i+2] ^= key[i-2];
      key[i+3] ^= key[i-1];
    }
    key[16] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[12]);
    key[17] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[13]);
    key[18] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[14]);
    key[19] ^= rxy_pgm_read_byte(REMOTEXY_AES_SBOX + key[15]);
  
    for (i = 20; i < 32; i += 4) {
      key[i+0] ^= key[i-4];
      key[i+1] ^= key[i-3];
      key[i+2] ^= key[i-2];
      key[i+3] ^= key[i-1];
    }
  }
  
  void setKey (uint8_t * key) {
    pkey = key;
  }

  // *buf (input 16 bytes) -> *buf (output 16 bytes)
  void encryptBlock (uint8_t * buf) {
    uint8_t i, rcon;
    uint8_t key[REMOTEXY_AES_KEY_SIZE]; 
  
    rxy_bufCopy (key, pkey, REMOTEXY_AES_KEY_SIZE);
    
    addRoundKey (buf, key);
    rcon = 1;
    for (i = 1; i < 14; ++i) {
      subBytes(buf);
      shiftRows(buf);
      mixColumns(buf);
      if (i & 1) {
        addRoundKey(buf, &key[16]);
      } 
      else {
        keyExpansion(key, &rcon);
        addRoundKey(buf, key);
      }
    }
    subBytes(buf);
    shiftRows(buf);
    keyExpansion(key, &rcon);
    addRoundKey(buf, key);
  }  
  
  
  
};

class CRemoteXYAesCbc : public CRemoteXYAes {
  public:
  uint8_t iv[REMOTEXY_AES_BLOCK_SIZE];
  
  CRemoteXYAesCbc () {
    uint8_t i = REMOTEXY_AES_BLOCK_SIZE;
    uint8_t *p = iv;
    while (i--) *p++ = (i+3) ^ (i << 4) ^ ((i+7) << 2);
  }
  
  // iv - 16 bytes
  void setIv (uint8_t * _iv) {
    rxy_bufCopy (iv, _iv, REMOTEXY_AES_BLOCK_SIZE);
  };
  
  void encryptBlockCbc (uint8_t * buf) {   
    for (uint8_t i = 0; i < REMOTEXY_AES_BLOCK_SIZE; i++) {
      buf[i] ^= iv[i];
    }
    encryptBlock (buf);
    rxy_bufCopy (iv, buf, REMOTEXY_AES_BLOCK_SIZE);
  }
 
};


class CRemoteXYAesCbcWriter {
  private:
  CRemoteXYAesCbc * aesCbc;
  CRemoteXYStream * stream;
  uint8_t block[REMOTEXY_AES_BLOCK_SIZE];
  uint8_t blockCnt;
  uint16_t dataSize;
  uint16_t crc;
  
  public:
  CRemoteXYAesCbcWriter (CRemoteXYStream * _stream, CRemoteXYAesCbc * _aesCbc) {
    aesCbc = _aesCbc;
    stream = _stream;
  }
  
  
  void writeStart (uint16_t size) {  
    blockCnt = 0;
    dataSize = size;  
    rxy_initCRC (&crc);
    stream->write (aesCbc->iv, REMOTEXY_AES_BLOCK_SIZE);
    writeToBlockUpdateCrc (size);
    writeToBlockUpdateCrc (size >> 8);
  }
  
  void write (uint8_t b) {    
    writeToBlockUpdateCrc (b);
    dataSize--;
    if (dataSize == 0) {
      writeToBlock (crc);
      writeToBlock (crc >> 8);
      while (blockCnt) {
        writeToBlock (++crc);
      }
      stream->flush();
    }
  }
  
  void write (uint8_t * buf, uint16_t size) {
    while (size--) write (*buf++);
  };   
  
  
  private:
  void writeToBlock (uint8_t b) {
    block[blockCnt++] = b;
    if (blockCnt >= REMOTEXY_AES_BLOCK_SIZE) {
      aesCbc->encryptBlockCbc (block);
      stream->write (block, REMOTEXY_AES_BLOCK_SIZE);
      blockCnt = 0;
    }
  }
  
  private:
  void writeToBlockUpdateCrc (uint8_t b) {
    rxy_updateCRC (&crc, b);
    writeToBlock (b);
  }
  
  public:
  static uint16_t getEncriptDataSize (uint16_t size) {
    size += 4;
    uint16_t blockCount = size / REMOTEXY_AES_BLOCK_SIZE;
    if (size % REMOTEXY_AES_BLOCK_SIZE) blockCount++;
    blockCount++;  // iv
    return blockCount * REMOTEXY_AES_BLOCK_SIZE;
  }  
  
};



CRemoteXYAesCbc RemoteXYAesCbc;

#endif // RemoteXYAes_h  