// Has this header been loaded before (Is __MOONMINSCANNER_H__ already defined?)
#ifndef __MOONMINSCANNER_H__

#include <Arduino.h>

// If not then load the header and NOW define it to prevent loading again.
#define __MOONMINSCANNER_H__

/*
  I2C slave address
*/
#define MOONMIN_I2C_SLAVE_ADDRESS 0x20

// Scanner type for future compatibility. We could technically use a different SPI module
#define SCANNER_TYPE_UNKNOWN 0x00;
#define SCANNER_TYPE_RFID 0x01;

/*
  Define the different available command bytes
*/
#define STATUS_COMMAND 0x00
#define STOP_COMMAND 0x01
#define EXTEND_COMMAND 0x02
#define RETRACT_COMMAND 0x03

/*
  Define the different status bytes for transmission
*/
#define STATUS_INIT 0x00
#define STATUS_ACTUATOR 0x01 
#define STATUS_MINERALSCAN 0x02

/*
  Define the different status values we can sent to the I2C connected device
*/
#define ACTUATOR_STATUS_IDLE_UNKNOWN_POSITION 0x00 // Unknown Position
#define ACTUATOR_STATUS_EXTENDED 0x01 // Full Extension
#define ACTUATOR_STATUS_RETRACTED 0x02// Fully Retracted
#define ACTUATOR_STATUS_EXTENDING 0x03 // EXTENDING Actuator
#define ACTUATOR_STATUS_RETRACTING 0x04 // RETRACTING actuator

/*
  Define the status types for a possible scan result
*/
#define SCANNER_RESULTS_NO_SCAN 0x00
#define SCANNER_RESULTS_NONE 0x01
#define SCANNER_RESULTS_FOUND 0x02

// Array Positions in the minerals array
#define OXYGEN      0
#define SILICON     1
#define IRON        2
#define MAGNESIUM   3
#define CALCIUM     4
#define ALUMINIUM   5
#define TOTAL_SCANNABLE_MATERIALS 6

class MoonMinScanner
{
  private:
  
    // What type of scanning module is attached.
    int mineralScannerModuleType = SCANNER_TYPE_UNKNOWN;

    // Did the module return that the scanner module is working?
    bool isScannerModuleWorking = false;

    // What version of firmware is running on the MoonMinScanner
    uint8_t connectedMoonMinVersion[3] { 0, 0, 0};

    // Send command via I2C function
    void SendCommand(uint8_t *command, int length);

    // Get response from buffer function
    int GetResponse(uint8_t *data, int len);

    // Storage for received status from the camera
    int actuatorStatus = ACTUATOR_STATUS_IDLE_UNKNOWN_POSITION;

    // Storage for received scanner information
    int scannerStatus = SCANNER_RESULTS_NO_SCAN;

    // What is the latest RFID card Unique Id that has been seen (if any is in range)
    String latestRFIDFound;

    // Function to convert bytes to hexidecimal string for RFID UID
    String ConvertIDToString(uint8_t *data, int len);

    // Mineral scan results
    uint8_t mineralResults[TOTAL_SCANNABLE_MATERIALS];

    // Were any mineral results found on the seen RFID card?
    bool mineralResultsFound = false;

  public:
    // Initialise the module (Grab firmware etc)
    bool Init();

    // Update the position data from the module
    bool Update();

    // Update the scan data from the module.
    bool UpdateScan();

    // Request the module extends into position
    void ExtendActuator();

    // Request the module retracts from position
    void RetractActuator();

    // Force stop. Emergency stop. (Unkown position)
    void ForceStop();
    
    // Current firmware version from the module.
    String GetFirmwareVersion();

    // Did the scanner module initialise on the MoonMin module. If not check 7 pin connector wiring on the Moonmin scanner.
    bool GetIsScannerModuleWorking();

    // Returns the latest status retrieved in Update()
    int GetActuatorStatus();

    // Returns the latest scanner results status retrieved in UpdateScan()
    int GetScannerStatus();

    // Returns the Hexidecimal string version of the RDID cards Unique ID (If present)
    String GetRFIDFromResults();

    // Returns the value of the specified mineral type seen in the current scan (If present, if not '0')
    uint8_t GetMineralResult(int mineralType);

    // Did the last scan find any mineral results?
    bool DidFindMinerals();
};

#endif