#include "MoonMinScanner.h"
#include <Wire.h> // I2C

/*
 Initialise the MoonMin Scanner module. Get the firmware version and status of the Scanner module.
*/
bool MoonMinScanner::Init()
{
  // Initialise and get the firmware to check we are connected
  Wire.begin();

  // Request firmware version to verify connection
  uint8_t command[2] = {STATUS_COMMAND, STATUS_INIT};

  // Send the comand
  SendCommand(command, sizeof(command));

  // Wait for the module to compile a response.
  delayMicroseconds(100);

  // Request the response
  uint8_t response[7];
  int size = this->GetResponse(response, sizeof(response));

  // Is it the correct response?
  if(size > 4)
  {
    if(response[0] == STATUS_COMMAND && response[1] == STATUS_INIT)
    {
      // What firmware version is running on the module.
      this->connectedMoonMinVersion[0] = response[2];
      this->connectedMoonMinVersion[1] = response[3];
      this->connectedMoonMinVersion[2] = response[4];

      // Mineral Scanner module type (Only one type at the moment, Todo add handling when needed)
      this->mineralScannerModuleType = response[5];

      // Status of the scanner module (RFID Module)
      this->isScannerModuleWorking = response[6];
      return true;
    }
  }

  // Not a valid response from the camera.
  return false;
}

/*
 Update the moonmin scanner position information by retrieving it from the module (I2C)
*/
bool MoonMinScanner::Update()
{
  // Get the latest status from the sensor
  uint8_t request[2] = { STATUS_COMMAND, STATUS_ACTUATOR};
  this->SendCommand(request, 2);

  // Wait for the module to compile a response.
  delayMicroseconds(100);

  uint8_t response[3];

  // Grab the response.
  int size = this->GetResponse(response, sizeof(response));
  if(size == 3)
  {
    // Make sure we are in sync
    if(response[0] == STATUS_COMMAND && response[1] == STATUS_ACTUATOR)
    {
      // Get the status of the Actuator
       this->actuatorStatus = response[2];
      return true;
    }
  }

  // Failed to get status
  return false;
}

/*
  Update the scanner information by retrieving it from the module (I2C)
*/
bool MoonMinScanner::UpdateScan()
{
  // Get the latest status from the sensor
  uint8_t request[2] = { STATUS_COMMAND, STATUS_MINERALSCAN};
  this->SendCommand(request, 2);
  
  // Wait for the module to compile a response.
  delayMicroseconds(100);
  
  uint8_t response[100];

  // Grab the response
  int size = this->GetResponse(response, sizeof(response));
  if(size > 10)
  {
    // Make sure we are in sync
    if(response[0] == STATUS_COMMAND && response[1] == STATUS_MINERALSCAN)
    {
      // Extract the status
      this->scannerStatus = response[2];

      // If we have results available extract the details
      if(this->scannerStatus == SCANNER_RESULTS_FOUND)
      {
        this->latestRFIDFound = this->ConvertIDToString(response + 4, response[3]);
      }

      int readPos = 4 + response[3];

      // Extract any mineral results
      if(response[readPos] == 1)
      {
        // Flag that a card with mineral results was found.
        this->mineralResultsFound = true;
        
        readPos++;

        // Found minerals
        for(int x = 0; x < TOTAL_SCANNABLE_MATERIALS; x++)
        {
          this->mineralResults[x] = response[readPos];
          readPos++;
        }
      }
      else
      {
        // No minerals on the card
        this->mineralResultsFound = false;
      }

      return true;
    }
  }

  // Failed to get status
  return false;
}

/*
  Convert an RFID Unique ID to a human readable hexidecimal string
*/
String MoonMinScanner::ConvertIDToString(uint8_t *data, int len)
{
  // Convert the ID to a string so that we can compare it and read it
  String newCardID = "";
  
  for(int x = 0; x < len; x++)
  {
    // Convert each byte of the ID number to a hex string
    // We don't technically have to do this, we could compare each byte but
    // this is easier to understand
    newCardID += String(data[x], HEX);
    
    // Add a space between each HEX character to aid with reading
    newCardID += " ";
  }

  // Uppercase to make it look nicer. We don't have to do this
  newCardID.toUpperCase();
  return newCardID;
}

/*
  Get the reading for a specified mineral type. See MoonMinScanner.H for available mineral types.
*/
uint8_t MoonMinScanner::GetMineralResult(int mineralType)
{
  // Found any minerals on the RFID tag?
  if(this->mineralResultsFound)
  {
    // Within known mineral types range?
    if(mineralType < TOTAL_SCANNABLE_MATERIALS)
    {
      return this->mineralResults[mineralType];
    }
  }

  // None found.
  return 0;
}

/*
  Return the current firmware version running on the module in a human readable format.
*/
String MoonMinScanner::GetFirmwareVersion()
{
  return String(this->connectedMoonMinVersion[0]) + String(":") + String(this->connectedMoonMinVersion[1]) + String(":") + String(this->connectedMoonMinVersion[2]);
}

/*
  Get whether the scanner module is working on the MoonMin scanner module. Check 7 pin connector and call init again if it is failing.
*/
bool MoonMinScanner::GetIsScannerModuleWorking()
{
  return this->isScannerModuleWorking;
}

/*
  Returns the current status of the linear actuator on the module. (Don't forget to call Update())
*/
int MoonMinScanner::GetActuatorStatus()
{
  return this->actuatorStatus;
}

/*
  Returns the current status if the MoonMin scanners 'Scanner Module'; RFID (Don't forget to call UpdateScanner())
*/
int MoonMinScanner::GetScannerStatus()
{
  return this->scannerStatus;
}

/*
  Did the scanner find mineral results on its last scan?
*/
bool MoonMinScanner::DidFindMinerals()
{
  return this->mineralResultsFound;
}

/*
  Returns the Unique ID of the RFID card/tag. If present.
*/
String MoonMinScanner::GetRFIDFromResults()
{
  return this->latestRFIDFound;
}

/*
  Request the MoonMin Scanner extends the actuator (Arm).
*/
void MoonMinScanner::ExtendActuator()
{
  // Send I2C Command
  uint8_t command[1] = { EXTEND_COMMAND };
  SendCommand(command, sizeof(command));
}

/*
  Request the MoonMin Scanner retracts the actuator (Arm).
*/
void MoonMinScanner::RetractActuator()
{
  // Send I2C Command
  uint8_t command[1] = { RETRACT_COMMAND };
  SendCommand(command, sizeof(command));
}

/*
  Force the actuator to stop in its current position (Emergency Stop). The position will be unknown.
*/
void MoonMinScanner::ForceStop()
{
  uint8_t command[1] = {STOP_COMMAND};
  SendCommand(command, sizeof(command));
}

/* 
  Private function to send a command via I2C to the module. 
*/
void MoonMinScanner::SendCommand(uint8_t *command, int length)
{
  Wire.beginTransmission(MOONMIN_I2C_SLAVE_ADDRESS);
  Wire.write(command, length);
  Wire.endTransmission();
}

/*
  Private function to decode the I2C response from the Module
*/
int MoonMinScanner::GetResponse(uint8_t *data, int len)
{
  // Request the response
  Wire.requestFrom(MOONMIN_I2C_SLAVE_ADDRESS, len);
  int readBytes = Wire.available();
  
  if(readBytes > 0)
  {
    for(int x = 0; x < readBytes; x++)
    {
      data[x] = Wire.read();
    }
  }

  return readBytes;
}