/*
  This example program connects the the Moonmin scanner and initialises it.

  Use the following commands typed into the serial input
  'e' - To extend the actuator into position.
  'r' - To retract the actuator into position.
  's' - To force stop.
*/

#include "MoonMinScanner.h"

MoonMinScanner scanner;

void setup() {
  // put your setup code here, to run once:
  Serial.begin(19200);

  // Initialise the MoonMin scanner module.
  Serial.println("Initialising MoonMin Scanner");
  if(scanner.Init())
  {
    Serial.print("Scanner Initialised Ok.\r\n Firmware Version: ");
    Serial.println(scanner.GetFirmwareVersion());

    if(scanner.GetIsScannerModuleWorking())
    {
      Serial.println("RFID Module Is Ready.");
    }
    else
    {
      Serial.println("RFID Module Failed to load. Check the 7 pin connector!!!");
    }

    // Reset by returning home
    scanner.RetractActuator();
  }
  else
  {
    Serial.println("Moonmin Scanner Failed to initialise. Check wires and power");
  }
}

void loop() {

  // Update the scanner regularly to ensure it checks its position and updates (e.g starts/stops motors)
  if(scanner.Update())
  {
    // Used for serial input and allowing control from the serial window.
    CheckForIncommingSerial();
    
    // If it returns true then we are connected and we have the latest status of the Moonmin scanner.
    // Show status update
    Serial.print("Current Actuator Status: ");
    switch(scanner.GetActuatorStatus())
    {
      case ACTUATOR_STATUS_IDLE_UNKNOWN_POSITION:
        Serial.println("Idle in unknown position"); // Unknown Position
        break;

      case ACTUATOR_STATUS_EXTENDED:
        Serial.print("At Full Extent, Scanning..."); // At Full Extension
        PerformMineralScan();
        break;

      case ACTUATOR_STATUS_RETRACTED:
        Serial.println("Fully retracted"); // Actuator is Fully Retracted
        break;

      case ACTUATOR_STATUS_EXTENDING:
        Serial.println ("Extending into position..."); // Actuator is extending. (Motor active)
        break;

      case ACTUATOR_STATUS_RETRACTING:
        Serial.println ("Retracting into position..."); // Actuator is retracting (Motor active)
        break;      
    }    
  }
  
  // Slow down the output
  delay(100);
}

/*
  Get the scan results from the RFID module on the MoonMin scanner.
*/
void PerformMineralScan()
{
  // If we at full extension request scan results
  if (scanner.UpdateScan())
  {
    switch(scanner.GetScannerStatus())
    {
      case SCANNER_RESULTS_NO_SCAN:
        // Not scanning
        Serial.println();
        break;

      case SCANNER_RESULTS_NONE:
        // None RFID card / tag Found
        Serial.println("None");
        break;

      case SCANNER_RESULTS_FOUND:
        // A tag was found. Did it have any minerals programmed onto it?
        if(scanner.DidFindMinerals())
        {
          // Results Found. Lets print the values read from the card.
          // Below is the list of available elements (See MoonMinScanner.h)
          // OXYGEN      0
          // SILICON     1
          // IRON        2
          // MAGNESIUM   3
          // CALCIUM     4
          // ALUMINIUM   5
          Serial.print("O2: ");
          Serial.print(scanner.GetMineralResult(OXYGEN));
          Serial.print("mg/cm3\tSi: ");
          Serial.print(scanner.GetMineralResult(SILICON));
          Serial.print("mg/cm3\t Fe: ");
          Serial.print(scanner.GetMineralResult(IRON));
          Serial.print("mg/cm3\t Mg: ");
          Serial.print(scanner.GetMineralResult(MAGNESIUM));
          Serial.print("mg/cm3\t Ca: ");
          Serial.print(scanner.GetMineralResult(CALCIUM));
          Serial.print("mg/cm3\t Au: ");
          Serial.print(scanner.GetMineralResult(ALUMINIUM));
          Serial.println("mg/cm3");
        }
        else
        {
          // No minerals were found. Just show the tags Unique ID for reference.
          Serial.print("RFID Card Found: ");
          Serial.println(scanner.GetRFIDFromResults());
        }
        break;
    } 
  }
}

/* This function allows for controlling the camera through the Serial monitor.
// e: extend the actuator arm into position.
// r: retract the actuator arm into position.
// 's': force stop.
*/
void CheckForIncommingSerial()
{
  if (Serial.available())
  {
      uint8_t val = Serial.read();

      switch (val)
      {
        case 'e':
            scanner.ExtendActuator();
          break;

        case 'r':
          scanner.RetractActuator();
          break;

        case 's': // Stop
          scanner.ForceStop();
          break;
      }
  }
}