#include "./Packet_Device.h"
#include <algorithm>
#include "Packet_Device.h"

template <typename R, uint16_t N>
void DevicePacket<R, N>::enableBulkRead(bool state)
{
  bulk_read_enabled = state;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setReceiver(std::map<String, void (*)(String, String)> receivers)
{
  insert_pram_data_cmnds = receivers;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setReceiver(std::map<String, void (*)(String)> receivers, bool prams)
{
  if (prams)
    get_pram_cmnds = receivers;
  else
    insert_data_cmnds = receivers;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setReceiver(std::map<String, void (*)()> receivers)
{
  get_process_cmnds = receivers;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setReceiver(std::map<String, void (*)(R *, uint8_t, uint16_t, uint16_t)> receivers)
{
  get_response_buff = receivers;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::onReceive(String name, void (*fun)(String, String))
{
  insert_pram_data_cmnds[name] = fun;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::onReceive(String name, void (*fun)(String), bool prams)
{
  if (prams)
    get_pram_cmnds[name] = fun;
  else
    insert_data_cmnds[name] = fun;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::onReceive(String name, void (*fun)())
{
  get_process_cmnds[name] = fun;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::onReceive(String name, void (*fun)(R *, uint8_t, uint16_t, uint16_t))
{
  get_response_buff[name] = fun;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::commandProcess(R *data, uint16_t len)
{

  // Serial.println("Receive:" + String(len));
  // Serial.flush();

  if (len >= 6 && data[0] == TRANSFER_DATA_BUFFER_SIG && data[1] == BUFFER_TEXT_RESPNOSE)
  {
    // TRANSFER_DATA_TEXT_HEADER_LEN+CRC_SIZE(2 bytes)=6
    // with crc
    if (verifyCRC<uint8_t>((uint8_t *)data, len))
    {
      len -= 2; // reduce crc
      uint8_t data_len_msb = data[2];
      uint8_t data_len_lsb = data[3];
      uint16_t data_len = ((data_len_msb << 8) | data_len_lsb) & 0xFFFF;
      if (data_len + TRANSFER_DATA_TEXT_HEADER_LEN <= len)
      {
        String text = String(data + TRANSFER_DATA_TEXT_HEADER_LEN, data_len);
        if (get_process_cmnds.find(text) != get_process_cmnds.end())
        {
          // Call the function if the key is found
          get_process_cmnds[text]();
        }
      }
    }
  }
  else if (len >= 8 && data[0] == TRANSFER_DATA_BUFFER_SIG && data[1] == BUFFER_PARAM_RESPNOSE)
  {
    // TRANSFER_DATA_PARAMS_HEADER_LEN+CRC_SIZE(2 bytes)=8
    //  Serial.println("Prams:"+String(len)+",type:"+String((uint8_t)data[4]));
    if (verifyCRC<uint8_t>((uint8_t *)data, len))
    {
      len -= 2; // reduce crc
      uint8_t data_type = data[2];
      uint8_t pram_len = data[3];
      uint8_t data_len_msb = data[4];
      uint8_t data_len_lsb = data[5];
      uint16_t data_len = ((data_len_msb << 8) | data_len_lsb) & 0xFFFF;

      // Serial.println("data_type:"+String(data_type)+",pram_len:"+String(pram_len)+",data_len:"+String(data_len));

      if (pram_len + data_len + TRANSFER_DATA_PARAMS_HEADER_LEN <= len)
      {
        String param = String(data + TRANSFER_DATA_PARAMS_HEADER_LEN, pram_len);

        // Serial.println("Data Pram-->"+param);

        // for(uint8_t i=5 + pram_len;i<len;i++){
        //   Serial.print(" "+String(data[i],HEX));
        // }
        // Serial.println();

        if (any_response_buff.find(param) != any_response_buff.end())
        {
          any_response_buff[param](data + (TRANSFER_DATA_PARAMS_HEADER_LEN + pram_len), data_type, data_len, 1); // single data
        }
        else if (get_response_buff.find(param) != get_response_buff.end())
        {
          // Call the function if the key is found
          get_response_buff[param](data + (TRANSFER_DATA_PARAMS_HEADER_LEN + pram_len), data_type, data_len, 1); // single data
        }
      }
    }
  }
  else if (len >= 9 && data[0] == TRANSFER_DATA_BUFFER_SIG && data[1] == BUFFER_ARRY_RESPNOSE)
  {
    // TRANSFER_DATA_ARRAY_HEADER_LEN+CRC_SIZE(2 bytes)=9
    //  Serial.println("Array:"+String(len));
    if (verifyCRC<uint8_t>((uint8_t *)data, len))
    {
      len -= 2; // reduce crc
      uint8_t data_type = data[2];
      uint8_t type_size = data[3];
      uint8_t pram_len = data[4];
      uint8_t data_size_msb = data[5];
      uint8_t data_size_lsb = data[6];
      uint16_t data_size = ((data_size_msb << 8) | data_size_lsb) & 0xFFFF;
      uint16_t data_len = type_size * data_size;
      if (pram_len + data_len + TRANSFER_DATA_ARRAY_HEADER_LEN <= len)
      {
        String param = String(data + TRANSFER_DATA_ARRAY_HEADER_LEN, pram_len);
        // Serial.println(param);
        // Serial.println(get_response_buff.size());
        if (any_response_buff.find(param) != any_response_buff.end())
        {
          any_response_buff[param](data + (TRANSFER_DATA_ARRAY_HEADER_LEN + pram_len), data_type, type_size, data_size);
        }
        else if (get_response_buff.find(param) != get_response_buff.end())
        {
          // Call the function if the key is found
          get_response_buff[param](data + (TRANSFER_DATA_ARRAY_HEADER_LEN + pram_len), data_type, type_size, data_size);
        }
      }
    }
  }
  else
  {
    String cmd = String(data, len);
    uint16_t cmd_len = cmd.length();

    // Serial.println(cmd);
    // Serial.flush();

    if (cmd_len > 6 && cmd[3] == ':' && cmd[5] == '=')
    {
      // for COMMAND:PRAM=DATA
      // if value setting command happen
      String f_cmd = cmd.substring(0, 3);
      // Check if the key exists in the map
      if (insert_pram_data_cmnds.find(f_cmd) != insert_pram_data_cmnds.end())
      {

        String m_cmd = cmd.substring(4, 5);
        String s_cmd = cmd.substring(6);

        // Call the function if the key is found
        insert_pram_data_cmnds[f_cmd](m_cmd, s_cmd);
      }
    }
    else if (cmd_len > 4 && cmd[3] == '=')
    {
      // for COMMAND=DATA
      // if value setting command happen
      String f_cmd = cmd.substring(0, 3);

      if (insert_data_cmnds.find(f_cmd) != insert_data_cmnds.end())
      {

        String s_cmd = cmd.substring(4);
        // Call the function if the key is found
        insert_data_cmnds[f_cmd](s_cmd);
      }
    }
    else if (cmd_len > 4 && cmd[3] == ':')
    {
      // for COMMAND:PRAM
      // if value setting command happen
      String f_cmd = cmd.substring(0, 3);

      if (get_pram_cmnds.find(f_cmd) != get_pram_cmnds.end())
      {
        String s_cmd = cmd.substring(4);
        // Call the function if the key is found
        get_pram_cmnds[f_cmd](s_cmd);
      }
    }
    else
    {
      if (get_process_cmnds.find(cmd) != get_process_cmnds.end())
      {
        // Call the function if the key is found
        get_process_cmnds[cmd]();
      }
    }
  }
}

template <typename R, uint16_t N>
bool DevicePacket<R, N>::queueCheck()
{
  if (commpleted_cmd_read)
  {
    this->receiver_lock();
    current_commands_length = 0; // it is restriction to write a variable from two different thread
    commpleted_cmd_read = false;
    packet_length = 0;
    this->receiver_unlock();
  }

  // if the queue is full then we will not process any receving buffer untill the queue read
  if (current_commands_length >= max_command_queue_length)
    return false;

  // full packet receive timeout check
  if (packet_timeout_at != 0 && packet_length != 0 && millis() > packet_timeout_at)
  {

    Command_t<R, N> *cmd = &(commands_holder[current_commands_length]);

    this->receiver_lock();
    // Serial.println("timeout:"+String(cmd->len)+",t:"+String( millis()-packet_timeout_at));
    cmd->len = 0;
    packet_length = 0;     // reset packet receiveing
    packet_timeout_at = 0; // reset the time checker, and
    this->receiver_unlock();
  }

  return true;
}

template <typename R, uint16_t N>
bool DevicePacket<R, N>::processEachData(R inchar)
{
  // TODO: remove debug print
  // Serial.printf("%c: %d or %02X \r\n",inchar, inchar,inchar);

  Command_t<R, N> *cmd = &(commands_holder[current_commands_length]);

  // store data
  this->receiver_lock();
  cmd->data[cmd->len] = inchar;
  cmd->len++;

  if (cmd->len >= N)
  {
    cmd->len = 0;
  }
  this->receiver_unlock();

  if (packet_length != 0)
  {
    // packet receiving mode
    if (cmd->len == packet_length)
    {
      // Serial.println("Data:"+String(cmd->data[0],HEX));

      // reset the packet receive
      packet_length = 0;
      // packet is ready for process
      packet_timeout_at = 0; // reset timeout

      this->receiver_lock();
      cmd->completed = true;     // mark it as completed
      current_commands_length++; // store for the next
      this->receiver_unlock();

      if (current_commands_length >= max_command_queue_length)
      {
        return false; // if the queue if full then not process any more receive
      }
    }
  }
  else
  {
    // non macket mode
    if (cmd->len >= PACKET_SIGNETURE_LEN)
    {
      size_t offset = cmd->len - PACKET_SIGNETURE_LEN;

      // Serial.println("offset:"+String(offset)+",len:"+String(cmd->len)+",l:"+String(PACKET_SIGNETURE_LEN));
      uint16_t packet_size = getPacketLength((uint8_t *)(cmd->data + offset));

      if (packet_size != 0)
      {
        // valid match
        this->receiver_lock();
        cmd->len = 0; // reset buffer index for making ready to receive actual buffer
        this->receiver_unlock();

        if (packet_size < N)
        {
          // Serial.println("Received:"+String(packet_size)+",l:"+String(current_commands_length));
          // packet size is valid
          packet_length = packet_size; // update packet size
          // register current time to register timeout of receving data
          packet_timeout_at = millis() + (packet_size * 2) + 100;
          // minimum baud rate could 4800bps that mean 600bytes for second, considering 2ms for each of byte, and some extra delay (100ms)
        }

        return true; // no more process until next byte receive
      }
    }

    if (cmd->len >= delimeter_len)
    {
      size_t offset = cmd->len - delimeter_len;
      // if data match for deliemter
      // if(std::equal(cmd->data+offset,cmd->data+cmd->len,delimeters)){

      if (memcmp(cmd->data + offset, delimeters, delimeter_len) == 0)
      {
        this->receiver_lock();
        // reset the packet receive
        packet_length = 0;

        cmd->len = offset; // orginal data length
        cmd->completed = true;
        current_commands_length++; // store for the next
        this->receiver_unlock();

        if (current_commands_length >= max_command_queue_length)
        {
          return false; // if the queue if full then not process any more receive
        }
      }
    }
  }

  return true;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::processBytes(R *all_bytes, size_t len)
{
  for (size_t x = 0; x < len; x++)
  {
    if (current_commands_length >= max_command_queue_length)
    {
      // if the queue if full then not process any more receive untill the queue read
      uint32_t start_time = (xTaskGetTickCount() * portTICK_PERIOD_MS); // gives ms time used for timeout
      // maximum wait for 1 second
      while (this->queueCheck() == false && (xTaskGetTickCount() * portTICK_PERIOD_MS - start_time) < 1000)
      {
        // wait until queue has space
        vTaskDelay(pdMS_TO_TICKS(10)); // wait for 10ms
      }
    }
    this->processEachData(all_bytes[x]); // if the queue full it return false
  }
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::feedBytes(R *all_bytes, size_t len)
{
  if (!this->queueCheck())
    return;
  this->processBytes(all_bytes, len);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::readSerialCommand()
{
  if (serial_dev == nullptr)
    return;
  // command receiving from receiving thread
  if (!this->queueCheck())
    return;

  if (bulk_read_enabled)
  {
    // if bulk read enabled
    while (true)
    {
      int avail = serial_dev->available();
      if (avail <= 0)
        break; // no more data

      size_t to_read = std::min((size_t)avail, (size_t)N);

      R all_bytes[to_read];
      size_t working_bytes = serial_dev->readBytes((uint8_t *)all_bytes, to_read); // size_t HardwareSerial::read(uint8_t *buffer, size_t size)

      this->processBytes(all_bytes, working_bytes);
    }
  }
  else
  {
    while (serial_dev->available())
    {
      if (!this->processEachData(serial_dev->read()))
        break; // if the queue if full then not process any more receive
    }
  }
  // Serial.println();
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::processingQueueCommands()
{
  // command process from listening thread
  if (current_commands_length > 0 && commpleted_cmd_read == false)
  { // if only the queue has data
    for (uint8_t i = 0; i < current_commands_length; i++)
    {
      Command_t<R, N> *cmd = &(commands_holder[i]);
      if (cmd->completed)
      {
        commandProcess(cmd->data, cmd->len); // process the command
        this->receiver_lock();
        cmd->completed = false;
        this->receiver_unlock();
      }
      this->receiver_lock();
      // restore default: when writing to that it is ensure that other thread is not writing in this
      cmd->len = 0;
      this->receiver_unlock();
    }
    // current_commands_length = 0;
    this->receiver_lock();
    commpleted_cmd_read = true;
    this->receiver_unlock();
  }
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setDevicePort(Stream *serial)
{
  serial_dev = serial;
}

template <typename R, uint16_t N>
bool DevicePacket<R, N>::getBufferMode()
{
  return response_buffer_mode;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setBufferMode(bool state)
{
  response_buffer_mode = state;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::setAutoFlush(bool state)
{
  auto_flush = state;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::flushDataPort()
{
  if (serial_dev == nullptr)
    return;
  // thread safe flush
  this->writer_lock();
  serial_dev->flush();
  this->writer_unlock();
}

template <typename R, uint16_t N>
bool DevicePacket<R, N>::writeToPort(uint8_t *buff, uint16_t size)
{
  if (serial_dev == nullptr)
    return false;

  // thread safe write
  this->writer_lock();
  serial_dev->write(buff, size);
  this->writer_unlock();

  if (auto_flush)
    flushDataPort();
  return true;
}

template <typename R, uint16_t N>
uint16_t DevicePacket<R, N>::getPacketLength(uint8_t *transfer_buff)
{
  if (transfer_buff[0] != packet_info[0] || transfer_buff[PACKET_SIGNETURE_LEN - 1] != packet_info[PACKET_SIGNETURE_LEN - 1])
    return 0;

  uint16_t packet_size = 0;
  for (uint8_t i = 1; i < (PACKET_SIGNETURE_LEN - 1); i++)
  {
    if (packet_info[i] == 0x0F)
    {
      // data
      if (transfer_buff[i] > 0x0F)
      {
        // if segment is invalid
        return 0;
      }
      packet_size = (packet_size << 4) | transfer_buff[i];
    }
    else if (packet_info[i] != transfer_buff[i])
    {
      // format is not matching
      return 0;
    }
  }

  return packet_size;
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::updatePacketLength(uint8_t *transfer_buff, uint16_t packet_size)
{
  memcpy(transfer_buff, packet_info, PACKET_SIGNETURE_LEN);
  // Serial.printf("Updating packet length: %d \r\n", packet_size);
  //{ (packet_size & 0xF000) >> 12, (packet_size & 0x0F00) >> 8, (packet_size & 0x00F0) >> 4, packet_size & 0x000F }
  for (uint8_t i = 0; i < PACKET_SIGNETURE_DATA_LEN; i++)
  {
    transfer_buff[(i * 2) + 1] = (packet_size >> (12 - (i * 4))) & 0x0F;
    // Serial.printf("Len byte %d : %02X \r\n", (i * 2) + 1, transfer_buff[(i * 2) + 1]);
  }
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::dataOutToSerial(uint8_t *buff, uint16_t size, uint8_t *header, uint8_t header_size)
{
  if (serial_dev == nullptr && size == 0)
    return;

  uint16_t crc = header_size > 0 ? getCRC<uint8_t>(header, header_size) : 0;
  crc = getCRC<uint8_t>(buff, size, crc);

  if (response_buffer_mode)
  {
    uint8_t crc_bytes[CRC_BYTE_LEN] = {(uint8_t)(crc >> 8), (uint8_t)crc};
    uint16_t packet_size = header_size + size + CRC_BYTE_LEN;

    uint8_t transfer_buff[PACKET_SIGNETURE_LEN];
    updatePacketLength(transfer_buff, packet_size);

    this->writer_lock();
    serial_dev->write(transfer_buff, PACKET_SIGNETURE_LEN);
    if (header_size > 0)
      serial_dev->write(header, header_size);
    serial_dev->write(buff, size);
    serial_dev->write(crc_bytes, CRC_BYTE_LEN);
    this->writer_unlock();
  }
  else
  {
    uint8_t end_bytes[CRC_BYTE_LEN + delimeter_len] = {(uint8_t)(crc >> 8), (uint8_t)crc};
    if (delimeter_len > 0)
      memcpy(end_bytes + CRC_BYTE_LEN, delimeters, delimeter_len);

    this->writer_lock();
    if (header_size > 0)
      serial_dev->write(header, header_size);
    serial_dev->write(buff, size);
    serial_dev->write(end_bytes, CRC_BYTE_LEN + delimeter_len);
    this->writer_unlock();
  }

  if (auto_flush)
    flushDataPort();
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::dataOutToSerial(String str)
{
  // uint16_t str_len=str.length();
  // uint8_t buff[str_len+1];
  // str.toCharArray((char *)buff,str_len+1);
  dataOutToSerial((uint8_t *)str.c_str(), str.length());
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutStr(String properties, String payload)
{
  restOut(properties, payload);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutFloat(String properties, float payload)
{
  restOut(properties, payload);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutInt(String properties, int payload)
{
  restOut(properties, payload);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutHex(String properties, uint32_t payload)
{
  restOut(properties, payload);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutBin(String properties, uint32_t payload)
{
  restOut(properties, payload);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutSuccess(String payload)
{
  restOutStr("payload", payload);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::restOutError(String err)
{
  restOutStr("error", err);
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::writer_lock()
{
#if defined(ARDUINO_ARCH_ESP32) || defined(ESP32) || defined(FREERTOS) || defined(configUSE_PREEMPTION)
  xSemaphoreTake(this->writter_locker, portMAX_DELAY);
#endif
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::writer_unlock()
{
#if defined(ARDUINO_ARCH_ESP32) || defined(ESP32) || defined(FREERTOS) || defined(configUSE_PREEMPTION)
  xSemaphoreGive(this->writter_locker);
#endif
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::receiver_lock()
{
#if defined(ARDUINO_ARCH_ESP32) || defined(ESP32) || defined(FREERTOS) || defined(configUSE_PREEMPTION)
  xSemaphoreTake(this->receiver_locker, portMAX_DELAY);
#endif
}

template <typename R, uint16_t N>
void DevicePacket<R, N>::receiver_unlock()
{
#if defined(ARDUINO_ARCH_ESP32) || defined(ESP32) || defined(FREERTOS) || defined(configUSE_PREEMPTION)
  xSemaphoreGive(this->receiver_locker);
#endif
}

// Explicit instantiation for specific types
template class DevicePacket<char, MAX_COMMAND_DEFAULT_LEN>; // Instantiating DevicePacket<char, MAX_COMMAND_DEFAULT_LEN>
