![CH32X035 USB HID Mouse Library](img/title.png)

# CH32X035 USB HID Mouse Library

A high-performance, efficient USB HID Mouse library tailored for the **WCH CH32X035** RISC-V microcontroller series.

This library leverages the hardware USBFS peripheral and synchronized interrupts to provide a robust, smooth mouse emulation interface. It allows your CH32X035 to control the cursor and buttons on any computer (Windows, macOS, Linux, Android) just like a standard physical mouse, with no drivers required.

## Features

*   **Class Compliant:** Works natively as a standard USB Mouse on all major operating systems.
*   **Hardware Synchronization:** Uses interrupt flags to synchronize data transfer with the USB host, ensuring smooth cursor movement without packet loss.
*   **High Performance:** Capable of sending movement updates at maximum USB polling speed.
*   **Full Control:** Support for Left, Right, and Middle clicks, plus relative X/Y movement and scroll wheel.

## API Overview

The library adheres to the standard Arduino `Mouse` API conventions while adding specific enhancements for the CH32X035 architecture.

| Function | Description |
| :--- | :--- |
| `Mouse.begin()` | Initializes the USB stack and enumerates the device. |
| `Mouse.click(button)` | Clicks and immediately releases a button. |
| `Mouse.move(x, y, wheel)` | Moves the cursor relatively (x, y) and scrolls (wheel). |
| `Mouse.press(button)` | Holds a mouse button down (for dragging). |
| `Mouse.release(button)` | Releases a specific mouse button. |
| `Mouse.isPressed(button)` | Checks if a button is currently held down. |
| `Mouse.setDelay(ms)` | Adds a safety delay (in ms) between reports. |

## Supported Buttons

| Constant | Description |
| :--- | :--- |
| `MOUSE_LEFT` | Left Click |
| `MOUSE_RIGHT` | Right Click |
| `MOUSE_MIDDLE` | Scroll Wheel Click |

## Installation

1.  Download this repository as a ZIP file.
2.  Open the Arduino IDE.
3.  Go to **Sketch** -> **Include Library** -> **Add .ZIP Library...**
4.  Select the downloaded ZIP file.
5.  Restart the IDE.

## Usage Guide

### 1. Basic Setup

Include the library and initialize it in `setup()`.

```cpp
#include <USBMouse.h>

void setup() {
    // Initialize USB Mouse stack
    Mouse.begin();
    
    // Recommended: Give the host OS time to enumerate
    delay(2000);
}

void loop() {
    // Your logic here
}
```

### 2. Moving the Cursor

Movement is **relative** to the current position. Coordinates range from -127 to 127 per report.

```cpp
void loop() {
    // Move cursor in a slow circle or pattern
    for (int i = 0; i < 100; i++) {
        Mouse.move(2, 0, 0); // Move 2 units Right
        delay(10);
    }
    
    for (int i = 0; i < 100; i++) {
        Mouse.move(0, 2, 0); // Move 2 units Down
        delay(10);
    }
    
    delay(2000);
}
```

### 3. Clicking and Dragging

Use `press()` and `release()` for drag-and-drop operations.

```cpp
void performDrag() {
    // 1. Position cursor over item
    Mouse.move(10, 10, 0);
    
    // 2. Hold Left Button
    Mouse.press(MOUSE_LEFT);
    delay(100);
    
    // 3. Move to target
    Mouse.move(100, 0, 0);
    delay(100);
    
    // 4. Release
    Mouse.release(MOUSE_LEFT);
}

void loop() {
    if (digitalRead(PA0) == LOW) {
        performDrag();
    }
}
```

## Credits & Acknowledgements

This library was developed by **NoNamedCat**.

It is architecturally inspired by and built upon the **[CH32X035_USBSerial](https://github.com/jobitjoseph/CH32X035_USBSerial)** library.

Special thanks to **[jobitjoseph](https://github.com/jobitjoseph)** for his excellent work on the CH32X035 USB CDC implementation, which provided the essential foundation for the low-level USBFS initialization and endpoint management used in this project.

## License

This project is licensed under the MIT License - see the LICENSE file for details.
