

#pragma once

#include <Arduino.h>

template<uint8_t Size>
class Neza74HC595 {
public:
    Neza74HC595(const uint8_t serialDataPin, const uint8_t clockPin, const uint8_t latchPin) {
        _clockPin = clockPin;
        _serialDataPin = serialDataPin;
        _latchPin = latchPin;

        pinMode(_clockPin, OUTPUT);
        pinMode(_serialDataPin, OUTPUT);
        pinMode(_latchPin, OUTPUT);

        digitalWrite(_clockPin, LOW);
        digitalWrite(_serialDataPin, LOW);
        digitalWrite(_latchPin, LOW);

        memset(_digitalValues, 0, Size);

        updateRegisters();
    }

    void setAll(const uint8_t* digitalValues) {
        if (digitalValues == nullptr) return;
        memcpy(_digitalValues, digitalValues, Size);
        updateRegisters();
    }

    uint8_t* getAll() {
        return _digitalValues;
    }

    void set(const uint8_t pin, const uint8_t value) {
        if (pin >= Size * 8) return;
        setNoUpdate(pin, value);
        updateRegisters();
    }

    void setNoUpdate(const uint8_t pin, uint8_t value) {
        if (pin >= Size * 8) return;
        value ? bitSet(_digitalValues[pin / 8], pin % 8) : bitClear(_digitalValues[pin / 8], pin % 8);
    }

    void updateRegisters() {
		 // Critical section for interrupt safety (optional, enable if needed)
        portDISABLE_INTERRUPTS(); //Disable
        for (int i = Size - 1; i >= 0; i--) {
            shiftOut(_serialDataPin, _clockPin, MSBFIRST, _digitalValues[i]);
        }
        digitalWrite(_latchPin, HIGH);
        digitalWrite(_latchPin, LOW);
        portENABLE_INTERRUPTS();  //Enable
    }

    void setAllHigh() {
        memset(_digitalValues, 0xFF, Size);
        updateRegisters();
    }

    void setAllLow() {
        memset(_digitalValues, 0, Size);
        updateRegisters();
    }

    uint8_t get(const uint8_t pin) {
        if (pin >= Size * 8) return 0;
        return (_digitalValues[pin / 8] >> (pin % 8)) & 1;
    }

private:
    uint8_t _clockPin;
    uint8_t _serialDataPin;
    uint8_t _latchPin;
    uint8_t _digitalValues[Size];
};

/**
 * Notes for ESP32 Usage:
 * - For high-performance applications, consider implementing hardware SPI (not included in this version).
 * - If used in interrupt-heavy applications, uncomment the portDISABLE_INTERRUPTS/portENABLE_INTERRUPTS lines in updateRegisters().
 */