#include "NeuraiCurve.h"
#include "Conversion.h"
#include "utility/trezor/rfc6979.h"
#include "utility/trezor/ecdsa.h"
#include "utility/trezor/secp256k1.h"

const ECPoint InfinityPoint;
const ECPoint GeneratorPoint("0279be667ef9dcbbac55a06295ce870b07029bfcdb2dce28d959f2815b16f81798");

size_t ECPoint::from_stream(ParseStream *s){
	static uint8_t first_byte;
	if(status == PARSING_FAILED){
		return 0;
	}
	if(status == PARSING_DONE){
		bytes_parsed = 0;
	}
	status = PARSING_INCOMPLETE;
	size_t bytes_to_read = 33;
	size_t bytes_read = 0;
	if(bytes_parsed > 0){ // we already know if it's compressed or not
		bytes_to_read = ECPoint::length()-bytes_parsed;
	}else{
		if(s->available()){
			uint8_t c = s->read();
			bytes_read++; bytes_to_read--;
			if(c < 0x02 || c > 0x04){
				status = PARSING_FAILED;
				bytes_parsed += bytes_read;
				return bytes_read;
			}
			first_byte = c;
			if(c == 0x04){ // uncompressed
				bytes_to_read += 32;
				compressed = false;
			}else{
				compressed = true;
			}
		}
	}
	while(s->available() && bytes_to_read > 0){ // actual data
		point[bytes_parsed+bytes_read-1] = s->read();
		bytes_read++; bytes_to_read--;
	}
	if(bytes_to_read==0){
		if(compressed){
			uint8_t buf[33];
			buf[0] = first_byte;
			memcpy(buf+1, point, 32);
            uint8_t arr[65];
            ecdsa_uncompress_pubkey(&secp256k1, buf, arr);
            memcpy(point, arr+1, 64);
		}
		status = PARSING_DONE;
		if(!ECPoint::isValid()){
			status = PARSING_FAILED;
		}
	}
	bytes_parsed += bytes_read;
	return bytes_read;
}
size_t ECPoint::to_stream(SerializeStream *s, size_t offset) const{
	size_t bytes_written = 0;
	if(!s->available()){
		return 0;
	}
	if(offset == 0){    		
		if(compressed){
			s->write(0x02 + (point[63] & 0x01));
		}else{
			s->write(0x04);
		}
		bytes_written ++;
		offset++;
	}
	while(s->available() > 0 && offset < ECPoint::length()){
		s->write(point[offset-1]);
		offset++; bytes_written++;
	}
    return bytes_written;
}
size_t ECPoint::sec(uint8_t * arr, size_t len) const{
	SerializeByteStream s(arr, len);
	return ECPoint::to_stream(&s);
}
size_t ECPoint::fromSec(const uint8_t * arr, size_t len){
	ParseByteStream s(arr, len);
	return ECPoint::from_stream(&s);
}

ECPoint::ECPoint(const uint8_t pubkeyArr[64], bool use_compressed){ 
	memcpy(point, pubkeyArr, 64);
	compressed = use_compressed;
};
ECPoint::ECPoint(const uint8_t * secArr){ 
	if(secArr[0] == 0x04){
		ECPoint::fromSec(secArr, 65);
	}else{
		ECPoint::fromSec(secArr, 33);
	}
};
ECPoint::ECPoint(const char * arr){
	reset();
	ECPoint::parse(arr, strlen(arr));
};

// bool verify(const Signature sig, const uint8_t hash[32]) const;
bool ECPoint::isValid() const{ 
	if(status != PARSING_DONE){
		return false;
	}
    curve_point pub;
	uint8_t buf[65];
	sec(buf, 65);
	return ecdsa_read_pubkey(&secp256k1, buf, &pub);
};

ECPoint ECPoint::operator+(const ECPoint& other) const{
	if(*this == InfinityPoint){
		return other;
	}
	if(other == InfinityPoint){
		return *this;
	}
    curve_point p1, p2;
	uint8_t buf[65];
	sec(buf, 65);
	ecdsa_read_pubkey(&secp256k1, buf, &p1);
	other.sec(buf, 65);
	ecdsa_read_pubkey(&secp256k1, buf, &p2);
    point_add(&secp256k1,&p1,&p2);
    ECPoint sum;
	bn_write_be(&p2.x, sum.point);
	bn_write_be(&p2.y, sum.point+32);
	return sum;
};
ECPoint ECPoint::operator-() const{
	if(*this == InfinityPoint){
		return *this;
	}
	ECPoint a = *this;
	bignum256 y;
	bn_read_be(point+32, &y);
	bn_inverse(&y, &secp256k1.prime);
	bn_write_be(&y, a.point+32);
	return a;
}
ECPoint ECPoint::operator-(const ECPoint& other) const{
	ECPoint a = -other;
	return *this+a;
}

/*********** ECScalar ******************/

size_t ECScalar::from_stream(ParseStream *s){
	if(status == PARSING_FAILED){
		return 0;
	}
	if(status == PARSING_DONE){
		bytes_parsed = 0;
	}
	status = PARSING_INCOMPLETE;
    size_t bytes_read = 0;
    while(s->available() > 0 && bytes_parsed+bytes_read < 32){
        num[bytes_parsed+bytes_read] = s->read();
        bytes_read++;
    }
    if(bytes_parsed+bytes_read == 32){
    	status = PARSING_DONE;
	    uint8_t zero[32] = { 0 };
	    if(memcmp(num, zero, 32)==0){ // should we add something else here?
	    	status = PARSING_FAILED;
	    }
	    bignum256 n;
	    bn_read_be(num, &n);
	    bn_mod(&n, &secp256k1.order);
	    bn_write_be(&n, num);
    }
    bytes_parsed += bytes_read;
    return bytes_read;
}
size_t ECScalar::to_stream(SerializeStream *s, size_t offset) const{
	size_t bytes_written = 0;
	while(s->available() && offset+bytes_written < 32){
		s->write(num[bytes_written+offset]);
		bytes_written++;
	}
	return bytes_written;
}
ECScalar ECScalar::operator+(const ECScalar& other) const{
    bignum256 a, b;
	bn_read_be(this->num, &a);
	bn_read_be(other.num, &b);
	bn_addmod(&a, &b, &secp256k1.order);
    bn_mod(&a, &secp256k1.order);
	ECScalar sum;
	bn_write_be(&a, sum.num);
	return sum;
}
ECScalar ECScalar::operator+(const uint32_t& i) const{
    bignum256 a;
	bn_read_be(this->num, &a);
	bn_addi(&a, i);
    bn_mod(&a, &secp256k1.order);
	ECScalar sum;
	bn_write_be(&a, sum.num);
	return sum;
}
ECScalar ECScalar::operator-() const{
    bignum256 a, b;
	bn_read_be(this->num, &a);
	bn_subtract(&secp256k1.order, &a, &b);
	ECScalar neg;
	bn_write_be(&b, neg.num);
	return neg;
}
ECScalar ECScalar::operator-(const uint32_t& i) const{
    bignum256 a;
	bn_read_be(this->num, &a);
	bn_subi(&a, i, &secp256k1.order);
    bn_mod(&a, &secp256k1.order);
	ECScalar sum;
	bn_write_be(&a, sum.num);
	return sum;
}
ECScalar ECScalar::operator-(const ECScalar& other) const{
	return (*this+(-other));
}
ECScalar ECScalar::operator*(const ECScalar& other) const{
    bignum256 a, b;
	bn_read_be(this->num, &a);
	bn_read_be(other.num, &b);
	bn_multiply(&b, &a, &secp256k1.order);
    bn_mod(&a, &secp256k1.order);
	ECScalar mul;
	bn_write_be(&a, mul.num);
	return mul;
}
ECScalar ECScalar::operator/(const ECScalar& other) const{
    bignum256 a, b;
	bn_read_be(this->num, &a);
	bn_read_be(other.num, &b);
	bn_inverse(&b, &secp256k1.order);
	bn_multiply(&b, &a, &secp256k1.order);
    bn_mod(&a, &secp256k1.order);
	ECScalar res;
	bn_write_be(&a, res.num);
	return res;
}
bool ECScalar::operator<(const ECScalar& other) const{
	bignum256 a,b;
	bn_read_be(num, &a);
	bn_read_be(other.num, &b);
	return bn_is_less(&a, &b);
}
bool ECPoint::operator<(const ECPoint& other) const{
	uint8_t sec1[65];
	uint8_t sec2[65];
	sec(sec1, sizeof(sec1));
	other.sec(sec2, sizeof(sec2));
	return memcmp(sec1, sec2, sizeof(sec1)) > 0;
}
ECPoint operator*(const ECScalar& scalar, const ECPoint& point){
	ECPoint r;
	uint8_t num[32];
	scalar.getSecret(num);
	if(point == GeneratorPoint){
		uint8_t pubkey[65];
		ecdsa_get_public_key65(&secp256k1, num, pubkey);
		r.parse(pubkey, 65);
	}else{
		bignum256 d;
		bn_read_be(num, &d);
		curve_point p, res;
		bn_read_be(point.point, &p.x);
		bn_read_be(point.point+32, &p.y);
		point_multiply(&secp256k1, &d, &p, &res);
		bn_write_be(&res.x, r.point);
		bn_write_be(&res.y, r.point+32);
	}
	r.compressed = point.compressed;
	return r;
}