#pragma once

#include <Arduino.h>
#include "InfiniteFunctionButtonConfig.h"

#if defined(ARDUINO_ARCH_ESP8266) || defined(ARDUINO_ARCH_ESP32)
  #define IFB_IRAM_ATTR IRAM_ATTR
#else
  #define IFB_IRAM_ATTR
#endif

#if defined(ARDUINO_ARCH_AVR)
  template<typename T, uint8_t N>
  class StaticVector {
  public:
    bool push_back(const T& v) {
      if (size_ >= N) return false;
      data_[size_++] = v;
      return true;
    }

    template<typename Compare>
    bool push_sorted(const T& v, Compare comp) {
      if (size_ >= N) return false;

      uint8_t i = size_;
      while (i > 0 && comp(v, data_[i - 1])) {
        data_[i] = data_[i - 1];
        --i;
      }

      data_[i] = v;
      ++size_;
      return true;
    }

    T& operator[](uint8_t i) { return data_[i]; }
    const T& operator[](uint8_t i) const { return data_[i]; }

    uint8_t size() const { return size_; }
    bool empty() const { return size_ == 0; }

    T& front() { return data_[0]; }
    T& back() { return data_[size_ - 1]; }

  private:
    T data_[N];
    uint8_t size_ = 0;
  };
#else
  #include <vector>
#endif

typedef void (*Callback)();

class InfiniteFunctionButton {
  public:
    InfiniteFunctionButton(uint8_t gpio, uint8_t debounceTime, bool activeHigh = false) : gpio(gpio), debounceTime(debounceTime), activeHigh(activeHigh) {
      /*!
        @brief      The class constructor.
        @param[in]  gpio The gpio to use.
        @param[in]  debounceTime The debounce time, in milliseconds.
        @param[in]  activeHigh Specify if the button is wired to output a HIGH state when pressed.
        @details    Pick a pin that supports interrupts on your microcontroller (2 or 3 for Arduino Uno).
                    When activeHigh is false or not specified, the pin will be set up as INPUT_PULLUP
                    and the button should be connected to short it to ground when pressed.
                    Likewise, when activeHigh is true, the pin will be set up as INPUT
                    and a HIGH state will represent the button being pressed.
                    Remember to provide your own pulldown resistor in this setting.
      */
    }
    void begin(void (isr)());
    void addIntermediateFunction(unsigned long timeout, Callback callback, Callback timeCallback);
    void setEndingFunction(Callback callback);
    void handleInterruptAndFunctions();
    void IFB_IRAM_ATTR onInterrupt();
  private:
    const uint8_t gpio;
    const uint8_t debounceTime;
    const bool activeHigh;
    volatile bool pressed = false;
    volatile bool interrupt = false;
    volatile unsigned long lastPressed = 0;
    volatile unsigned long lastInterrupt = 0;
    class IntermediateFunction {
      public:
        IntermediateFunction() : timeout(0), callback(nullptr), timeCallback(nullptr) {}
        IntermediateFunction(unsigned long timeout, Callback callback, Callback timeCallback) : timeout(timeout), callback(callback), timeCallback(timeCallback) {}
        unsigned long timeout;
        Callback callback;
        Callback timeCallback;
        bool timeCallbackCalled = false;
    };
    #if defined(ARDUINO_ARCH_AVR)
      StaticVector<IntermediateFunction, IFB_MAX_FUNCTIONS> intermediateFunctions;
    #else
      std::vector<IntermediateFunction> intermediateFunctions;
    #endif
    Callback endingFunction = NULL;
};