#include "InfiniteFunctionButton.h"

void InfiniteFunctionButton::begin(void (isr)()) {
  /*!
    @brief      The begin function.
    @details    Sets up the gpio and attaches an interrupt.
    @param[in]  isr The Interrupt Service Routine to use for this button.
  */
  pinMode(gpio, activeHigh ? INPUT : INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(gpio), isr, CHANGE);
  interrupt = false;
}

void InfiniteFunctionButton::addIntermediateFunction(unsigned long timeout, Callback callback, Callback timeCallback) {
  /*!
    @brief      Add a new intermediate function for the button.
    @details    Intermediate functions are functions to be activated when the button is released before
                the specified timeout, and after the timeout of other intermediate function(s), if any.
    @param[in]  timeout The time since the button press, until which this function is selected for activation, in milliseconds.
    @param[in]  callback The callback to be called if this function is activated (the button is released before timeout).
    @param[in]  timeCallback (can be NULL) The callback to be called when the time frame of this function begins (immediately
                after the button press if this is the only intermediate function). Can be used for informing the user that
                this function is currently selected for activation and thus prompting them to release the button, for
                example by lighting up an LED or changing some pixels on a display.
  */
  #if defined(ARDUINO_ARCH_AVR)
    intermediateFunctions.push_sorted(
      IntermediateFunction(timeout, callback, timeCallback),
      [](const IntermediateFunction& a, const IntermediateFunction& b) {
        return a.timeout < b.timeout;
      }
    );
  #else
    #include <algorithm>
    intermediateFunctions.emplace_back(timeout, callback, timeCallback);
    std::sort(intermediateFunctions.begin(), intermediateFunctions.end(), [](const IntermediateFunction& a, const IntermediateFunction& b) {
      return a.timeout < b.timeout;
    });
  #endif
}

void InfiniteFunctionButton::setEndingFunction(Callback callback) {
  /*!
    @brief      Set the ending function for the button.
    @details    Ending function will be activated when the button is not released past the timeouts of all intermediate functions.
                If no intermediate functions are defined, it will be called as soon as the button is pressed.
    @param[in]  callback The callback to be called if this function is activated (the button is not released).
  */
  this->endingFunction = callback;
}

void InfiniteFunctionButton::handleInterruptAndFunctions() {
  /*!
    @brief      The loop function.
    @details    Handles the interrupt and runs the defined functions.
  */
  if (interrupt) {
    unsigned long diff = millis() - lastInterrupt;
    if (diff > debounceTime) {
      interrupt = false;
      if (activeHigh != digitalRead(gpio)) { // button released
        pressed = false;
        unsigned long pressedFor = lastInterrupt - lastPressed;
        #ifdef IFB_DEBOUNCE_TUNE
          Serial.println("0");
        #endif
        for (size_t i = 0; i < intermediateFunctions.size(); i++) {
          IntermediateFunction& function = intermediateFunctions[i];
          if (pressedFor < function.timeout) {
            function.callback();
            break;
          }
        }
      } else { // button pressed
        pressed = true;
        lastPressed = lastInterrupt;
        #ifdef IFB_DEBOUNCE_TUNE
          Serial.println("1");
        #endif

        if (intermediateFunctions.empty()) {
          if (endingFunction != NULL) {
            endingFunction();
          }
          return;
        }

        for (size_t i = 0; i < intermediateFunctions.size(); i++) {
          intermediateFunctions[i].timeCallbackCalled = false;
        }

        IntermediateFunction& function = intermediateFunctions.front();
        if (function.timeCallback == NULL) {
          return;
        }
        function.timeCallback();
        function.timeCallbackCalled = true;
      }
    }
    return;
  }

  if (pressed) {
    unsigned long pressedFor = millis() - lastPressed;
    if (pressedFor >= intermediateFunctions.back().timeout) { // for the ending action; if it's pressed for longer than the longest timeout
      if (endingFunction != NULL) {
        endingFunction();
      }
      pressed = false;
      return;
    }

    for (size_t i = 1; i < intermediateFunctions.size(); i++) {
      IntermediateFunction& current = intermediateFunctions[i];
      if (current.timeCallback == NULL || current.timeCallbackCalled) {
        continue;
      }
      IntermediateFunction& previous = intermediateFunctions[i - 1];
      if (pressedFor >= previous.timeout) {
        current.timeCallback();
        current.timeCallbackCalled = true;
      }
    }
  }
}

void IFB_IRAM_ATTR InfiniteFunctionButton::onInterrupt() {
  /*!
    @brief      The internal Interrupt Service Routine.
    @details    Call this function from the static ISR passed to begin.
  */
  interrupt = true;
  lastInterrupt = millis();
}