/*
© 2025 Microchip Technology Inc. and its subsidiaries. All rights reserved.

Subject to your compliance with these terms, you may use this Microchip software and any derivatives 
exclusively with Microchip products. You are responsible for complying with third party license terms 
applicable to your use of third party software (including open source software) that may accompany this 
Microchip software. SOFTWARE IS “AS IS.” NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR 
STATUTORY, APPLY TO THIS SOFTWARE, INCLUDING ANY IMPLIED WARRANTIES OF NON-INFRINGEMENT,
MERCHANTABILITY, OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT WILL 
MICROCHIP BE LIABLE FOR ANY INDIRECT, SPECIAL, PUNITIVE, INCIDENTAL OR CONSEQUENTIAL LOSS, 
DAMAGE, COST OR EXPENSE OF ANY KIND WHATSOEVER RELATED TO THE SOFTWARE, HOWEVER 
CAUSED, EVEN IF MICROCHIP HAS BEEN ADVISED OF THE POSSIBILITY OR THE DAMAGES ARE 
FORESEEABLE. TO THE FULLEST EXTENT ALLOWED BY LAW, MICROCHIP’S TOTAL LIABILITY ON ALL 
CLAIMS RELATED TO THE SOFTWARE WILL NOT EXCEED AMOUNT OF FEES, IF ANY, YOU PAID DIRECTLY 
TO MICROCHIP FOR THIS SOFTWARE
*/

#include <Arduino.h>
#include "rnwf_wifi.h"
#include "rnwf_interface.h"
#include "rnwf_net.h"
#include "rnwf_mqtt.h"

#include <string.h>
#include <stdio.h>
#include <stddef.h>
#include <stdbool.h>
#include <stdlib.h>

#define RNWFserial Serial1
#define USBserial Serial
#define SERIAL_BAUDRATE 230400
#define RNWF_BAUDRATE 230400
#define RESET_PIN A3

#define SYS_RNWF_WIFI_STA_SSID "DEMO_AP_RNWF"
#define SYS_RNWF_WIFI_STA_PWD "password"
#define SYS_RNWF_COUNTRYCODE "GEN"
#define SYS_RNWF_SNTP_ADDRESS "163.227.192.148"
#define SYS_RNWF_MQTT_VERSION 3
#define SYS_RNWF_MQTT_CLOUD_URL "test.mosquitto.org"
#define SYS_RNWF_MQTT_CLOUD_PORT 1883
#define SYS_RNWF_MQTT_CLIENT_ID "rnwf02"
#define SYS_RNWF_MQTT_CLOUD_USER_NAME ""
#define SYS_RNWF_MQTT_PASSWORD ""
#define SYS_RNWF_MQTT_KEEPALIVE 60
#define SYS_RNWF_MQTT_SUB_TOPIC_0 "MCHP/Sample321"
#define SYS_RNWF_MQTT_SUB_TOPIC_0_QOS 0

#define SYS_RNWF_MQTT_PUB_TOPIC_0_DUP 0
#define SYS_RNWF_MQTT_PUB_TOPIC_0_QOS 0
#define SYS_RNWF_MQTT_PUB_TOPIC_0_RETAIN 0
#define SYS_RNWF_MQTT_PUB_TOPIC_1_RETAIN 1
#define SYS_RNWF_MQTT_PUB_TOPIC_0 "MCHP/Sample123"

bool Err;
bool clientStatus = false;
uint16_t receivedDataSize = 0;
uint32_t socketId = 0;
String clientId = "";
String wifiIPAddress = "";
String TEST_MESSAGE = "HELLO WORLD\r\n";

typedef enum {
  /* TODO: Define states used by the application state machine. */
  RNWF_ECHO_OFF,
  RNWF_MAN_ID,
  RNWF_REV_ID,
  RNWF_WIFI_INFO,
  RNWF_WIFI_SET_DOMAIN,
  RNWF_WIFI_GET_SOFT_AP_STATE,
  RNWF_WIFI_SOFT_AP_DISABLE,
  RNWF_WIFI_GET_STA_STATE,
  RNWF_WIFI_DISCONNECT,
  RNWF_WIFI_SET_STA_SSID,
  RNWF_WIFI_SET_STA_PWD,
  RNWF_WIFI_SET_STA_SEC,
  RNWF_WIFI_CONNECT,
  RNWF_WIFI_TCP_SOCKET_OPEN,
  RNWF_WIFI_SOCK_BIND_LOCAL,
  RNWF_WIFI_TCP_SOCKET_EST,
  RNWF_WIFI_SET_SNTP_STATIC,
  RNWF_WIFI_SET_SNTP_URL,
  RNWF_WIFI_SNTP_ENABLE,
  RNWF_WIFI_MQTT_PROTOCOL,
  RNWF_WIFI_MQTT_BROKER_URL,
  RNWF_WIFI_MQTT_BROKER_PORT,
  RNWF_WIFI_MQTT_BROKER_CLIENT_ID,
  RNWF_WIFI_MQTT_BROKER_USERNAME,
  RNWF_WIFI_MQTT_BROKER_PASSWORD,
  RNWF_WIFI_MQTT_BROKER_KEEPALIVE,
  RNWF_WIFI_MQTT_CONNECT,
  RNWF_WIFI_MQTT_BROKER_PUBLISH,
  RNWF_WIFI_MQTT_BROKER_SUBSCRIBE
} STATES;

typedef struct
{
  /* The application's current state */
  STATES state;

} RNWF_STATE;

RNWF_STATE rnwf_state;
WIFI wifi;
NET net;
MQTT mqtt;

void setup() {

  // initialize both serial ports:
  USBserial.begin(SERIAL_BAUDRATE);
  RNWFserial.begin(RNWF_BAUDRATE);
  RNWF_SerialInit(&RNWFserial);
  pinMode(RESET_PIN, OUTPUT);
  RNWF_ResetTargetHW();
  USBserial.println("RNWF Resetting...");

  // Wait for reset to complete
  while (true) {
    if (RNWFserial.available()) {
      String response = RNWFserial.readStringUntil('\n');
      response.trim();
      if (response.startsWith("+BOOT:")) {
        USBserial.println("!!! RNWF Initialized !!!");
        break;
      }
    }
  }
}

void loop() {
  RNWF_TCP_SERVER();
}

void RNWF_TCP_SERVER() {
  switch (rnwf_state.state) {
    case RNWF_ECHO_OFF:
      {
        Err = wifi.RNWF_SetEchoOff();
        if (Err) {
          Err = false;
          USBserial.println("RNWF Echo Off");
          rnwf_state.state = RNWF_MAN_ID;
        }
      }
      break;
    case RNWF_MAN_ID:
      {
        Err = wifi.RNWF_GetManId();
        if (Err) {
          Err = false;
          USBserial.println("RNWF MAN ID RECEIVED");
          rnwf_state.state = RNWF_REV_ID;
        }
      }
      break;
    case RNWF_REV_ID:
      {
        Err = wifi.RNWF_GetRevId();
        if (Err) {
          Err = false;
          USBserial.println("RNWF REV ID RECEIVED");
          rnwf_state.state = RNWF_WIFI_INFO;
        }
      }
      break;
    case RNWF_WIFI_INFO:
      {
        Err = wifi.RNWF_WifiInfo();
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI INFO RECEIVED");
          rnwf_state.state = RNWF_WIFI_SET_DOMAIN;
        }
      }
      break;
    case RNWF_WIFI_SET_DOMAIN:
      {
        char command[30];
        sprintf(command, SYS_RNWF_WIFI_SET_REG_DOMAIN, SYS_RNWF_COUNTRYCODE);
        Err = wifi.RNWF_WifiSetRegDomain(command);
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI DOMAIN SET");
          rnwf_state.state = RNWF_WIFI_GET_SOFT_AP_STATE;
        }
      }
      break;
    case RNWF_WIFI_GET_SOFT_AP_STATE:
      {
        Err = wifi.RNWF_WifiGetSoftApState();
        if (Err) {
          Err = false;
          String response = "";
          response = RNWFserial.readString();

          USBserial.println("`````````````````SOFT AP STATE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          // Check if response starts with "+WAP:"
          if (response.startsWith("+WAP:")) {
            USBserial.println("RNWF WIFI SOFT AP STATE RECEIVED");
            int separatorIndex = response.indexOf(':');
            if (separatorIndex != -1 && separatorIndex < response.length() - 1) {
              char state = response.charAt(separatorIndex + 1);
              if (state == '0') {
                USBserial.println("Soft AP is DISABLED");
                rnwf_state.state = RNWF_WIFI_GET_STA_STATE;
              } else if (state == '1') {
                USBserial.println("Soft AP is ENABLED");
                rnwf_state.state = RNWF_WIFI_SOFT_AP_DISABLE;
              }
            }
          }
        }
      }
      break;
    case RNWF_WIFI_SOFT_AP_DISABLE:
      {
        Err = wifi.RNWF_WifiSoftApDisable();
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI SOFT AP DISABLED");
          rnwf_state.state = RNWF_WIFI_DISCONNECT;
        }
      }
      break;
    case RNWF_WIFI_GET_STA_STATE:
      {
        Err = wifi.RNWF_WifiGetStaState();
        if (Err) {
          Err = false;
          String response = "";
          response = RNWFserial.readString();
          USBserial.println("```````````````````STA STATE```````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          // Check if response starts with "+WSTA:"
          if (response.startsWith("+WSTA:")) {
            USBserial.println("RNWF WIFI STA STATE RECEIVED");
            int separatorIndex = response.indexOf(':');
            if (separatorIndex != -1 && separatorIndex < response.length() - 1) {
              char state = response.charAt(separatorIndex + 1);
              if (state == '0') {
                USBserial.println("STA is DISABLED");
                rnwf_state.state = RNWF_WIFI_SET_STA_SSID;
              } else if (state == '1') {
                USBserial.println("STA is ENABLED");
                rnwf_state.state = RNWF_WIFI_DISCONNECT;
              }
            }
          }
        }
      }
      break;
    case RNWF_WIFI_DISCONNECT:
      {
        Err = wifi.RNWF_WifiDisconnect();
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI DISCONNECTED");
          rnwf_state.state = RNWF_WIFI_SET_STA_SSID;
        }
      }
      break;
    case RNWF_WIFI_SET_STA_SSID:
      {
        char command[30];
        sprintf(command, SYS_RNWF_WIFI_SET_STA_SSID, SYS_RNWF_WIFI_STA_SSID);
        Err = wifi.RNWF_WifiSetStaSsid(command);
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI STA SSID SET");
          rnwf_state.state = RNWF_WIFI_SET_STA_PWD;
        }
      }
      break;
    case RNWF_WIFI_SET_STA_PWD:
      {
        char command[30];
        sprintf(command, SYS_RNWF_WIFI_SET_STA_PWD, SYS_RNWF_WIFI_STA_PWD);
        Err = wifi.RNWF_WifiSetStaPwd(command);
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI STA PASSWORD SET");
          rnwf_state.state = RNWF_WIFI_SET_STA_SEC;
        }
      }
      break;
    case RNWF_WIFI_SET_STA_SEC:
      {
        char command[30];
        sprintf(command, SYS_RNWF_WIFI_SET_STA_SEC, SYS_RNWF_WIFI_SECURITY_WPA2);
        Err = wifi.RNWF_WifiSetStaSec(command);
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI STA SECURITY SET");
          rnwf_state.state = RNWF_WIFI_CONNECT;
        }
      }
      break;
    case RNWF_WIFI_CONNECT:
      {
        Err = wifi.RNWF_WifiConnect();
        unsigned long startTime = millis();
        String receivedString;
        if (Err) {
          Err = false;

          // Read data for 15 second
          while (millis() - startTime < 15000) {
            receivedString += UART_RNWF_Read();
          }

          USBserial.println("````````````WIFI CONNECTION RESPONSE```````````");
          USBserial.print(receivedString);
          USBserial.println("```````````````````````````````````````````````");

          if (receivedString.indexOf("+WSTALU:") == -1) {
            USBserial.println("!!! STA NOT FOUND !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI CONNECTED");
            rnwf_state.state = RNWF_WIFI_SET_SNTP_STATIC;

            int ipIndex = receivedString.indexOf("+WSTAAIP:");
            if (ipIndex != -1) {
              int start = ipIndex + 12;
              int end = receivedString.indexOf("\"", start);
              if (end != -1) {
                wifiIPAddress = receivedString.substring(start, end);
                USBserial.print("Extracted IP Address: ");
                USBserial.println(wifiIPAddress);
              }
            }
          }
        }
      }
      break;      
    case RNWF_WIFI_SET_SNTP_STATIC:
      {
        Err = wifi.RNWF_WifiSetSntpStatic();
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI STATIC SNTP SET");
          rnwf_state.state = RNWF_WIFI_SET_SNTP_URL;
        }
      }
      break;
    case RNWF_WIFI_SET_SNTP_URL:
      {
        char command[30];
        sprintf(command, RNWF_SET_SNTP_URL,SYS_RNWF_SNTP_ADDRESS);
        Err = wifi.RNWF_WifiSetSntpUrl(command);
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI SNTP URL SET");
          rnwf_state.state = RNWF_WIFI_SNTP_ENABLE;
        }
      }
      break;
    case RNWF_WIFI_SNTP_ENABLE:
      {
        Err = wifi.RNWF_WifiSntpEnable();
        if (Err) {
          Err = false;
          USBserial.println("RNWF WIFI SNTP ENABLED");
          rnwf_state.state = RNWF_WIFI_MQTT_PROTOCOL;
        }
      }
      break;    
    case RNWF_WIFI_MQTT_PROTOCOL:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_PROTO_VER, SYS_RNWF_MQTT_VERSION);
        Err = mqtt.RNWF_WifiMqttProtocol(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT PROTOCOL VERSION SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_URL;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_URL:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_BROKER_URL, SYS_RNWF_MQTT_CLOUD_URL);
        Err = mqtt.RNWF_WifiMqttBrokerUrl(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT BROKER URL SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_PORT;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_PORT:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_BROKER_PORT, SYS_RNWF_MQTT_CLOUD_PORT);
        Err = mqtt.RNWF_WifiMqttBrokerPort(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT BROKER PORT SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_CLIENT_ID;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_CLIENT_ID:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_CLIENT_ID, SYS_RNWF_MQTT_CLIENT_ID);
        Err = mqtt.RNWF_WifiMqttBrokerClientId(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT BROKER CLIENT ID SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_USERNAME;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_USERNAME:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_USERNAME, SYS_RNWF_MQTT_CLOUD_USER_NAME);
        Err = mqtt.RNWF_WifiMqttBrokerUsername(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT BROKER USERNAME SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_PASSWORD;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_PASSWORD:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_PASSWORD, SYS_RNWF_MQTT_PASSWORD);
        Err = mqtt.RNWF_WifiMqttBrokerPassword(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT BROKER PASSWORD SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_KEEPALIVE;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_KEEPALIVE:
      {
        char command[30];
        sprintf(command, SYS_RNWF_MQTT_SET_KEEPALIVE, SYS_RNWF_MQTT_KEEPALIVE);
        Err = mqtt.RNWF_WifiMqttBrokerKeepalive(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {            
            USBserial.println("RNWF WIFI MQTT BROKER KEEPALIVE SET");            
            rnwf_state.state = RNWF_WIFI_MQTT_CONNECT;                       
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_CONNECT:
      {
        Err = mqtt.RNWF_WifiMqttConnect();
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT CONNECTED");
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_SUBSCRIBE;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_SUBSCRIBE:
      {
        char command[50];
        sprintf(command, SYS_RNWF_MQTT_CMD_SUBSCRIBE_QOS, SYS_RNWF_MQTT_SUB_TOPIC_0,SYS_RNWF_MQTT_SUB_TOPIC_0_QOS); 
        Err = mqtt.RNWF_WifiMqttBrokerSubscribe(command);
        if (Err) {
          Err = false;
          String response = RNWFserial.readString();

          USBserial.println("`````````````````MQTT RESPONSE`````````````````");
          USBserial.println(response);
          USBserial.println("```````````````````````````````````````````````");

          if (response.indexOf("+WSTALD:1") != -1) {
            USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
            rnwf_state.state = RNWF_WIFI_CONNECT;
          } else {
            USBserial.println("RNWF WIFI MQTT BROKER SUBSCRIBED");            
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_PUBLISH;
          }
        }
      }
      break;
    case RNWF_WIFI_MQTT_BROKER_PUBLISH:
      {
        char payload[15];
        long randomValue = random(10,50);
        // Convert long to string
        sprintf(payload, "%ld", randomValue);
        char mqttPublish[50];
        sprintf(mqttPublish, SYS_RNWF_MQTT_CMD_PUBLISH, SYS_RNWF_MQTT_PUB_TOPIC_0_DUP,SYS_RNWF_MQTT_PUB_TOPIC_0_QOS,SYS_RNWF_MQTT_PUB_TOPIC_0_RETAIN,SYS_RNWF_MQTT_PUB_TOPIC_0,payload);
               
        Err = mqtt.RNWF_WifiMqttBrokerPublish(mqttPublish);

        if (Err) {
        Err = false;
        String response = RNWFserial.readString();

        USBserial.println("`````````````````MQTT RESPONSE`````````````````");
        USBserial.println(response);
        USBserial.println("```````````````````````````````````````````````");

        if (response.indexOf("+WSTALD:1") != -1) {
          USBserial.println("!!! WIFI DISCONNECTED - RECONNECTING !!!");
          rnwf_state.state = RNWF_WIFI_CONNECT;
        } 
        // Check for MQTT subscription receive
        else if (response.indexOf("+MQTTSUBRX") != -1) {
            int lastQuote = response.lastIndexOf('"');
            int secondLastQuote = response.lastIndexOf('"', lastQuote - 1);  
            
            if (lastQuote != -1 && secondLastQuote != -1) {
                String data = response.substring(secondLastQuote + 1, lastQuote);
                USBserial.print("MQTT SUBSCRIBE DATA RECEIVED: ");
                USBserial.println(data);
            } else {
                USBserial.println("MQTT SUBSCRIBE DATA PARSE ERROR");
            }
            rnwf_state.state = RNWF_WIFI_MQTT_BROKER_PUBLISH;
        }else {
          USBserial.println("RNWF WIFI MQTT BROKER PUBLISHED");            
          rnwf_state.state = RNWF_WIFI_MQTT_BROKER_PUBLISH;
        }        
      }
      delay(5000);
      }
      break;    
  }
}

void serial_transfer() {

  // read from RNWF and Print on Arduino Zero
  if (RNWFserial.available()) {
    String BU_data = RNWFserial.readString();
    USBserial.println(BU_data);
  }

  // read from Arduino Zero and Print on RNWF
  if (USBserial.available()) {
    String AR_data = USBserial.readString();
    RNWFserial.print(AR_data);
  }
}

void RNWF_ResetTargetHW(void) {
  digitalWrite(RESET_PIN, LOW);
  delay(50);
  digitalWrite(RESET_PIN, HIGH);
}
