#pragma once

#include <Arduino.h>
#include <Wire.h>

/**
 * BH1730FVC 简易驱动（常见地址 0x29）。
 * - IIC 需外部先初始化：Wire.begin(SDA, SCL)
 * - 通过 begin(TwoWire*, addr, timing) 注入
 */
class BH1730FVC {
public:
  BH1730FVC();

  // timing 默认 0x02（48ms 积分，参考 README.md），gain 默认 0x00（1x增益）
  bool begin(TwoWire *wire, uint8_t address = 0x29, uint8_t timing = 0x02, uint8_t gain = 0x00);

  // 读取通道原始值
  bool readRaw(uint16_t &ch0, uint16_t &ch1);

  // Lux 计算（使用 DATA0/1 比例，参考官方分段公式）
  // gain: 传感器增益倍数（如 8.0 表示 8x，默认使用初始化时的增益）
  // integrationMs: 积分时间；传 <=0 将使用当前 timing 估算值
  float readLux(float gain = -1.0f, float integrationMs = -1.0f);

  // 设置增益值（运行时动态设置）
  // gain: 增益寄存器值 (0x00=1x, 0x01=2x, 0x02=4x, 0x03=8x)
  // 返回 true 表示设置成功，false 表示失败
  bool setGain(uint8_t gain);

  uint16_t lastCh0() const { return _lastCh0; }
  uint16_t lastCh1() const { return _lastCh1; }

private:
  // 寄存器地址（不带命令位，命令位在读写时添加）
  static constexpr uint8_t REG_COMMAND = 0x00;
  static constexpr uint8_t REG_TIMING_BASE = 0x01;
  static constexpr uint8_t REG_CONTROL_BASE = 0x02;
  static constexpr uint8_t REG_GAIN_BASE = 0x07;
  static constexpr uint8_t REG_DATA0_L_BASE = 0x14; // ch0 low
  static constexpr uint8_t REG_DATA0_H_BASE = 0x15; // ch0 high
  static constexpr uint8_t REG_DATA1_L_BASE = 0x16; // ch1 low
  static constexpr uint8_t REG_DATA1_H_BASE = 0x17; // ch1 high
  
  // 命令字（带命令位 0x80）
  static constexpr uint8_t CMD_WRITE_REG(uint8_t reg) { return 0x80 | reg; }
  static constexpr uint8_t CMD_READ_REG(uint8_t reg) { return 0x80 | reg; }

  TwoWire *_wire;
  uint8_t _addr;
  uint16_t _lastCh0;
  uint16_t _lastCh1;
  uint8_t _timing;
  uint8_t _gain;

  bool powerOn();
  bool writeReg(uint8_t regBase, uint8_t val);
  uint8_t readByte(uint8_t regBase);
  uint16_t read16BitData(uint8_t startRegBase);
  bool readBlock(uint8_t startRegBase, uint8_t *buf);
  uint16_t integrationMsFromTiming(uint8_t timing);
};

