#include "BH1730FVC.h"

BH1730FVC::BH1730FVC() : _wire(nullptr), _addr(0x29), _lastCh0(0), _lastCh1(0), _timing(0x00) {}

bool BH1730FVC::begin(TwoWire *wire, uint8_t address, uint8_t timing, uint8_t gain) {
  _wire = wire;
  _addr = address;
  _timing = timing;
  _gain = gain;
  if (_wire == nullptr) return false;
  
  delay(100); // 给传感器上电稳定时间
  
  // ① 配置 TIMING 寄存器（积分时间）
  if (!writeReg(REG_TIMING_BASE, _timing)) {
      Serial.println("writeReg TIMING failed");
      return false;
  }
  delay(2); // 寄存器写入后稳定时间
  
  // ② 配置 GAIN 寄存器（增益，提高灵敏度）
  if (!writeReg(REG_GAIN_BASE, _gain)) {
      Serial.println("writeReg GAIN failed");
      return false;
  }
  delay(2);
  
  // ③ 配置 CONTROL 寄存器（启动连续测量）
  if (!powerOn()) {
      Serial.println("powerOn failed");
      return false;
  }
  delay(2);
  
  // 等待首次测量完成（大于积分时间）
  delay(integrationMsFromTiming(_timing) + 10);
  
  return true;
}

bool BH1730FVC::readRaw(uint16_t &ch0, uint16_t &ch1) {
  if (_wire == nullptr){
      Serial.println("wire is nullptr");
      return false;
  }
  
  // 读取 DATA0 和 DATA1（参考 README.md 的 read16BitData）
  uint8_t buf[4];
  if (!readBlock(REG_DATA0_L_BASE, buf)){
      Serial.println("readBlock failed");
      return false;
  }

  // 数据格式：buf[0]=DATA0低字节, buf[1]=DATA0高字节, buf[2]=DATA1低字节, buf[3]=DATA1高字节
  ch0 = (uint16_t)((buf[1] << 8) | buf[0]);
  ch1 = (uint16_t)((buf[3] << 8) | buf[2]);

  _lastCh0 = ch0;
  _lastCh1 = ch1;
  return true;
}

float BH1730FVC::readLux(float gain, float integrationMs) {
  uint16_t ch0, ch1;
  if (!readRaw(ch0, ch1)){
      Serial.println("readRaw failed");
      return NAN;
  }
  if (ch0 == 0){
      Serial.println("ch0 is 0");
      return NAN;
  }

  // 积分时间自动估算
  float itime = (integrationMs > 0.0f) ? integrationMs : (float)integrationMsFromTiming(_timing);
  
  // 增益转换：寄存器值 0x03 = 8x，0x00 = 1x，0x01 = 2x，0x02 = 4x
  float actualGain = (gain > 0.0f) ? gain : (float)(1 << _gain); // 默认使用初始化时的增益
  
  if (actualGain <= 0.0f || itime <= 0.0f){
      Serial.println("gain or itime is <= 0");
      return NAN;
  }

  float ratio = (float)ch1 / (float)ch0;
  float luxCalc = 0.0f;
  if (ratio < 0.26f) {
    luxCalc = (1.290f * ch0 - 2.733f * ch1);
  } else if (ratio < 0.55f) {
    luxCalc = (0.795f * ch0 - 0.859f * ch1);
  } else if (ratio < 1.09f) {
    luxCalc = (0.510f * ch0 - 0.345f * ch1);
  } else if (ratio < 2.13f) {
    luxCalc = (0.276f * ch0 - 0.130f * ch1);
  } else {
    luxCalc = 0.0f;
  }

  // 按手册示例缩放：/Gain * 102.6 / ITIME_ms
  luxCalc = (luxCalc / actualGain) * (102.6f / itime);
  if (luxCalc < 0.0f) luxCalc = 0.0f;

  _lastCh0 = ch0;
  _lastCh1 = ch1;
  return luxCalc;
}

bool BH1730FVC::setGain(uint8_t gain) {
  // 验证增益值是否有效（0x00-0x03）
  if (gain > 0x03) {
    Serial.println("setGain: invalid gain value (must be 0x00-0x03)");
    return false;
  }
  
  if (_wire == nullptr) {
    Serial.println("setGain: wire is nullptr");
    return false;
  }
  
  // 写入 GAIN 寄存器
  if (!writeReg(REG_GAIN_BASE, gain)) {
    Serial.println("setGain: writeReg GAIN failed");
    return false;
  }
  
  // 更新内部增益值
  _gain = gain;
  delay(2); // 寄存器写入后稳定时间
  
  return true;
}

bool BH1730FVC::powerOn() {
  // CONTROL 寄存器：bit1=1 启动连续测量模式（参考 README.md）
  if (!writeReg(REG_CONTROL_BASE, 0x02)){
      Serial.println("writeReg CONTROL failed");
      return false;
  }
  return true;
}

bool BH1730FVC::writeReg(uint8_t regBase, uint8_t val) {
  _wire->beginTransmission(_addr);
  _wire->write(CMD_WRITE_REG(regBase)); // 添加命令位 0x80
  _wire->write(val);
  byte error = _wire->endTransmission(true);
  if (error != 0) {
      if (error == 1) Serial.println("writeReg: data too long");
      else if (error == 2) Serial.println("writeReg: NACK on address");
      else if (error == 3) Serial.println("writeReg: NACK on data");
      else if (error == 4) Serial.println("writeReg: other error");
      return false;
  }
  return true;
}

uint8_t BH1730FVC::readByte(uint8_t regBase) {
  _wire->beginTransmission(_addr);
  _wire->write(CMD_READ_REG(regBase)); // 添加命令位 0x80
  byte error = _wire->endTransmission(false);
  if (error != 0) return 0xFF;
  
  if (_wire->requestFrom((int)_addr, (int)1) != 1) return 0xFF;
  return _wire->available() ? _wire->read() : 0xFF;
}

uint16_t BH1730FVC::read16BitData(uint8_t startRegBase) {
  _wire->beginTransmission(_addr);
  _wire->write(CMD_READ_REG(startRegBase)); // 指定起始寄存器（低字节），地址自动递增
  byte error = _wire->endTransmission(false);
  if (error != 0) {
      Serial.println("read16BitData: endTransmission failed");
      return 0;
  }
  
  // 读取低字节+高字节（地址自动递增到高字节寄存器）
  if (_wire->requestFrom((int)_addr, (int)2) != 2) {
      Serial.println("read16BitData: requestFrom failed");
      return 0;
  }
  
  if (_wire->available() >= 2) {
      uint8_t lowByte = _wire->read();
      uint8_t highByte = _wire->read();
      return (uint16_t)((highByte << 8) | lowByte);
  }
  return 0;
}

bool BH1730FVC::readBlock(uint8_t startRegBase, uint8_t *buf) {
  // 读取 DATA0 和 DATA1（各16位，共4字节）
  // 从 DATA0_LOW (0x14) 开始，地址自动递增：0x14->0x15->0x16->0x17
  _wire->beginTransmission(_addr);
  _wire->write(CMD_READ_REG(startRegBase)); // 0x80 | 0x14
  byte error = _wire->endTransmission(false);
  if (error != 0) {
      Serial.println("readBlock: endTransmission failed");
      return false;
  }
  
  // 读取4字节（DATA0低、DATA0高、DATA1低、DATA1高）
  if (_wire->requestFrom((int)_addr, (int)4) != 4) {
      Serial.println("readBlock: requestFrom failed");
      return false;
  }
  
  for (uint8_t i = 0; i < 4; i++) {
      if (_wire->available()) {
          buf[i] = _wire->read();
      } else {
          Serial.println("readBlock: not enough data");
          return false;
      }
  }
  
  return true;
}

uint16_t BH1730FVC::integrationMsFromTiming(uint8_t timing) {
  // 根据 README.md，timing=0x02 对应 48ms
  // 使用线性插值估算
  switch(timing) {
    case 0x00: return 160;
    case 0x01: return 100;
    case 0x02: return 48;  // README.md 明确说明
    default: {
      // 其他值使用公式估算
      float base = 160.0f - 2.72f * timing;
      if (base < 5.0f) base = 5.0f;
      return (uint16_t)base;
    }
  }
}

