/*
  BH1730FVC Basic Example
  
  This example demonstrates how to use the BH1730FVC library to read
  ambient light sensor data and calculate lux values.
  
  Hardware connections:
  - VCC to 3.3V or 5V
  - GND to GND
  - SDA to I2C SDA (e.g., GPIO 21 on ESP32, A4 on Arduino Uno)
  - SCL to I2C SCL (e.g., GPIO 22 on ESP32, A5 on Arduino Uno)
  
  Note: Make sure to initialize I2C before using the sensor:
  - ESP32: Wire.begin(SDA_PIN, SCL_PIN);
  - Arduino Uno: Wire.begin();
*/

#include <Wire.h>
#include <BH1730FVC.h>

BH1730FVC sensor;

// For ESP32, define your I2C pins here
// #define SDA_PIN 21
// #define SCL_PIN 22

void setup() {
  Serial.begin(115200);
  delay(1000);
  
  Serial.println("BH1730FVC Ambient Light Sensor Test");
  Serial.println("=================================");
  
  // Initialize I2C
  // For ESP32:
  // Wire.begin(SDA_PIN, SCL_PIN);
  // For Arduino Uno/Nano:
  Wire.begin();
  
  // Initialize sensor with default settings:
  // - Address: 0x29 (default)
  // - Timing: 0x02 (48ms integration time)
  // - Gain: 0x00 (1x gain)
  if (!sensor.begin(&Wire, 0x29, 0x02, 0x00)) {
    Serial.println("Failed to initialize BH1730FVC sensor!");
    Serial.println("Please check your wiring and I2C connections.");
    while(1) delay(1000);
  }
  
  Serial.println("BH1730FVC initialized successfully!");
  Serial.println();
}

void loop() {
  // Read raw channel values
  uint16_t ch0, ch1;
  if (sensor.readRaw(ch0, ch1)) {
    Serial.print("Channel 0 (Visible + IR): ");
    Serial.println(ch0);
    Serial.print("Channel 1 (IR only): ");
    Serial.println(ch1);
    
    // Calculate and display lux value
    float lux = sensor.readLux();
    if (!isnan(lux)) {
      Serial.print("Lux: ");
      Serial.println(lux, 2);
    } else {
      Serial.println("Failed to calculate lux");
    }
    
    Serial.println("---");
  } else {
    Serial.println("Failed to read sensor data!");
  }
  
  delay(1000); // Wait 1 second before next reading
}

