# BH1730FVC Arduino 库

适用于 BH1730FVC 环境光传感器（ALS - Ambient Light Sensor）的 Arduino 库。

[English Documentation](README.md) | [中文文档](README_zh.md)

## 简介

BH1730FVC 是一款数字环境光传感器，可提供精确的光强度测量。本库提供了易于使用的接口，用于读取传感器数据并根据传感器的双通道测量值计算照度（lux）值。

## 特性

- 简单的 I2C 接口
- 使用官方公式自动计算照度值
- 可配置的积分时间和增益设置
- 支持读取原始通道数据
- 兼容 Arduino、ESP32 和其他平台

## 安装

### 使用 Arduino 库管理器

1. 打开 Arduino IDE
2. 转到 **项目** → **加载库** → **管理库**
3. 搜索 "BH1730FVC"
4. 点击 **安装**

### 手动安装

1. 下载或克隆此仓库
2. 将 `BH1730FVC` 文件夹复制到您的 Arduino 库目录：
   - Windows: `Documents\Arduino\libraries\`
   - Mac: `~/Documents/Arduino/libraries/`
   - Linux: `~/Arduino/libraries/`
3. 重启 Arduino IDE

## 硬件连接

| BH1730FVC 引脚 | Arduino 连接 |
|------------|-------------|
| VCC        | 3.3V 或 5V  |
| GND        | GND         |
| SDA        | I2C SDA (Uno 为 A4，ESP32 为 GPIO 21) |
| SCL        | I2C SCL (Uno 为 A5，ESP32 为 GPIO 22) |

**注意：** BH1730FVC 通常使用 I2C 地址 **0x29**。

## 使用方法

### 基础示例

```cpp
#include <Wire.h>
#include <BH1730FVC.h>

BH1730FVC sensor;

void setup() {
  Serial.begin(115200);
  
  // 初始化 I2C
  Wire.begin();  // ESP32: Wire.begin(SDA_PIN, SCL_PIN);
  
  // 初始化传感器
  if (!sensor.begin(&Wire, 0x29, 0x02, 0x00)) {
    Serial.println("传感器初始化失败！");
    return;
  }
}

void loop() {
  // 读取照度值
  float lux = sensor.readLux();
  if (!isnan(lux)) {
    Serial.print("照度: ");
    Serial.println(lux);
  }
  
  delay(1000);
}
```

### 读取原始通道数据

```cpp
uint16_t ch0, ch1;
if (sensor.readRaw(ch0, ch1)) {
  Serial.print("通道 0: ");
  Serial.println(ch0);
  Serial.print("通道 1: ");
  Serial.println(ch1);
}
```

### 自定义设置

```cpp
// 使用自定义设置初始化：
// - 地址: 0x29
// - 积分时间: 0x01 (100ms)
// - 增益: 0x02 (4x 增益，提高灵敏度)
sensor.begin(&Wire, 0x29, 0x01, 0x02);
```

### 运行时更改增益

```cpp
// 初始化后动态更改增益
// 在低光条件下设置为 8x 增益以提高灵敏度
if (sensor.setGain(0x03)) {
  Serial.println("增益已成功设置为 8x");
} else {
  Serial.println("设置增益失败");
}
```

## API 参考

### 构造函数

```cpp
BH1730FVC()
```

创建一个新的 BH1730FVC 传感器对象。

### 方法

#### `begin(TwoWire *wire, uint8_t address, uint8_t timing, uint8_t gain)`

初始化传感器。

- **参数：**
  - `wire`: TwoWire 实例指针（通常为 `&Wire`）
  - `address`: I2C 地址（默认: 0x29）
  - `timing`: 积分时间寄存器值（默认: 0x02 = 48ms）
    - `0x00` = 160ms
    - `0x01` = 100ms
    - `0x02` = 48ms
  - `gain`: 增益设置（默认: 0x00 = 1x）
    - `0x00` = 1x
    - `0x01` = 2x
    - `0x02` = 4x
    - `0x03` = 8x
- **返回值：** 初始化成功返回 `true`，否则返回 `false`

#### `readRaw(uint16_t &ch0, uint16_t &ch1)`

从传感器读取原始通道值。

- **参数：**
  - `ch0`: 用于存储通道 0 值的引用（可见光 + 红外）
  - `ch1`: 用于存储通道 1 值的引用（仅红外）
- **返回值：** 读取成功返回 `true`，否则返回 `false`

#### `readLux(float gain, float integrationMs)`

计算并返回照度值。

- **参数：**
  - `gain`: 传感器增益倍数（可选，传 -1 使用初始化时的值）
  - `integrationMs`: 积分时间（毫秒）（可选，传 -1 使用 timing 寄存器值）
- **返回值：** 照度值（float），计算失败返回 `NAN`

#### `setGain(uint8_t gain)`

在运行时动态设置增益值。

- **参数：**
  - `gain`: 增益寄存器值
    - `0x00` = 1x
    - `0x01` = 2x
    - `0x02` = 4x
    - `0x03` = 8x
- **返回值：** 成功返回 `true`，否则返回 `false`

#### `lastCh0()` / `lastCh1()`

返回最后读取的通道值。

- **返回值：** 最后读取的通道 0/1 值（uint16_t）

## 积分时间参考

| 时序值 | 积分时间 |
|--------|---------|
| 0x00   | 160ms   |
| 0x01   | ~100ms  |
| 0x02   | 48ms    |

## 增益设置

较高的增益值会提高灵敏度，但会降低可测量的最大光强度。

| 增益值 | 倍数 |
|--------|------|
| 0x00   | 1x   |
| 0x01   | 2x   |
| 0x02   | 4x   |
| 0x03   | 8x   |

## 示例

库在 `examples` 文件夹中包含示例程序：

- **Basic**: 演示基本传感器使用的简单示例

## 系统要求

- Arduino IDE 1.8.x 或更高版本
- Wire 库（Arduino 自带）
- 兼容 I2C 的硬件（Arduino Uno、ESP32 等）

## 许可证

本库基于 MIT 许可证发布。详细信息请参阅 LICENSE 文件。

## 贡献

欢迎贡献！如有问题或功能请求，请随时提交 pull request 或创建 issue。

## 作者

MR01Right

## 致谢

- 基于 BH1730FVC 数据手册和应用笔记


