# BH1730FVC Arduino Library

Arduino library for the BH1730FVC ambient light sensor (ALS - Ambient Light Sensor).

[中文文档](README_zh.md) | [English Documentation](README.md)

## Description

The BH1730FVC is a digital ambient light sensor that provides accurate light intensity measurements. This library provides an easy-to-use interface for reading sensor data and calculating lux values based on the sensor's dual-channel measurements.

## Features

- Simple I2C interface
- Automatic lux calculation using official formulas
- Configurable integration time and gain settings
- Support for reading raw channel data
- Compatible with Arduino, ESP32, and other platforms

## Installation

### Using Arduino Library Manager

1. Open Arduino IDE
2. Go to **Sketch** → **Include Library** → **Manage Libraries**
3. Search for "BH1730FVC"
4. Click **Install**

### Manual Installation

1. Download or clone this repository
2. Copy the `BH1730FVC` folder to your Arduino libraries directory:
   - Windows: `Documents\Arduino\libraries\`
   - Mac: `~/Documents/Arduino/libraries/`
   - Linux: `~/Arduino/libraries/`
3. Restart Arduino IDE

## Hardware Connections

| BH1730FVC Pin | Arduino Connection |
|------------|-------------------|
| VCC        | 3.3V or 5V       |
| GND        | GND              |
| SDA        | I2C SDA (A4 on Uno, GPIO 21 on ESP32) |
| SCL        | I2C SCL (A5 on Uno, GPIO 22 on ESP32) |

**Note:** The BH1730FVC typically uses I2C address **0x29**.

## Usage

### Basic Example

```cpp
#include <Wire.h>
#include <BH1730FVC.h>

BH1730FVC sensor;

void setup() {
  Serial.begin(115200);
  
  // Initialize I2C
  Wire.begin();  // For ESP32: Wire.begin(SDA_PIN, SCL_PIN);
  
  // Initialize sensor
  if (!sensor.begin(&Wire, 0x29, 0x02, 0x00)) {
    Serial.println("Sensor initialization failed!");
    return;
  }
}

void loop() {
  // Read lux value
  float lux = sensor.readLux();
  if (!isnan(lux)) {
    Serial.print("Lux: ");
    Serial.println(lux);
  }
  
  delay(1000);
}
```

### Reading Raw Channel Data

```cpp
uint16_t ch0, ch1;
if (sensor.readRaw(ch0, ch1)) {
  Serial.print("Channel 0: ");
  Serial.println(ch0);
  Serial.print("Channel 1: ");
  Serial.println(ch1);
}
```

### Custom Settings

```cpp
// Initialize with custom settings:
// - Address: 0x29
// - Timing: 0x01 (100ms integration time)
// - Gain: 0x02 (4x gain for higher sensitivity)
sensor.begin(&Wire, 0x29, 0x01, 0x02);
```

### Changing Gain at Runtime

```cpp
// Change gain dynamically after initialization
// Set to 8x gain for higher sensitivity in low light conditions
if (sensor.setGain(0x03)) {
  Serial.println("Gain set to 8x successfully");
} else {
  Serial.println("Failed to set gain");
}
```

## API Reference

### Constructor

```cpp
BH1730FVC()
```

Creates a new BH1730FVC sensor object.

### Methods

#### `begin(TwoWire *wire, uint8_t address, uint8_t timing, uint8_t gain)`

Initializes the sensor.

- **Parameters:**
  - `wire`: Pointer to TwoWire instance (usually `&Wire`)
  - `address`: I2C address (default: 0x29)
  - `timing`: Integration time register value (default: 0x02 = 48ms)
    - `0x00` = 160ms
    - `0x01` = 100ms
    - `0x02` = 48ms
  - `gain`: Gain setting (default: 0x00 = 1x)
    - `0x00` = 1x
    - `0x01` = 2x
    - `0x02` = 4x
    - `0x03` = 8x
- **Returns:** `true` if initialization successful, `false` otherwise

#### `readRaw(uint16_t &ch0, uint16_t &ch1)`

Reads raw channel values from the sensor.

- **Parameters:**
  - `ch0`: Reference to store Channel 0 value (Visible + IR)
  - `ch1`: Reference to store Channel 1 value (IR only)
- **Returns:** `true` if read successful, `false` otherwise

#### `readLux(float gain, float integrationMs)`

Calculates and returns lux value.

- **Parameters:**
  - `gain`: Sensor gain multiplier (optional, -1 to use initialization value)
  - `integrationMs`: Integration time in milliseconds (optional, -1 to use timing register value)
- **Returns:** Lux value (float), or `NAN` if calculation failed

#### `setGain(uint8_t gain)`

Sets the gain value dynamically at runtime.

- **Parameters:**
  - `gain`: Gain register value
    - `0x00` = 1x
    - `0x01` = 2x
    - `0x02` = 4x
    - `0x03` = 8x
- **Returns:** `true` if successful, `false` otherwise

#### `lastCh0()` / `lastCh1()`

Returns the last read channel values.

- **Returns:** Last Channel 0/1 value (uint16_t)

## Integration Time Reference

| Timing Value | Integration Time |
|--------------|------------------|
| 0x00         | 160ms            |
| 0x01         | ~100ms           |
| 0x02         | 48ms             |

## Gain Settings

Higher gain values increase sensitivity but reduce the maximum measurable light level.

| Gain Value | Multiplier |
|------------|------------|
| 0x00       | 1x         |
| 0x01       | 2x         |
| 0x02       | 4x         |
| 0x03       | 8x         |

## Examples

The library includes example sketches in the `examples` folder:

- **Basic**: Simple example demonstrating basic sensor usage

## Requirements

- Arduino IDE 1.8.x or later
- Wire library (included with Arduino)
- I2C-compatible hardware (Arduino Uno, ESP32, etc.)

## License

This library is released under the MIT License. See LICENSE file for details.

## Contributing

Contributions are welcome! Please feel free to submit pull requests or open issues for bugs and feature requests.

## Author

MR01Right

## Acknowledgments

- Based on the BH1730FVC datasheet and application notes


