/**********************************************************************
  Menu.cpp

  Robots-For-All (R4A)
  Menu support
**********************************************************************/

#include "R4A_Robot.h"

//****************************************
// Locals
//****************************************

static int32_t r4aMenuUsers;

//*********************************************************************
// Constructor
// Inputs:
//   menu: Address of an R4A_MENU data structure
//   menuTable: Address of table containing the menu descriptions, the
//              main menu must be the first entry in the table.
//   menuEntries: Number of entries in the menu table
//   blankLineBeforePreMenu: Display a blank line before the preMenu
//   blankLineBeforeMenuHeader: Display a blank line before the menu header
//   blankLineAfterMenuHeader: Display a blank line after the menu header
//   alignCommands: Align the commands
//   blankLineAfterMenu: Display a blank line after the menu
void r4aMenuBegin(R4A_MENU * menu,
                  const R4A_MENU_TABLE * menuTable,
                  int menuEntries,
                  bool blankLineBeforePreMenu,
                  bool blankLineBeforeMenuHeader,
                  bool blankLineAfterMenuHeader,
                  bool alignCommands,
                  bool blankLineAfterMenu)
{
    // Initialize the R4A_MENU data structure
    log_v("r4aMenuBegin: menu %p", menu);
    memset(menu, 0, sizeof(*menu));
    menu->_menuTable = menuTable;
    menu->_menuTableEntries = menuEntries;
    menu->_blankLineBeforePreMenu = blankLineBeforePreMenu;
    menu->_blankLineBeforeMenuHeader = blankLineBeforeMenuHeader;
    menu->_blankLineAfterMenuHeader = blankLineAfterMenuHeader;
    menu->_alignCommands = alignCommands;
    menu->_blankLineAfterMenu = blankLineAfterMenu;
}

//*********************************************************************
// Display the boolean as enabled or disabled
void r4aMenuBoolHelp(const R4A_MENU_ENTRY * menuEntry, const char * align, Print * display)
{
    bool * value = (bool *)menuEntry->menuParameter;
    display->printf("%s: %s%s: %s\r\n", menuEntry->command, align,
                    menuEntry->helpText, *value ? "Enabled" : "Disabled");
}

//*********************************************************************
// Toggle boolean value
void r4aMenuBoolToggle(const R4A_MENU_ENTRY * menuEntry, const char * command, Print * display)
{
    bool * value = (bool *)menuEntry->menuParameter;
    *value ^= 1;
}

//*********************************************************************
// Display the menu object contents
void r4aMenuDisplay(R4A_MENU * menu, Print * display)
{
    display->printf("Menu @ %p\r\n", menu);
    display->printf("    _menu: %p\r\n", menu->_menu);
    display->printf("    _menuTable: %p\r\n", menu->_menuTable);
    display->printf("    _menuTableEntries: %d\r\n", menu->_menuTableEntries);
    display->printf("    _blankLineBeforePreMenu: %d\r\n", menu->_blankLineBeforePreMenu);
    display->printf("    _blankLineBeforeMenuHeader: %d\r\n", menu->_blankLineBeforeMenuHeader);
    display->printf("    _blankLineAfterMenuHeader: %d\r\n", menu->_blankLineAfterMenuHeader);
    display->printf("    _alignCommands: %d\r\n", menu->_alignCommands);
    display->printf("    _blankLineAfterMenu: %d\r\n", menu->_blankLineAfterMenu);
}

//*********************************************************************
// Get the string of parameters
String r4aMenuGetParameters(const struct _R4A_MENU_ENTRY * menuEntry,
                            const char * command)
{
    // Get the parameter name
    String line = String(&command[strlen(menuEntry->command)]);

    // Strip white space from the beginning and the end of the name
    line.trim();
    return line;
}

//*********************************************************************
// Display the menu item with a suffix and help text.  The suffix is
// specified as the menu parameter.
// Inputs:
//   menuEntry: Address of the object describing the menu entry
//   align: Zero terminated string of spaces for alignment
//   display: Device used for output
void r4aMenuHelpSuffix(const struct _R4A_MENU_ENTRY * menuEntry,
                       const char * align,
                       Print * display)
{
    display->printf("%s %s: %s%s\r\n",
                    menuEntry->command,
                    (const char *)menuEntry->menuParameter,
                    align,
                    menuEntry->helpText);
}

//*********************************************************************
// Determine if the menu system is active
// Returns true when the menu system is active and false when inactive
bool r4aMenuIsActive()
{
    return r4aMenuUsers ? true : false;
}

//*********************************************************************
// Process the menu command or display the menu
// Returns true when exiting the menu system
bool r4aMenuProcess(R4A_MENU * menu,
                    const char * command,
                    Print * display)
{
    String align("");
    int alignSpaces;
    const char * cmd;
    bool found;
    int length;
    int maxLength;
    const R4A_MENU_ENTRY * menuEnd;
    const R4A_MENU_ENTRY * menuEntry;
    const char * spaces = "                                                  ";
    int spaceCount;

    log_v("r4aMenuProcess: menu %p, command %p, display %p", menu, command, display);
    if (menu->_debug)
    {
        Serial.printf("command: %p %s%s%s\r\n",
                      command,
                      command ? "(" : "",
                      command ? command : "",
                      command ? ")" : "");
        Serial.printf("display: %p\r\n", display);
        Serial.printf("_menu: %p\r\n", menu->_menu);
    }

    // Always start with the main menu
    if (!menu->_menu)
    {
        // https://gcc.gnu.org/wiki/Atomic/GCCMM/LIbrary
        r4aAtomicAdd32(&r4aMenuUsers, 1);
        menu->_menu = &menu->_menuTable[0];
        if (menu->_debug)
            Serial.printf("_menu: %p\r\n", menu->_menu);
    }

    // Process the command
    if (command)
    {
        // Walk the menu table
        menuEntry = menu->_menu->firstEntry;
        menuEnd = &menuEntry[menu->_menu->menuEntryCount];
        while (menuEntry < menuEnd)
        {
            if (menu->_debug)
                Serial.printf("menuEntry: %p\r\n", menuEntry);

            // Determine if the command has parameters
            cmd = menuEntry->command;
            if (menu->_debug)
                Serial.printf("menuEntry->command: %p %s%s%s\r\n",
                              cmd,
                              cmd ? "(" : "",
                              cmd ? cmd : "",
                              cmd ? ")" : "");
            if (menuEntry->align)
            {
                length = strlen(cmd);
                found = (r4aStrincmp(command, cmd, length) == 0);
            }
            else
                found = (r4aStricmp(command, cmd) == 0);
            if (menu->_debug)
                Serial.printf("found: %d\r\n", found);

            // Determine if the command was found
            if (found)
            {
                if (menu->_debug)
                    Serial.printf("menuEntry->menuRoutine: %p\r\n", menuEntry->menuRoutine);

                // Process the command
                if (menuEntry->menuRoutine)
                {
                    if (menu->_debug)
                    {
                        Serial.printf("menuEntry: %p\r\n", menuEntry);
                        Serial.printf("command: %p %s%s%s\r\n",
                                      command,
                                      command ? "(" : "",
                                      command ? command : "",
                                      command ? ")" : "");
                        Serial.printf("display: %p\r\n", display);
                    }
                    menuEntry->menuRoutine(menuEntry, command, display);
                }
                else
                {
                    // Get the next menu index
                    uint32_t index = (int)menuEntry->menuParameter;
                    if (menu->_debug)
                    {
                        Serial.printf("index: %ld\r\n", index);
                        Serial.printf("_menuTableEntries: %d\r\n", menu->_menuTableEntries);
                    }

                    // Validate the next menu index
                    if (index <= menu->_menuTableEntries)
                    {
                        // Select the next menu
                        if (index)
                            menu->_menu = &menu->_menuTable[index - 1];

                        // Exit the menu system
                        else
                        {
                            // https://gcc.gnu.org/wiki/Atomic/GCCMM/LIbrary
                            r4aAtomicSub32(&r4aMenuUsers, 1);
                            menu->_menu = nullptr;
                        }
                        if (menu->_debug)
                            Serial.printf("_menu: %p\r\n", menu->_menu);
                    }
                    else
                        r4aReportFatalError("Invalid menu index!");
                }
                break;
            }
            menuEntry++;
        }

        // Display the error message
        if ((menuEntry >= menuEnd) && strlen(command))
            display->println("Error: Invalid command entered");
    }
    else
    {
        // Start at the beginning of the line
        display->print('\r');

        if (menu->_debug)
            Serial.printf("_menu->preMenu: %p\r\n", menu->_menu->preMenu);
        if (menu->_menu->preMenu)
        {
            // Separate the preMenu from previous output
            if (menu->_debug)
                Serial.printf("_blankLineBeforePreMenu: %d\r\n", menu->_blankLineBeforePreMenu);
            if (menu->_blankLineBeforePreMenu)
                display->println();

            // Display the data before the menu
            if (menu->_menu->preMenu(display) == false)
            {
                // Return to the main menu
                menu->_menu = &menu->_menuTable[0];
                if (menu->_debug)
                    Serial.printf("_menu: %p\r\n", menu->_menu);
                return false;
            }
        }

        // Separate the preMenu display from the menu header
        if (menu->_debug)
            Serial.printf("_blankLineBeforeMenuHeader: %d\r\n", menu->_blankLineBeforeMenuHeader);
        if (menu->_blankLineBeforeMenuHeader)
            display->println();

        // Display the menu header
        if (menu->_debug)
            Serial.printf("_menu->menuName: %p %s%s%s\r\n",
                          menu->_menu->menuName,
                          menu->_menu->menuName ? "(" : "",
                          menu->_menu->menuName ? menu->_menu->menuName : "",
                          menu->_menu->menuName ? ")" : "");
        display->printf("%s\r\n", menu->_menu->menuName);
        for (int length = strlen(menu->_menu->menuName); length > 0; length--)
            display->print("-");
        display->println();

        // Separate the menu header from the commands
        if (menu->_debug)
            Serial.printf("_blankLineAfterMenuHeader: %d\r\n", menu->_blankLineAfterMenuHeader);
        if (menu->_blankLineAfterMenuHeader)
            display->println();

        // Determine the maximum command length
        if (menu->_debug)
        {
            Serial.printf("_alignCommands: %d\r\n", menu->_alignCommands);
            Serial.printf("_menu->menuEntryCount: %ld\r\n", menu->_menu->menuEntryCount);
            Serial.printf("_menu->firstEntry: %p\r\n", menu->_menu->firstEntry);
        }
        maxLength = 0;
        if (menu->_alignCommands)
        {
            menuEntry = menu->_menu->firstEntry;
            menuEnd = &menuEntry[menu->_menu->menuEntryCount];
            while (menuEntry < menuEnd)
            {
                length = strlen(menuEntry->command) + menuEntry->align + 1;
                if (maxLength < length)
                    maxLength = length;
                menuEntry++;
            }
        }

        // Display the menu items
        menuEntry = menu->_menu->firstEntry;
        menuEnd = &menuEntry[menu->_menu->menuEntryCount];
        while (menuEntry < menuEnd)
        {
            if (menu->_debug)
            {
                Serial.printf("menuEntry: %p\r\n", menuEntry);
                Serial.printf("menuEntry->command: %p %s%s%s\r\n",
                              menuEntry->command,
                              menuEntry->command ? "(" : "",
                              menuEntry->command ? menuEntry->command : "",
                              menuEntry->command ? ")" : "");
                Serial.printf("menuEntry->align: %d\r\n", menuEntry->align);
                Serial.printf("menuEntry->helpRoutine: %p\r\n", menuEntry->helpRoutine);
            }

            // Align the menu items
            if (menu->_alignCommands)
            {
                align = String("");
                length = strlen(menuEntry->command) + menuEntry->align
                       + (menuEntry->align ? 1 : 0);
                spaceCount = strlen(spaces);
                alignSpaces = maxLength - length;
                while (alignSpaces > 0)
                {
                    length = alignSpaces;
                    if (length > spaceCount)
                        length = spaceCount;
                    align += &spaces[spaceCount - length];
                    alignSpaces -= length;
                }
            }

            // Display the menu item
            if (menuEntry->helpRoutine)
                menuEntry->helpRoutine(menuEntry, align.c_str(), display);
            else
                display->printf("%s: %s%s\r\n", menuEntry->command,
                                align.c_str(), menuEntry->helpText);
            menuEntry++;
        }

        // Separate the menu from any following text
        if (menu->_blankLineAfterMenu)
            display->println();
    }

    // Determine if exiting the menu system
    if (menu->_debug)
        Serial.printf("_menu: %p\r\n", menu->_menu);
    return (menu->_menu == nullptr);
}
