/**********************************************************************
  LED.cpp

  Robots-For-All (R4A)
  Multicolor LED support
**********************************************************************/

#include "R4A_Robot.h"

//****************************************
// Constants
//****************************************

#define DISPLAY_COLOR       0

#define R4A_LED_RESET       (r4aLEDIntensityTableEntryBytes * 6) // Zero bytes to cause reset
#define R4A_LED_ONES        0       // One bytes to prevent reset

//****************************************
// Globals
//****************************************

uint32_t * r4aLEDColor;             // Stored in shift order (31 - 8 or 31 - 0)
volatile bool r4aLEDColorWritten;
uint8_t *  r4aLEDFourColorsBitmap;  // One bit per LED (0: 3 color, 1: 4 color)
uint8_t r4aLEDIntensity = 255;
uint8_t r4aLEDs;                    // Number of LEDs
const struct _R4A_SPI_DEVICE * r4aLEDSpi;
uint8_t *  r4aLEDTxBuffer;
const uint8_t * r4aLEDIntensityTable;   // Color intensity bits to bytes
int r4aLEDIntensityTableEntryBytes;

//*********************************************************************
// The WS2812 LED specification uses the following bit definitions at
// approximately 800 KHz:
//
//            WS2812 Specification                              SPI
//         __________                               ________
//  0:  __|          |_____________________|       |        |________________|
//         0.35 uSec          0.8 uSec              0.5 uSec      0.75 uSec
//         +/- 150 uSec       +/- 150 uSec
//         ____________________                     ________________
//  1:  __|                    |_____________|     |                |________|
//               0.7 uSec        0.6 uSec               0.75 uSec    0.5 uSec
//               +/- 150 uSec    +/- 150 uSec
//
//
// The SK6812RGBW LED specification uses the following bit definitions
// at approximately 800 KHz.
//
//         ________
//  0:  __|        |________________________|
//         0.3 uSec           0.9 uSec
//         +/- 150 uSec       +/- 150 uSec
//
//         ________________
//  1:  __|                |________________|
//          0.6 uSec          0.6 uSec
//         +/- 150 uSec       +/- 150 uSec
//
//
// This data stream is approximated using a SPI data stream clocked at
// 4 MHz.
//
// The color data is sent most significant bit first in the order of green
// then red then blue.
//
//                          +-----------+-----------+-----------+
//      WS2811 LED  <-----  |7   Red   0|7  Green  0|7  Blue   0|
//                          +-----------+-----------+-----------+
//
//                          +-----------+-----------+-----------+
//      WS2812 LED  <-----  |7  Green  0|7   Red   0|7  Blue   0|
//                          +-----------+-----------+-----------+
//
//                              +-----------+-----------+-----------+-----------+
//      SK6812RGBW LED  <-----  |7   Red   0|7  Green  0|7  Blue   0|7  White  0|
//                              +-----------+-----------+-----------+-----------+

//*********************************************************************
// Set the WS2811 LED colors
void r4aLEDSetWs2811Color(uint8_t ledNumber, uint32_t color)
{
    // Color parameter encoding:
    // Red Bits:   23 - 16
    // Green Bits: 15 -  8
    // Blue Bits:   7 -  0

    // Verify the LED number
    if (ledNumber < r4aLEDs)
    {
        // Indicate that this LED uses 3 colors
        r4aLEDFourColorsBitmap[ledNumber >> 3] &= ~(1 << (ledNumber & 7));

        // Set the LED color in shift order
        // Red Bits:   31 - 24
        // Green Bits: 23 - 16
        // Blue Bits:  15 -  8
        r4aLEDColor[ledNumber] = color << 8;
        r4aLEDColorWritten = true;
    }
    else
        Serial.printf("ERROR: ledNumber needs to be in the range of 0 - %d!",
                      r4aLEDs);
}

//*********************************************************************
// Set the WS2811 LED colors
void r4aLEDSetWs2811ColorRgb(uint8_t ledNumber,
                             uint8_t red,
                             uint8_t green,
                             uint8_t blue)
{
    uint32_t color;

    // Construct the color value
    // Red Bits:   23 - 16
    // Green Bits: 15 -  8
    // Blue Bits:   7 -  0
    color = (((uint32_t)red) << R4A_LED_RED_SHIFT)
          | (((uint32_t)green) << R4A_LED_GREEN_SHIFT)
          | (((uint32_t)blue) << R4A_LED_BLUE_SHIFT);

    // Set the color
    r4aLEDSetWs2811Color(ledNumber, color);
}

//*********************************************************************
// Set the WS2812 LED colors
void r4aLEDSetWs2812Color(uint8_t ledNumber, uint32_t color)
{
    uint8_t blue;
    uint8_t green;
    uint8_t red;

    // Color parameter encoding:
    // Red Bits:   23 - 16
    // Green Bits: 15 -  8
    // Blue Bits:   7 -  0

    // Separate the colors
    red = (color >> R4A_LED_RED_SHIFT) & 0xff;
    green = (color >> R4A_LED_GREEN_SHIFT) & 0xff;
    blue = (color >> R4A_LED_BLUE_SHIFT) & 0xff;

    // Set the color
    r4aLEDSetWs2812ColorRgb(ledNumber, red, green, blue);
}

//*********************************************************************
// Set the WS2812 LED colors
void r4aLEDSetWs2812ColorRgb(uint8_t ledNumber,
                             uint8_t red,
                             uint8_t green,
                             uint8_t blue)
{
    uint32_t color;

    // Verify the LED number
    if (ledNumber < r4aLEDs)
    {
        // Construct the color value in shift order
        color = (((uint32_t)green) << 24)
              | (((uint32_t)red) << 16)
              | (((uint32_t)blue) << 8);

        // Indicate that this LED uses 3 colors
        r4aLEDFourColorsBitmap[ledNumber >> 3] &= ~(1 << (ledNumber & 7));

        // Set the LED color
        r4aLEDColor[ledNumber] = color;
        r4aLEDColorWritten = true;
    }
    else
        Serial.printf("ERROR: ledNumber needs to be in the range of 0 - %d!",
                      r4aLEDs);
}

//*********************************************************************
// Set the SK6812RGBW LED colors
void r4aLEDSetSk6812RgbwColor(uint8_t ledNumber, uint32_t color)
{
    // Color parameter encoding:
    // White Bits: 31 - 24
    // Red Bits:   23 - 16
    // Green Bits: 15 -  8
    // Blue Bits:   7 -  0

    // Verify the LED number
    if (ledNumber < r4aLEDs)
    {
        // Indicate that this LED uses 4 colors
        r4aLEDFourColorsBitmap[ledNumber >> 3] |= 1 << (ledNumber & 7);

        // Set the LED color in shift order
        // Red Bits: 31 - 24
        // Green Bits:   23 - 16
        // Blue Bits: 15 -  8
        // White Bits:   7 -  0
        r4aLEDColor[ledNumber] = (color << 8) | (color >> 24);
        r4aLEDColorWritten = true;
    }
    else
        Serial.printf("ERROR: ledNumber needs to be in the range of 0 - %d!",
                      r4aLEDs);
}

//*********************************************************************
// Set the SK6812RGBW LED colors
void r4aLEDSetSk6812RgbwColorWrgb(uint8_t ledNumber,
                                  uint8_t white,
                                  uint8_t red,
                                  uint8_t green,
                                  uint8_t blue)
{
    uint32_t color;

    // Verify the LED number
    if (ledNumber < r4aLEDs)
    {
        // Construct the color value in shift order
        // Red Bits: 31 - 24
        // Green Bits:   23 - 16
        // Blue Bits: 15 -  8
        // White Bits:   7 -  0
        color = (((uint32_t)red) << 24)
              | (((uint32_t)green) << 16)
              | (((uint32_t)blue) << 8)
              | ((uint32_t)white);

        // Indicate that this LED uses 4 colors
        r4aLEDFourColorsBitmap[ledNumber >> 3] |= 1 << (ledNumber & 7);

        // Set the LED color
        r4aLEDColor[ledNumber] = color;
        r4aLEDColorWritten = true;
    }
    else
        Serial.printf("ERROR: ledNumber needs to be in the range of 0 - %d!",
                      r4aLEDs);
}

//*********************************************************************
// Set the LED intensity
void r4aLEDSetIntensity(uint8_t intensity)
{
    // Change the intensity value
    r4aLEDIntensity = intensity;
    r4aLEDColorWritten = true;
}

//*********************************************************************
// Build the translation table, data bits to TX data
bool r4aLedTranslationTableBuild(const uint8_t * zero,
                                 const uint8_t * one,
                                 int entryBits)
{
    uint8_t * data;
    int dataBitNumber;
    int entryBitNumber;
    size_t length;
    uint8_t * table;
    int txBitNumber;
    const uint8_t * txBits;
    int value;
    int valueBitNumber;

    // Allocate the translation table
    length = entryBits << 8;
    table = (uint8_t *)r4aMalloc(length, "RGB LED Translation Table");
    if (table == nullptr)
        return false;

    // Initialize the table
    memset(table, 0, length);

    // Walk the list of values
    data = table;
    for (value = 0; value <= 255; value++)
    {
        // Walk the value bits (7 - 0)
        txBitNumber = 0;
        for (valueBitNumber = 7; valueBitNumber >= 0; valueBitNumber--)
        {
            // Get the transmit bits for the next value bit
            txBits = (value & (1 << valueBitNumber)) ? one : zero;

            // Walk the entry bits ((n-1) - 0)
            for (entryBitNumber = entryBits - 1; entryBitNumber >= 0; entryBitNumber--)
            {
                // Add to the tx data (bits 7 - 0, then bytes 0 - n)
                dataBitNumber = 7 - (txBitNumber & 7);

                // Determine the data bit
                if (txBits[entryBitNumber >> 3] & (1 << (entryBitNumber & 7)))
                    data[txBitNumber >> 3] |= 1 << dataBitNumber;
                else
                    data[txBitNumber >> 3] &= ~(1 << dataBitNumber);

                // Set the next bit number
                txBitNumber++;
            }
        }

        // Set the next value
        data += entryBits;
    }

    // Save the table and entry size
    r4aLedTranslationTableSet(table, entryBits);
    return true;
}

//*********************************************************************
// Set the translation table
void r4aLedTranslationTableSet(const uint8_t * table, int entryBits)
{
    r4aLEDIntensityTable = table;
    r4aLEDIntensityTableEntryBytes = entryBits;
}

//*********************************************************************
// Build the translation table, data bits to TX data
void r4aLedTranslationTableToC(const char * variable, Print * display)
{
    int bit;
    int bitNumber;
    int bitNumberMax;
    int byteNumber;
    const uint8_t * data;
    int previousBit;
    int value;
    int valueBit;

    // Verify that the table exists
    if ((r4aLEDIntensityTable == nullptr) || (r4aLEDIntensityTableEntryBytes == 0))
        return;

    // Add the variable
    display->printf("const uint8_t %s[] =\r\n", variable);
    display->printf("{\r\n");
    display->printf("};\r\n");

    // Walk the intensity values (0 - 255)
    data = r4aLEDIntensityTable;
    for (value = 0; value < 256; value++)
    {
        // Display the upper comment line
        display->printf("    ");
        for (byteNumber = 0; byteNumber < r4aLEDIntensityTableEntryBytes; byteNumber++)
            display->printf("      ");
        display->printf("//      ");

        // Display the one bits on the upper comment line
        previousBit = 0;
        bitNumberMax = r4aLEDIntensityTableEntryBytes << 3;
        for (bitNumber = 0; bitNumber < bitNumberMax; bitNumber++)
        {
            bit = (data[bitNumber >> 3] >> (7 - (bitNumber & 7))) & 1;
            if (bit != previousBit)
            {
                previousBit = bit;
                display->printf(" ");
            }
            display->printf(bit ? "_" : " ");
        }
        display->printf("\r\n");

        // Display the bytes
        display->printf("    ");
        for (byteNumber = 0; byteNumber < r4aLEDIntensityTableEntryBytes; byteNumber++)
        {
            // Display the bits necessary for the intensity value
            display->printf("0x%02x, ", data[byteNumber]);
        }

        // Display the comment
        display->printf("// %3d  ", value);

        // Display the zero bits on the lower comment line
        previousBit = 0;
        bitNumberMax = r4aLEDIntensityTableEntryBytes << 3;
        for (bitNumber = 0; bitNumber < bitNumberMax; bitNumber++)
        {
            bit = (data[bitNumber >> 3] >> (7 - (bitNumber & 7))) & 1;
            if (bit != previousBit)
            {
                previousBit = bit;
                if (bit)
                    display->printf("|");
                else
                    display->printf("%d", (value >> (7 - (bitNumber / r4aLEDIntensityTableEntryBytes))) & 1);
            }

            // Display the bit
            display->printf(bit ? " " : "_");
        }
        display->printf("\r\n");

        // Set the next value
        data += r4aLEDIntensityTableEntryBytes;
    }
    display->printf("\r\n");
}

//*********************************************************************
// Initialize the LEDs
bool r4aLEDSetup(const R4A_SPI_DEVICE * spiDevice,
                 uint8_t numberOfLEDs)
{
    int ledBytes;
    int length;

    do
    {
        // Determine if the SPI controller object was allocated
        if (!spiDevice)
        {
            Serial.println("ERROR: No SPI data structure specified!");
            break;
        }
        r4aLEDSpi = spiDevice;

        // Remember the number of LEDs
        r4aLEDs = numberOfLEDs;

        // Set the translation table
        if (r4aLEDIntensityTable == nullptr)
        {
            r4aLEDIntensityTable = r4aLEDIntensityTable_2_3_4MHz;
            r4aLEDIntensityTableEntryBytes = 5;
        }

        // Allocate the TX DMA buffer
        // This buffer needs to be in static RAM to allow DMA access
        // Assume all LEDs support 4 colors, 8-bits per color and x SPI
        // bits / color bit packed in 8 bits per byte
        ledBytes = numberOfLEDs * 4 * r4aLEDIntensityTableEntryBytes;
        length = R4A_LED_RESET + ledBytes + R4A_LED_ONES;
        r4aLEDTxBuffer = (uint8_t *)r4aMalloc(length, "LED color buffer (r4aLEDTxBuffer)");
        if (!r4aLEDTxBuffer)
        {
            Serial.println("ERROR: Failed to allocate r4aLEDTxBuffer!");
            break;
        }
        memset(r4aLEDTxBuffer, 0, length);

        // Allocate the color array, assume four 8-bit colors per LED
        length = numberOfLEDs << 2;
        r4aLEDColor = (uint32_t *)r4aMalloc(length, "LED color array (r4aLEDColor)");
        if (!r4aLEDColor)
        {
            Serial.println("ERROR: Failed to allocate r4aLEDColor!");
            break;
        }
        memset(r4aLEDColor, 0, length);

        // Allocate the 4 color bitmap
        length = (numberOfLEDs + 7) >> 3;
        r4aLEDFourColorsBitmap = (uint8_t *)r4aMalloc(length, "LED 4 color bitmap (r4aLEDFourColorsBitmap)");
        if (!r4aLEDFourColorsBitmap)
        {
            Serial.println("ERROR: Failed to allocate r4aLEDFourColorsBitmap!");
            break;
        }

        // Assume all LEDs are 4 color
        memset(r4aLEDFourColorsBitmap, 0xff, length);

        // Turn off the LEDs
        r4aLEDColorWritten = true;
        r4aLEDUpdate(true);
        return true;
    } while (0);

    // Free the allocated memory
    if (r4aLEDColor)
    {
        r4aFree((void *)r4aLEDColor, "LED color array (r4aLEDColor)");
        r4aLEDColor = nullptr;
    }
    if (r4aLEDFourColorsBitmap)
    {
        r4aFree((void *)r4aLEDFourColorsBitmap, "LED 4 color bitmap (r4aLEDFourColorsBitmap)");
        r4aLEDFourColorsBitmap = nullptr;
    }
    if (r4aLEDTxBuffer)
    {
        r4aDmaFree((void *)r4aLEDTxBuffer, "LED color buffer (r4aLEDTxBuffer)");
        r4aLEDTxBuffer = nullptr;
    }
    return false;
}

//*********************************************************************
// Turn off the LEDs
void r4aLEDsOff()
{
    // Set the LED colors
    for (uint8_t led = 0; led < r4aLEDs; led++)
        if (r4aLEDFourColorsBitmap[led >> 7] & (1 << (led & 7)))
            r4aLEDSetColorWrgb(led, 0);
        else
            r4aLEDSetColorRgb(led, 0);
}

//*********************************************************************
// Update the colors on the LEDs
void r4aLEDUpdate(bool updateRequest, Print * display)
{
    uint32_t color;
    int colorCount;
    uint8_t * data;
    int entryBytes;
    int index;
    int intensity;
    static int length;

    // Check for a color change
    if (r4aLEDColorWritten)
    {
        r4aLEDColorWritten = false;
        updateRequest = true;

        // Add the reset sequence
        data = r4aLEDTxBuffer;
        if (R4A_LED_RESET == 0)
        {
            memset(data, 0, R4A_LED_RESET);
            data += R4A_LED_RESET;
        }

        // Walk the array of LEDs
        entryBytes = r4aLEDIntensityTableEntryBytes;
        for (int led = 0; led < r4aLEDs; led++)
        {
            // Copy the color array
            color = r4aLEDColor[led];

            // Determine the LED type
            colorCount = 3;
            if (r4aLEDFourColorsBitmap[led >> 3] & (1 << (led & 7)))
                colorCount = 4;

            // Walk the colors
            if (DISPLAY_COLOR)
                Serial.printf("LED[%3d]: 0x%08x (", led, color);
            while (--colorCount >= 0)
            {
                // Adjust the color intensity
                intensity = (color >> 24) & 0xff;
                intensity = (intensity * r4aLEDIntensity) / 255;
                if (DISPLAY_COLOR)
                {
                    Serial.printf("%3d", intensity);
                    if (colorCount)
                        Serial.printf(", ");
                }

                // Map the intensity to the x bytes of data to send the 8 bits
                index = intensity * entryBytes;
                memcpy(data, &r4aLEDIntensityTable[index], entryBytes);
                data += entryBytes;

                // Set the next color
                color <<= 8;
            }
            if (DISPLAY_COLOR)
                Serial.printf(")\r\n");
        }

        // Set the ones
        if (R4A_LED_ONES != 0)
        {
            memset(data, 0xff, R4A_LED_ONES);
            data += R4A_LED_ONES;
        }

        // Determine the amount of data to send to the LEDs
        length = data - r4aLEDTxBuffer;
    }

    // Determine if an update to the WS2812 LEDs is needed
    if (updateRequest && r4aLEDSpi)
    {
        uint32_t currentUsec;
        bool delayInProgress;
        static uint32_t lastUpdateUsec;

        // Delay long enough for the WS2812 devices to reset (allow an update)
        do
        {
            currentUsec = micros();
            delayInProgress = ((currentUsec - lastUpdateUsec) < (50 + 10));
            lastUpdateUsec = currentUsec;
        } while (delayInProgress);

        // Output the color data to the LEDs
        r4aSpiTransfer(r4aLEDSpi, r4aLEDTxBuffer, nullptr, length, display);
    }
}

//****************************************
// LED Menu API
//****************************************

//*********************************************************************
// Display the help text with mm and ssss
[[deprecated("Use r4aMenuHelpSuffix instead.")]]
void r4aLEDMenuHelpiii(const struct _R4A_MENU_ENTRY * menuEntry,
                       const char * align,
                       Print * display)
{
    display->printf("%s iii: %s%s\r\n",
                    menuEntry->command, align, menuEntry->helpText);
}

//*********************************************************************
// Display the help text with mm and ssss
[[deprecated("Use r4aMenuHelpSuffix instead.")]]
void r4aLEDMenuHelpllcccc(const struct _R4A_MENU_ENTRY * menuEntry,
                          const char * align,
                          Print * display)
{
    display->printf("%s ll cccccccc: %s%s\r\n",
                    menuEntry->command, align, menuEntry->helpText);
}

//*********************************************************************
// Get the intensity
bool r4aLEDMenuGetIntensity(const R4A_MENU_ENTRY * menuEntry,
                            const char * command,
                            int * values,
                            uint8_t * intensity)
{
    int i;

    // Get the parameter name
    String line = r4aMenuGetParameters(menuEntry, command);

    // Get the value
    *values = sscanf(line.c_str(), "%d", &i);

    // Determine if the value is within range
    if ((*values == 1) && (i >= 0) && (i <= 255))
    {
        *intensity = i;
        return true;
    }
    return false;
}

//*********************************************************************
// Get the LED number and color value
bool r4aLEDMenuGetLedColor(const R4A_MENU_ENTRY * menuEntry,
                           const char * command,
                           int * values,
                           uint8_t * led,
                           uint32_t * color)
{
    int c;
    int l;

    // Get the parameter name
    String line = r4aMenuGetParameters(menuEntry, command);

    // Get the values
    *values = sscanf(line.c_str(), "%d %x", &l, &c);

    // Determine if the values are within range
    if ((*values == 2)
        && (l >= 0)
        && (l < r4aLEDs))
    {
        *led = l;
        *color = c;
        return true;
    }
    else if (*values == 1)
        *led = l;
    return false;
}

//*********************************************************************
// Set the WS2812 LED color
// Inputs:
//   menuEntry: Address of the object describing the menu entry
//   command: Zero terminated command string
//   display: Device used for output
void r4aLEDMenuColor3(const R4A_MENU_ENTRY * menuEntry, const char * command, Print * display)
{
    uint8_t led;
    uint32_t color;
    int values;

    if (r4aLEDMenuGetLedColor(menuEntry, command, &values, &led, &color))
    {
        r4aLEDSetColorRgb(led, color);
        r4aLEDUpdate(true);
    }
    else if (values == 1)
    {
        if (display)
            display->println("ERROR: Please specify a color in hex as RRGGBB");
    }
    else
    {
        if (display)
            display->printf("ERROR: Please specify a LED number in the range of (0 - %d)", r4aLEDs - 1);
    }
}

//*********************************************************************
// Set the SK6812RGBW LED color
// Inputs:
//   menuEntry: Address of the object describing the menu entry
//   command: Zero terminated command string
//   display: Device used for output
void r4aLEDMenuColor4(const R4A_MENU_ENTRY * menuEntry, const char * command, Print * display)
{
    uint8_t led;
    uint32_t color;
    int values;

    if (r4aLEDMenuGetLedColor(menuEntry, command, &values, &led, &color))
    {
        r4aLEDSetColorWrgb(led, color);
        r4aLEDUpdate(true);
    }
    else if (values == 1)
    {
        if (display)
            display->println("ERROR: Please specify a color in hex as WWRRGGBB");
    }
    else
    {
        if (display)
            display->printf("ERROR: Please specify a LED number in the range of (0 - %d)", r4aLEDs - 1);
    }
}

//*********************************************************************
// Display the LED status
// Inputs:
//   menuEntry: Address of the object describing the menu entry
//   command: Zero terminated command string
//   display: Device used for output
void r4aLEDMenuDisplay(const R4A_MENU_ENTRY * menuEntry, const char * command, Print * display)
{
    uint8_t blue;
    uint32_t color;
    uint8_t green;
    uint32_t intensity;
    uint8_t red;
    uint8_t white;

    if (display)
    {
        //                ll:   xxx    xxx    xxx    xxx   0xxxxxxxxx   0xxxxxxxxx
        display->println("LED  White   Red   Green  Blue          Hex   Programmed");
        display->println("--------------------------------------------------------");
        for (int led = 0; led < r4aLEDs; led++)
        {
            // Breakup the color value
            color = r4aLEDColor[led];
            white = color >> 24;
            red = (color >> 16) & 0xff;
            green = (color >> 8) & 0xff;
            blue = color & 0xff;
            intensity = (((white * r4aLEDIntensity) / 255) << 24)
                      | (((red   * r4aLEDIntensity) / 255) << 16)
                      | (((green * r4aLEDIntensity) / 255) << 8)
                      |  ((blue  * r4aLEDIntensity) / 255);

            if (r4aLEDFourColorsBitmap[led >> 3] & (1 << (led & 7)))
                display->printf("%2d:   %3d    %3d    %3d    %3d   0x%08lx   0x%08lx\r\n",
                                led, white, red, green, blue, color, intensity);

            else
                display->printf("%2d:          %3d    %3d    %3d     0x%06lx     0x%06lx\r\n",
                                led, red, green, blue, color, intensity);
        }
        display->printf("Intensity: %d\r\n", r4aLEDIntensity);
    }
}

//*********************************************************************
// Set the LED intensity
// Inputs:
//   menuEntry: Address of the object describing the menu entry
//   command: Zero terminated command string
//   display: Device used for output
void r4aLEDMenuIntensity(const R4A_MENU_ENTRY * menuEntry, const char * command, Print * display)
{
    uint8_t intensity;
    int values;

    if (r4aLEDMenuGetIntensity(menuEntry, command, &values, &intensity))
    {
        r4aLEDSetIntensity(intensity);
        r4aLEDUpdate(true);
    }
    else
    {
        if (display)
            display->println("ERROR: Please specify an intensity value in the range of (0 - 255)");
    }
}

//*********************************************************************
// Turn off the LEDs
// Inputs:
//   menuEntry: Address of the object describing the menu entry
//   command: Zero terminated command string
//   display: Device used for output
void r4aLEDMenuOff(const R4A_MENU_ENTRY * menuEntry, const char * command, Print * display)
{
    r4aLEDsOff();
    r4aLEDUpdate(true);
}
