/**********************************************************************
  VK16K33.cpp

  Robots-For-All (R4A)
  LED controller support
**********************************************************************/

#include "R4A_I2C.h"

//****************************************
// Constants
//****************************************

#define R4A_VK16K33_CMD_DATA_ADDRESS        0x00
#define R4A_VK16K33_CMD_SYSTEM_SET          0x20
#define R4A_VK16K33_CMD_DISPLAY_SET         0x80
#define R4A_VK16K33_CMD_DISPLAY_BRIGHTNESS  0xe0

#define R4A_VK16K33_CSS_ON              0x01    // Turn on the controller

#define R4A_VK16K33_CDS_ON              0x01    // Turn on the display
#define R4A_VK16K33_CDS_BLINK_OFF       0       // Disable blinking
#define R4A_VK16K33_CDS_BLINK_2HZ       0x02    // 2 Hz blink frequency
#define R4A_VK16K33_CDS_BLINK_1HZ       0x02    // 1 Hz blink frequency
#define R4A_VK16K33_CDS_BLINK_0_5HZ     0x02    // 0.5 Hz blink frequency

#define R4A_VK16K33_CDB_1_16            0   // 1 / 16
#define R4A_VK16K33_CDB_2_16            1   // 2 / 16
#define R4A_VK16K33_CDB_3_16            2   // 3 / 16
#define R4A_VK16K33_CDB_4_16            3   // 4 / 16
#define R4A_VK16K33_CDB_5_16            4   // 5 / 16
#define R4A_VK16K33_CDB_6_16            5   // 6 / 16
#define R4A_VK16K33_CDB_7_16            6   // 7 / 16
#define R4A_VK16K33_CDB_8_16            7   // 8 / 16
#define R4A_VK16K33_CDB_9_16            8   // 9 / 16
#define R4A_VK16K33_CDB_10_16           9   // 10 / 16
#define R4A_VK16K33_CDB_11_16           10  // 11 / 16
#define R4A_VK16K33_CDB_12_16           11  // 12 / 16
#define R4A_VK16K33_CDB_13_16           12  // 13 / 16
#define R4A_VK16K33_CDB_14_16           13  // 14 / 16
#define R4A_VK16K33_CDB_15_16           14  // 15 / 16
#define R4A_VK16K33_CDB_16_16           15  // 16 / 16

// LED Matrix (VK16K33) menu
const R4A_MENU_ENTRY r4aVk16k33MenuTable[] =
{
    // Command  menuRoutine     menuParam       HelpRoutine align   HelpText
    {"c", r4aVk16k33MenuClear,  0,              nullptr,    0,      "Clear the LED matrix"},
    {"d", r4aVk16k33MenuDate,   0,              nullptr,    0,      "Display the data"},
    {"f", r4aVk16k33MenuFill,   0,              nullptr,    0,      "Fill the LED matrix"},
    {"h", r4aVk16k33MenuHalt,   0,              nullptr,    0,      "Display Halt"},
    {"i", r4aVk16k33MenuIdle,   0,              nullptr,    0,      "Display Idle"},
    {"t", r4aVk16k33MenuTime,   0,              nullptr,    0,      "Display the time"},
    {"w", r4aVk16k33MenuWrite, (intptr_t)"ccc", nullptr,    3,      "Write up to 3 characters to the LED matrix"},
    {"x",       nullptr,        R4A_MENU_MAIN,  nullptr,    0,      "Return to the main menu"},
};

//****************************************
// Locals
//****************************************

R4A_VK16K33 * r4aVk16k33;
bool r4aVk16k33WriteColumnFast; // Rows in the correct order

//*********************************************************************
// Set the brightness (0-15)
bool r4aVk16k33Brightness(R4A_VK16K33 * vk16k33,
                          uint8_t brightness,
                          Print * display)
{
    uint8_t cmd;
    bool success = false;

    // Use the Display Brightness command to set the pulse width, see
    // VK16K33 specification v1.2, page 27
    cmd = R4A_VK16K33_CMD_DISPLAY_BRIGHTNESS | (brightness & 0xf);
    success = r4aI2cBusWrite(vk16k33->i2cBus,
                             vk16k33->i2cAddress,
                             &cmd,
                             sizeof(cmd),
                             display);
    if ((!success) && display)
        display->printf("ERROR: Failed to set VK16K33 brightness!\r\n");
    return success;
}

//*********************************************************************
// Clear the RAM buffer
void r4aVk16k33BufferClear(R4A_VK16K33 * vk16k33)
{
    memset(&vk16k33->pixels[R4A_VK16K33_PIXEL_OFFSET], 0, R4A_VK16K33_MAX_COLUMNS);
}

//*********************************************************************
// Fill the RAM buffer
void r4aVk16k33BufferFill(R4A_VK16K33 * vk16k33, uint8_t data)
{
    memset(&vk16k33->pixels[R4A_VK16K33_PIXEL_OFFSET], data, R4A_VK16K33_MAX_COLUMNS);
}

//*********************************************************************
// Display a character on the LED matrix
void r4aVk16k33DisplayChar(R4A_VK16K33 * vk16k33, int xColumn, char data)
{
    int columnCount;
    const uint8_t * font;

    // Get the font data
    font = nullptr;
    columnCount = 5;
    switch (data)
    {
    default:
        break;
    case '*':
        font = r4a5x7Font_asterisk;
        break;

    case '-':
        font = r4a5x7Font_dash;
        break;

    case '.':
        columnCount = 1;
        font = &r4a5x7Font_dp;
        break;

    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
        font = r4a5x7Numbers[data - '0'];
        break;

    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'G':
    case 'H':
    case 'I':
        font = r4a5x7UcAtoI[data - 'A'];
        break;

    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
    case 'g':
    case 'h':
        font = r4a5x7LcAtoH[data - 'a'];
        break;

    case 'l':
        columnCount = 1;
        font = &r4a5x7Font_l;
        break;

    case 't':
        font = r4a5x7Font_t;
        break;
    }

    // Set the pixels in the columns
    if (font)
        for (int column = xColumn; column < (xColumn + columnCount); column++)
            r4aVk16k33WriteColumn(vk16k33, vk16k33->columnMap[column], *font++);
}

//*********************************************************************
// Display "Halt" on the LED matrix
void r4aVk16k33DisplayHalt(R4A_VK16K33 * vk16k33)
{
    r4aVk16k33BufferClear(vk16k33);
    r4aVk16k33DisplayChar(vk16k33, 0, 'H');
    r4aVk16k33DisplayChar(vk16k33, 5, 'a');
    r4aVk16k33DisplayChar(vk16k33, 10, 'l');
    r4aVk16k33DisplayChar(vk16k33, 11, 't');
    r4aVk16k33DisplayPixels(vk16k33);
    r4aVk16k33DisplayOn(vk16k33);
}

//*********************************************************************
// Display "Idle" on the LED matrix
void r4aVk16k33DisplayIdle(R4A_VK16K33 * vk16k33)
{
    r4aVk16k33BufferClear(vk16k33);
    r4aVk16k33DisplayChar(vk16k33, 0, 'I');
    r4aVk16k33DisplayChar(vk16k33, 5, 'd');
    r4aVk16k33DisplayChar(vk16k33, 10, 'l');
    r4aVk16k33DisplayChar(vk16k33, 11, 'e');
    r4aVk16k33DisplayPixels(vk16k33);
    r4aVk16k33DisplayOn(vk16k33);
}

//*********************************************************************
// Turn on the display
bool r4aVk16k33DisplayOn(R4A_VK16K33 * vk16k33, Print * display)
{
    uint8_t cmd;
    bool success = false;

    // Use the Display Set command to turn on the display and disable
    // blinking, see VK16K33 specification v1.2, page 27
    cmd = R4A_VK16K33_CMD_DISPLAY_SET
        | R4A_VK16K33_CDS_BLINK_OFF
        | R4A_VK16K33_CDS_ON;
    success = r4aI2cBusWrite(vk16k33->i2cBus,
                             vk16k33->i2cAddress,
                             &cmd,
                             sizeof(cmd),
                             display);
    if (!success)
        Serial.printf("ERROR: Failed to turn on VK16K33 display!\r\n");
    return success;
}

//*********************************************************************
// Copy the RAM buffer to the display
// Start bit, I2C device address, ACK, register address, ACK, 16 data bytes
// with ACKs and a stop bit, all at 400 KHz
// ~410 uSec = (1+8+1+8+1+((8+1)×16)+1)÷(400×1000)
bool r4aVk16k33DisplayPixels(R4A_VK16K33 * vk16k33, Print * display)
{
    bool success = false;

    do
    {
        // Concatenate the command and pixel data
        // Copy the RAM buffer to the display
        // HT16K33 specification v1.10, page 30
        vk16k33->pixels[0] = R4A_VK16K33_CMD_DATA_ADDRESS | 0;
        success = r4aI2cBusWrite(vk16k33->i2cBus,
                                 vk16k33->i2cAddress,
                                 vk16k33->pixels,
                                 sizeof(vk16k33->pixels),
                                 display);
        if (!success)
        {
            if (display)
                display->printf("ERROR: Failed to write VK16K33 pixel data!\r\n");
            break;
        }
    } while (0);
    return success;
}

//*********************************************************************
// Turn off all pixels in the LED matrix
void r4aVk16k33MenuClear(const R4A_MENU_ENTRY * menuEntry,
                         const char * command,
                         Print * display)
{
    // Clear the display
    r4aVk16k33BufferClear(r4aVk16k33);
    r4aVk16k33DisplayPixels(r4aVk16k33);
}

//*********************************************************************
// Display the date on the LED matrix
void r4aVk16k33MenuDate(const R4A_MENU_ENTRY * menuEntry,
                        const char * command,
                        Print * display)
{
    String localTime;

    // Clear the display
    r4aVk16k33BufferClear(r4aVk16k33);

    // Determine if the time is valid
    r4aNtpUpdate(WiFi.STA.connected());
    if (r4aNtpIsTimeValid())
    {
        // Get the time
        time_t seconds = r4aNtpGetEpochTime();
        localTime = r4aNtpGetDate(seconds);
        const char * date = localTime.c_str();

        // Display the date     0123456789
        //                      yyyy-mm-dd
        date = &date[5];
        if (date[5] == '1')
            r4aVk16k33DisplayChar(r4aVk16k33, 0, 'l');
        date++;
        r4aVk16k33DisplayChar(r4aVk16k33, 1, *date++);
        date++;
        r4aVk16k33DisplayChar(r4aVk16k33, 6, *date++);
        r4aVk16k33DisplayChar(r4aVk16k33, 11, *date++);
    }

    // Display the date
    r4aVk16k33DisplayPixels(r4aVk16k33);
}

//*********************************************************************
// Turn on all pixels in the LED matrix
void r4aVk16k33MenuFill(const R4A_MENU_ENTRY * menuEntry,
                        const char * command,
                        Print * display)
{
    r4aVk16k33BufferFill(r4aVk16k33, 0xff);
    r4aVk16k33DisplayPixels(r4aVk16k33);
}

//*********************************************************************
// Display Halt on the LED matrix
void r4aVk16k33MenuHalt(const R4A_MENU_ENTRY * menuEntry,
                        const char * command,
                        Print * display)
{
    r4aVk16k33DisplayHalt(r4aVk16k33);
}

//*********************************************************************
// Display Idle on the LED matrix
void r4aVk16k33MenuIdle(const R4A_MENU_ENTRY * menuEntry,
                        const char * command,
                        Print * display)
{
    r4aVk16k33DisplayIdle(r4aVk16k33);
}

//*********************************************************************
// Display the current time on the LED matrix
void r4aVk16k33MenuTime(const R4A_MENU_ENTRY * menuEntry,
                        const char * command,
                        Print * display)
{
    String localTime;

    // Clear the display
    r4aVk16k33BufferClear(r4aVk16k33);

    // Determine if the time is valid
    r4aNtpUpdate(WiFi.STA.connected());
    if (r4aNtpIsTimeValid())
    {
        // Get the time         01234567
        //                      hh:mm:ss
        time_t seconds = r4aNtpGetEpochTime();
        localTime = r4aNtpGetTime12(seconds);
        const char * time = localTime.c_str();
        if (*time == '1')
            r4aVk16k33DisplayChar(r4aVk16k33, 0, 'l');
        time++;
        r4aVk16k33DisplayChar(r4aVk16k33, 1, *time++);
        time++;
        r4aVk16k33DisplayChar(r4aVk16k33, 6, *time++);
        r4aVk16k33DisplayChar(r4aVk16k33, 11, *time++);
    }

    // Display the data
    r4aVk16k33DisplayPixels(r4aVk16k33);
}

//*********************************************************************
// Display up to 3 characters on the display
void r4aVk16k33MenuWrite(const R4A_MENU_ENTRY * menuEntry,
                         const char * command,
                         Print * display)
{
    const char * data;
    static String parameter;

    // Locate the parameter
    parameter = r4aMenuGetParameters(menuEntry, command);

    // Clear the display
    r4aVk16k33BufferClear(r4aVk16k33);

    // Get the string
    data = parameter.c_str();
    if (*data)
        r4aVk16k33DisplayChar(r4aVk16k33, 1, *data++);
    if (*data)
        r4aVk16k33DisplayChar(r4aVk16k33, 6, *data++);
    if (*data)
        r4aVk16k33DisplayChar(r4aVk16k33, 11, *data++);

    // Display the data
    r4aVk16k33DisplayPixels(r4aVk16k33);
}

//*********************************************************************
// Turn on the VK16K33 LED controller
bool r4aVk16k33On(R4A_VK16K33 * vk16k33, Print * display)
{
    uint8_t cmd;
    bool success;

    // Use the System Set command to turn on the controller, see
    // VK16K33 specification v1.2, page 27
    cmd = R4A_VK16K33_CMD_SYSTEM_SET | R4A_VK16K33_CSS_ON;
    success = r4aI2cBusWrite(vk16k33->i2cBus,
                             vk16k33->i2cAddress,
                             &cmd,
                             sizeof(cmd),
                             display);
    if (!success)
        Serial.printf("ERROR: Failed to turn on VK16K33!\r\n");
    return success;
}

//*********************************************************************
// Clear a pixel in the RAM buffer
bool r4aVk16k33PixelClear(R4A_VK16K33 * vk16k33, uint8_t column, uint8_t row)
{
    do
    {
        // Verify the column and row
        if (column >= vk16k33->columns)
        {
            Serial.printf("ERROR: Invalid column number, must be <= %d\r\n",
                          vk16k33->columns);
            break;
        }
        if (row >= vk16k33->rows)
        {
            Serial.printf("ERROR: Invalid row number, must be <= %d\r\n",
                          vk16k33->rows);
            break;
        }

        // Clear the pixel
        uint8_t bitMask = 1 << row;
        vk16k33->pixels[R4A_VK16K33_PIXEL_OFFSET + column] &= ~bitMask;
        return true;
    } while (0);
    return false;
}

//*********************************************************************
// Set a pixel in the RAM buffer
bool r4aVk16k33PixelSet(R4A_VK16K33 * vk16k33, uint8_t column, uint8_t row)
{
    do
    {
        // Verify the column and row
        if (column >= vk16k33->columns)
        {
            Serial.printf("ERROR: Invalid column number, must be <= %d\r\n",
                          vk16k33->columns);
            break;
        }
        if (row >= vk16k33->rows)
        {
            Serial.printf("ERROR: Invalid row number, must be <= %d\r\n",
                          vk16k33->rows);
            break;
        }

        // Set the pixel
        uint8_t bitMask = 1 << row;
        vk16k33->pixels[R4A_VK16K33_PIXEL_OFFSET + column] |= bitMask;
        return true;
    } while (0);
    return false;
}

//*********************************************************************
// Initialize the VK16K33
bool r4aVk16k33Setup(R4A_VK16K33 * vk16k33, Print * display)
{
    uint8_t cmd;
    int row;
    bool success = false;

    do
    {
        // Verify the number of columns and rows
        if (vk16k33->columns > R4A_VK16K33_MAX_COLUMNS)
        {
            if (display)
                display->printf("ERROR: Too many columns, columns <= %d\r\n",
                                R4A_VK16K33_MAX_COLUMNS);
            break;
        }
        if (vk16k33->rows > R4A_VK16K33_MAX_ROWS)
        {
            if (display)
                display->printf("ERROR: Too many rows, rows <= %d\r\n",
                                R4A_VK16K33_MAX_ROWS);
            break;
        }

        // Verify the order of the rows
        r4aVk16k33WriteColumnFast = false;
        for (row = 0; row < R4A_VK16K33_MAX_ROWS; row++)
            if (vk16k33->rowMap[row] != row)
                break;
        if (row == R4A_VK16K33_MAX_ROWS)
            r4aVk16k33WriteColumnFast = true;

        // Turn on the controller
        success = r4aVk16k33On(vk16k33, display);
        if (!success)
            break;

        // Clear the display buffer
        r4aVk16k33BufferClear(vk16k33);
        success = r4aVk16k33DisplayPixels(vk16k33, display);
        if (!success)
            break;

        // Turn on the display, start the scanning of the LEDs
        success = r4aVk16k33DisplayOn(vk16k33, display);
        if (!success)
            break;

        // Use the Display Brightness command to set the pulse width, see
        // VK16K33 specification v1.2, page 27
        success = r4aVk16k33Brightness(vk16k33, vk16k33->brightness, display);
        if (!success)
            break;

        // Remember this LED matrix controller
        r4aVk16k33 = vk16k33;
    } while (0);
    return success;
}

//*********************************************************************
// Write a column of eight pixels in the RAM buffer
bool r4aVk16k33WriteColumn(R4A_VK16K33 * vk16k33, uint8_t column, uint8_t data)
{
    // Verify the column and row
    if (column >= vk16k33->columns)
    {
        Serial.printf("ERROR: Invalid column number, must be <= %d\r\n",
                      vk16k33->columns);
        return false;
    }

    // Set the pixels
    if (r4aVk16k33WriteColumnFast)
        vk16k33->pixels[R4A_VK16K33_PIXEL_OFFSET + column] = data;
    else
    {
        for (int row = 0; row < R4A_VK16K33_MAX_ROWS; row++)
        {
            int pixel = data & (1 << row);
            if (pixel)
                r4aVk16k33PixelSet(vk16k33, column, vk16k33->rowMap[row]);
            else
                r4aVk16k33PixelClear(vk16k33, column, vk16k33->rowMap[row]);
        }
    }
    return true;
}
